//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_HashId_h)
#define Courier_Util_HashId_h

#include <Courier/Util/Limits.h>
#include <FeatStd/Util/Hash.h>
#include <FeatStd/Platform/String.h>

#ifdef COURIER_ENHANCED_ENABLED
#include <Candera/EngineBase/Common/StringIdentifier.h>
#endif

namespace Courier {
    /// @addtogroup COURIER_UTIL
    /// @{

    class HashId {
        public:
            static const UInt32 cInvalidHashValue = FeatStd::Hash::cInvalidHashValue;

            HashId() {
                mHash = cInvalidHashValue;
            }

            explicit HashId(const Char *name) {
                mHash = FeatStd::Hash::CalcBinBufferHash(reinterpret_cast<const UInt8*>(name), FeatStd::Internal::String::Length(name));
            }

            HashId(const Char *name, UInt32 nameLen) {
                mHash = FeatStd::Hash::CalcBinBufferHash(reinterpret_cast<const UInt8*>(name), nameLen);
            }

            explicit HashId(UInt32 hash) {
                mHash = hash;
            }

            HashId(const HashId &hashId) {
                mHash = hashId.mHash;
            }

#ifdef COURIER_ENHANCED_ENABLED
            HashId(const Candera::StringIdentifier *stringIdentifier, bool nodeOnly) {
                if ((0 != stringIdentifier) && (0 != stringIdentifier->GetId())) {
                    FeatStd::UInt32 seed = FeatStd::Hash::cInitialSeed;
                    mHash = FeatStd::Hash::cInitialHashValue;
                    do {
                        UpdateStringIdentifierHash(stringIdentifier, nodeOnly);
                    } while (!FeatStd::Hash::FinishBinBufferHash(mHash, seed));
                }
                else {
                    mHash = cInvalidHashValue;
                }
            }
#endif

            bool IsValid() const {
                return mHash != cInvalidHashValue;
            }

            UInt32 HashValue() const {
                return mHash;
            }

            bool operator==(const HashId &id) const {
                return mHash == id.mHash;
            }

            bool operator==(UInt32 hash) const {
                return mHash == hash;
            }

            bool operator!=(const HashId &id) const {
                return mHash != id.mHash;
            }

            bool operator!=(UInt32 hash) const {
                return mHash != hash;
            }

            bool operator<(const HashId &id) const {
                return mHash < id.mHash;
            }

            bool operator>(const HashId &id) const {
                return mHash > id.mHash;
            }

        private:
#ifdef COURIER_ENHANCED_ENABLED
            void UpdateBinBufferHash(const FeatStd::Char* begin, const FeatStd::Char* end)
            {
                FeatStd::Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(begin), reinterpret_cast<const UInt8*>(end), mHash);
            }

            void UpdateStringIdentifierHash(const Candera::StringIdentifier *stringIdentifier, bool nodeOnly)
            {
                if (0 != stringIdentifier->GetOwner()) {
                    UpdateStringIdentifierHash(stringIdentifier->GetOwner(), nodeOnly);
                    const FeatStd::Char* separator = "/";
                    UpdateBinBufferHash(separator, separator + 1);
                }
                const FeatStd::Char* begin = stringIdentifier->GetId();
                const FeatStd::Char* end = begin + FeatStd::Internal::String::Length(begin);
                if (nodeOnly) {
                    // StringIdentifier looks like:
                    // "/Scene2D:MyModule#MyScene2D/MyNode/MyComposite2DInstance",
                    // "/Scene:MyModule#MyScene/MyNode/MyComposite",
                    // "/Composite2D:MyModule#MyComposite2D/MyNode/MyComposite2DInstance" or
                    // "/Composite:MyModule#MyComposite/MyNode/MyCompositeInstance"
                    if (*begin == '/') {
                        // skip initial '/' character
                        ++begin;
                        while ((begin != end) && (*begin != ':')) {
                            ++begin;
                        }
                        while ((begin != end) && (*begin != '/')) {
                            ++begin;
                        }
                        if (*begin == '/') {
                            ++begin;
                        }
                    }
                }
                UpdateBinBufferHash(begin, end);
            }
#endif
            UInt32 mHash;
    };

    /// @}
}

#endif
