//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_FixedSizeList_h)
#define Courier_Util_FixedSizeList_h

#include <Courier/Base.h>
#include <Courier/Util/Limits.h>
#include <Courier/Util/TemplateOperations.h>
#include <Courier/Diagnostics/ErrorHandling.h>

#include <FeatStd/MemoryManagement/PlacementNew.h>

namespace Courier {
    /// @addtogroup COURIER_UTIL
    /// @{
    /** FixedSizeList is a container with fixed number of items
        FixedSizeList does not use dynamic memory allocation, all memory is
        reserved statically at compile time. Insert and remove operations are
        fast, random item access time independent from number of elements in
        the container.
        @param T the item type
        @param maxSize capacity of the container */
    template<typename T, FeatStd::SizeType maxSize> class FixedSizeList {
        public:
            static const FeatStd::SizeType cCapacity = maxSize;

            //@{
            /** FixedSizeList default and copy ctors */
            FixedSizeList() {
                Init();
            }

            FixedSizeList(const FixedSizeList &fsList) {
                Init();
                *this = fsList;
            }

            template<typename T1, FeatStd::SizeType maxSize1> FixedSizeList(const FixedSizeList<T1, maxSize1> &fsList) {
                Init();
                *this = fsList;
            }
            //@}

            /** FixedSizeList dtor - destructs all objects in the container */
            ~FixedSizeList() {
                Clear();
            }

            /** return the number of items in the container
                @return number of items in the container */
            inline  FeatStd::SizeType Count() const {
                return  FeatStd::SizeType(mCount);
            }

            /** returns the maximum number of items that can be stored in the container
                @return the maximum number of items  */
            inline  FeatStd::SizeType Capacity() const {
                return maxSize;
            }

            //@{
            /** returns the item at position index in the container
                @param index where 0 <= index < Count()
                @return the item at position index */
            inline T& Item(FeatStd::SizeType index) {
                FEATSTD_DEBUG_ASSERT(index < mCount);
                return *ItemPtr(index);
            }

            inline const T& Item(FeatStd::SizeType index) const {
                FEATSTD_DEBUG_ASSERT(index < mCount);
                return *ItemPtr(index);
            }
            //@}

            //@{
            /** operator[] returns the list item with the given index
                @param index where 0 <= index < Count()
                @return the item at position index */
            inline T& operator[](FeatStd::SizeType index) {
                return Item(index);
            }

            inline const T& operator[](FeatStd::SizeType index) const {
                return Item(index);
            }
            //@}

            /** appends the given item at the end of the container
                @param item the item to insert
                @return true if the item could be appended, false otherwise */
            bool PushBack(const T &item) {
                if (Count() >= Capacity()) {
                    return false;
                }
                FeatStd::MemoryManagement::Construct(ItemPtr(mCount), item);
                ++mCount;
                return true;
            }

            /** inserts the given item before the given position
                @param index the index the item shall be inserted before - 0 <= index <= Count()
                @param item the item to insert
                @return true if the item could be inserted, false otherwise */
            bool InsertAt(FeatStd::SizeType index, const T &item) {
                bool ok = Internal::ShiftUp(index, mIndexMap, mCount, cCapacity);
                if (ok) {
                    FeatStd::MemoryManagement::Construct(ItemPtr(index), item);
                }
                return ok;
            }

            /** removes the item at the given position from the container
                @param index the position of the item that shall be removed
                @return true if the item could be removed, false otherwise */
            bool RemoveAt(FeatStd::SizeType index) {
                bool ok = index < mCount;
                if (ok) {
                    FeatStd::MemoryManagement::Destruct(ItemPtr(index));
                    ok = Internal::ShiftDown(index, mIndexMap, mCount);
                }
                return ok;
            }

            /**
                <summary>Moves item at index src to index dst</summary>
                <param name="dst">Destination index for the item.</param>
                <param name="src">Source index for the item.</param>
                <returns>true item could be moved, false otherwise.</returns>
             */
            inline bool Move(FeatStd::SizeType dst, FeatStd::SizeType src) {
                return Internal::Move(dst, src, mIndexMap, mCount);
            }

            /** removes all items from the container */
            void Clear() {
                for (FeatStd::SizeType i = 0; i < Count(); ++i) {
                    FeatStd::MemoryManagement::Destruct(ItemPtr(i));
                }
                mCount = 0;
            }

            //@{
            /** assignment operator
                @param fsList the FixedSizeList to copy items from
                @return the result of the assignment */
            template<typename T1, FeatStd::SizeType maxSize1> FixedSizeList& operator=(const FixedSizeList<T1, maxSize1> &fsList) {
                Clear();
                FeatStd::SizeType last = (Capacity() < fsList.Count()) ? Capacity() : fsList.Count();
                while (mCount < last) {
                    PushBack(fsList.Item(mCount));
                }
                return *this;
            }

            FixedSizeList& operator=(const FixedSizeList &fsList) {
                Clear();
                while (mCount < fsList.Count()) {
                    PushBack(fsList.Item(mCount));
                }
                return *this;
            }
            //@}

        private:
            enum {
                SizeTypeBufferSize = (sizeof(T[maxSize]) + sizeof(FeatStd::SizeType) - 1) / sizeof(FeatStd::SizeType)
            };

            typedef typename Courier::Internal::TaylorUnsignedIntType<maxSize>::Type SizeType;

            SizeType mCount;
            SizeType mIndexMap[maxSize];
            FeatStd::SizeType mItem[SizeTypeBufferSize];

            inline void* Buffer() {
                return mItem;
            }

            inline const void* Buffer() const {
                return mItem;
            }

            inline T* ItemPtr(FeatStd::SizeType index) {
                return reinterpret_cast<T*>(Buffer()) + Map(index);
            }

            inline const T* ItemPtr(FeatStd::SizeType index) const {
                return reinterpret_cast<const T*>(Buffer()) + Map(index);
            }

            inline FeatStd::SizeType Map(FeatStd::SizeType index) const {
                if (index >= Capacity())
                {
                    FEATSTD_PANIC("FixedSizeList: out of bound access");
                    return 0;
                } 
                else
                {
                    return mIndexMap[index];
                }
            }

            void Init() {
                mCount = 0;
                for (FeatStd::SizeType i = 0; i < maxSize; ++i) {
                    mIndexMap[i] = SizeType(i);
                }
            }
    };
    /// @}
} // namespace
#endif // Courier_Util_FixedSizeList_h
