//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_ArrayCopy_h)
#define Courier_Util_ArrayCopy_h

#include <Courier/Util/Traits.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>
#include <FeatStd/Platform/Memory.h>

namespace Courier { namespace Internal {

    /// @addtogroup COURIER_UTIL
    /// @{
    /** implements the array copy function for non POD types
        @tparam length number of elements in the array
        @tparam T element type of the array
        @tparam isPodType indicates if the type is a POD type */
    template <size_t length, typename T, bool isPodType = false> struct ArrayCopyTrait
    {
    /** implements the array copy function for non POD types
        @param dest destination array of the copy operation
        @param source source array of the copy operation */
        static void Copy(T (& dest)[length], const T * source)
        {
            for (size_t i= 0; i < length; ++i) {
                dest[i] = source[i];
            }
        }
    /** implements the array initialization function for non POD types
        @param dest destination array of the initialization operation */
        static void Init(T (& dest)[length])
        {
            for (size_t i= 0; i < length; ++i) {
                dest[i] = T();
            }
        }
    };

    /** implements the array copy function for POD types
        @tparam length number of elements in the array
        @tparam T element type of the array
        @tparam isPodType indicates if the type is a POD type */
    template<size_t length, typename T> struct ArrayCopyTrait<length, T, true>
    {
    /** implements the array copy function for non POD types
        @param dest destination array of the copy operation
        @param source source array of the copy operation */
        static void Copy(T (& dest)[length], const T *  source)
        {
            FeatStd::Internal::Memory::Copy(dest, source, sizeof(T[length]));
        }

    /** implements the array initialization function for non POD types
        @param dest destination array of the initialization operation */
        static void Init(T (& dest)[length])
        {
            FeatStd::Internal::Memory::Set(dest, 0, sizeof(T[length]));
        }
    };

}} // namespace

namespace Courier {

    /** implements the array copy function
        @tparam length number of elements in the array
        @tparam T element type of the array
        @param dest destination array of the copy operation
        @param source source array of the copy operation */
    template<size_t length, typename T> void ArrayCopy(T (& dest)[length], const T *  source)
    {
        ::Courier::Internal::ArrayCopyTrait<length, T, ::FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Copy(dest, source);
    }

    /** implements the array initialization function
        @tparam length number of elements in the array
        @tparam T element type of the array
        @param dest destination array of the initialization operation  */
    template<size_t length, typename T> void ArrayInit(T (& dest)[length])
    {
        ::Courier::Internal::ArrayCopyTrait<length, T, ::FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Init(dest);
    }

    /// @}
} // namespace
#endif // Courier_Util_ArrayCopy_h
