//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Platform_MessageQueue_h)
#define Courier_Platform_MessageQueue_h

#include <Courier/Platform/Base.h>
#include <Courier/Platform/Ticks.h>

namespace Courier {
    class MessageReferrer;
    class Message;
}

namespace Courier { namespace Platform {
    /** Observe Function prototype.
        @param  pointer to a data item, may be zero and depends on the implementation of the function itself.
        @return msg the message itself.*/
    typedef void (*MessageObserveFct)(void * data, const Message * msg);
}}

#include COURIER_PLATFORM_INCLUDE(MessageQueue)

namespace Courier { namespace Platform {
    /// @addtogroup COURIER_PLATFORM
    /// @{
    /** */
    class MessageQueue : private Impl::MessageQueue {
        typedef Impl::MessageQueue Base;
    public:
        /** Adds (pushes) a MessageReferrer to the queue.
            @param  msgRef      MessageReferrer, which should be added at the
                                end of the queue.
            @return <em>true</em>   Successfully push message into queue,
                    <em>false</em>  failed to push message into queue.*/
        bool Push(const Courier::MessageReferrer & msgRef);

        /** Retrieves and deletes (pops) a MessageReferrer from the queue.
            @param  msgRef      MessageReferrer, which should be added at the end of the queue.
            @return <em>true</em>   if next message in queue could be read.
                    <em>false</em>  if no message is in queue.
                                    Note: In this case parameter <em>msgRef</em> is undefined. */
        bool Pop(Courier::MessageReferrer & msgRef);

        /** Waits until the first entry of the queue can be received.
            The first entry will be deleted after receiving.
            @param  msgRef  Out-parameter, which contains the first entry of the queue.
            @param  timeout Timeout in milliseconds for waiting.
            @param  priorityOnly Only check priority message queue.
            @return <em>true</em>   if the first entry could
                                    be received before the specified timeout.
                    <em>false</em>  if no message is received.
                                    Note: In this case parameter <em>msgRef</em> is undefined. */
        bool Wait(Courier::MessageReferrer & msgRef, const Ticks & timeout, bool priorityOnly);

        /** Returns the size of the message queue.
            @return Size of message queue. */
        FeatStd::SizeType GetSize() const;

        /** Returns the number of priority messages in the message queue.
            @return Size of message queue. */
        FeatStd::SizeType GetPriorityMsgsSize() const;

        /** Returns the number of priority messages in the message queue.
            @return Size of message queue. */
        FeatStd::SizeType GetNormalMsgsSize() const;

        /** Calls MessageObserveFct for each Message inside the message queue.
            @param  data to a data item, may be zero and depends on the implementation of the MessageObserveFct.
            @param  fct the function which shall be called.*/
        void Observe(void * data, MessageObserveFct fct);
   };

    /*  --------------------------------------------------------------------
        Push
        -------------------------------------------------------------------- */
    inline bool MessageQueue::Push(const Courier::MessageReferrer & msgRef)
    {
        return Base::Push(msgRef);
    }

    /*  --------------------------------------------------------------------
        Pop
        -------------------------------------------------------------------- */
    inline bool MessageQueue::Pop(Courier::MessageReferrer & msgRef)
    {
        return Base::Pop(msgRef);
    }

    /*  --------------------------------------------------------------------
        Wait
        -------------------------------------------------------------------- */
    inline bool MessageQueue::Wait(Courier::MessageReferrer & msgRef, const Ticks & timeout, bool priorityOnly = false)
    {
        return Base::Wait(msgRef, timeout, priorityOnly);
    }

    /*  --------------------------------------------------------------------
        GetSize
        -------------------------------------------------------------------- */
    inline FeatStd::SizeType MessageQueue::GetSize() const
    {
        return Base::GetSize();
    }

    /*  --------------------------------------------------------------------
        GetPriorityMsgsSize
        -------------------------------------------------------------------- */
    inline FeatStd::SizeType MessageQueue::GetPriorityMsgsSize() const
    {
        return Base::GetPriorityMsgsSize();
    }

    /*  --------------------------------------------------------------------
        GetNormalMsgsSize
        -------------------------------------------------------------------- */
    inline FeatStd::SizeType MessageQueue::GetNormalMsgsSize() const
    {
        return Base::GetNormalMsgsSize();
    }

    /*  --------------------------------------------------------------------
        Observe
        -------------------------------------------------------------------- */
    inline void MessageQueue::Observe(void * data, MessageObserveFct fct) 
    {
        return Base::Observe(data,fct);
    }

    /// @}
}}

#endif
