//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Platform_Impl_Os_Generic_GenericMessageQueue_h)
#define Courier_Platform_Impl_Os_Generic_GenericMessageQueue_h

#include <Courier/Base.h>
#include <Courier/Messaging/MessageReferrer.h>
#include <Courier/Util/LinkedList.h>
#include <Courier/Platform/CriticalSection.h>
#include <Courier/Platform/Semaphore.h>
#include <Courier/Platform/Ticks.h>

namespace Courier { namespace UnitTest { 
    class MessageQueueTest;
    class MessageRouterTest;
}}

namespace Courier { namespace Platform { namespace Os { namespace Generic {
    /// @addtogroup COURIER_PLATFORM_GENERIC
    /// @{
    /** Message queue implements a thread safe FIFO queue for messages. */
    class GenericMessageQueue {
    public:
        ///
        GenericMessageQueue();
        ///
        ~GenericMessageQueue();

        /** Adds (pushes) a MessageReferrer to the queue.
            @param  msgRef      MessageReferrer, which should be added at the
                                end of the queue.
            @return <em>true</em>   Successfully push message into queue,
                    <em>false</em>  failed to push message into queue.*/
        bool Push(const Courier::MessageReferrer & msgRef);

        /** Retrieves and deletes (pops) a MessageReferrer from the queue.
            @param  msgRef      MessageReferrer, which should be added at the end of the queue.
            @return <em>true</em>   if next message in queue could be read.
                    <em>false</em>  if no message is in queue.
                                    Note: In this case parameter <em>msgRef</em> is undefined. */
        bool Pop(Courier::MessageReferrer & msgRef);

        /** Waits until the first entry of the queue can be received.
            The first entry will be deleted after receiving.
            @param  msgRef  Out-parameter, which contains the first entry of the queue.
            @param  timeout Timeout in milliseconds for waiting.
            @param  priorityOnly Only try the priority queue.
            @return <em>true</em>   if the first entry could
                                    be received before the specified timeout.
                    <em>false</em>  if no message is received.
                                    Note: In this case parameter <em>msgRef</em> is undefined. */
        bool Wait(Courier::MessageReferrer & msgRef, const Courier::Platform::Ticks & timeout, bool priorityOnly = false);

        /** Returns the size of the message queue.
            @return Size of message queue. */
        FeatStd::SizeType GetSize() const;

        /** Returns the number of priority messages in the message queue.
            @return Number of priority messages in queue. */
        FeatStd::SizeType GetPriorityMsgsSize() const;

        /** Returns the number of priority messages in the message queue.
            @return Number of normal messages in queue. */
        FeatStd::SizeType GetNormalMsgsSize() const;

        /** Calls MessageObserveFct for each message inside the message queue. */
        void Observe(void * data, MessageObserveFct fct);

    private:
        /** Test private methods. */
        friend class Courier::UnitTest::MessageQueueTest;

        /** Test private methods. */
        friend class Courier::UnitTest::MessageRouterTest;

        typedef Courier::Internal::SingleLinkedList<Courier::MessageReferrer> MessageList;

        /// Critical section for message list access
        Courier::Platform::CriticalSection mMsgListCs;

        /// Message wait synchronization semaphore
        Platform::Semaphore mMsgWaitSem;

        /// List for queuing messages
        MessageList mMsgList;

        /// List for priority messages
        MessageList mPriorityMsgList;

        /// Private copy constructor to prevent from cloning.
        GenericMessageQueue(const GenericMessageQueue & msgQueue);
        /// Private assignment operator to prevent from cloning
        GenericMessageQueue & operator=(const GenericMessageQueue & msgQueue);
    };
    /// @}
}}}}

//! [PlatformSpecializationNamespaceForward]
namespace Courier { namespace Platform { namespace Impl {
    typedef Courier::Platform::Os::Generic::GenericMessageQueue MessageQueue;
}}}
//! [PlatformSpecializationNamespaceForward]

#endif
