//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MessageSubscriberListModifier.h"

#include <Courier/Platform/CriticalSectionLocker.h>

namespace Courier {  namespace Internal {
    // ------------------------------------------------------------------------
    static Platform::CriticalSection mCs;
    static const ComponentId InvalidComponentId = ComponentId(ComponentType::Invalid);

    // ------------------------------------------------------------------------
    MessageSubscriberListModifier::MessageSubscriberListModifier(
        ComponentId * list,
        UInt8 listCapacity)
        : mList(list), mListCapacity(listCapacity)
    {
        FEATSTD_DEBUG_ASSERT(0 != list);
        FEATSTD_DEBUG_ASSERT(0 != listCapacity);
    }

    // ------------------------------------------------------------------------
    bool MessageSubscriberListModifier::Add(ComponentId componentId, ComponentId before) {
        FEATSTD_DEBUG_ASSERT(InvalidComponentId != componentId);
        Platform::CriticalSectionLocker lLock(&mCs);
        bool lInserted = false;
        UInt8 lCount = Count();
        if ((lCount != Capacity()) && (!Exists(componentId))) {
            // Add new command ID at the end
            if (InvalidComponentId == before) {
                *(mList + lCount) = componentId;
                lInserted = true;
            }
            // Add new command ID before given command ID (parameter "before")
            else {
                ComponentId lShiftId = InvalidComponentId;
                for (UInt8 i = 0; i < mListCapacity; i++) {
                    if (lInserted) {
                        ComponentId lTemp = *(mList + i);
                        *(mList + i) = lShiftId;
                        lShiftId = lTemp;
                    } else {
                        if (before == *(mList + i)) {
                            lShiftId = before;
                            lInserted = true;
                            *(mList + i) = componentId;
                        }
                    }
                }
            }
        }
        return lInserted;
    }

    // ------------------------------------------------------------------------
    bool MessageSubscriberListModifier::Remove(ComponentId componentId) {
        FEATSTD_DEBUG_ASSERT(InvalidComponentId != componentId);
        Platform::CriticalSectionLocker lLock(&mCs);
        bool lShift = false;
        if ((InvalidComponentId != componentId) && (0 != Count())) {
            for (UInt8 i = 0; i < mListCapacity; i++) {
                // If item to remove found, shift all succeeding values by one to left
                lShift = lShift || (componentId == *(mList + i));
                if (lShift) {
                    if (i == (mListCapacity - 1)) {
                        *(mList + i) = InvalidComponentId;
                    } else {
                        *(mList + i) = *(mList + i + 1);
                    }
                }
            }
        }
        return lShift;
    }

    // ------------------------------------------------------------------------
    bool MessageSubscriberListModifier::Exists(ComponentId componentId) const {
        FEATSTD_DEBUG_ASSERT(InvalidComponentId != componentId);
        Platform::CriticalSectionLocker lLock(&mCs);
        bool lExists = false;
        for (UInt8 i = 0; ((!lExists) && (i < mListCapacity)); i++) {
            lExists = (componentId == *(mList + i));
        }
        return lExists;
    }

    // ------------------------------------------------------------------------
    UInt8 MessageSubscriberListModifier::Count() const {
        Platform::CriticalSectionLocker lLock(&mCs);
        UInt8 lCount = 0;
        while ((lCount < mListCapacity) && (InvalidComponentId != *(mList + lCount))) {
            ++lCount;
        }
        return lCount;
    }

    /// @}
}}
