//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Messaging_MessageRouter_h)
#define Courier_Messaging_MessageRouter_h

#include <Courier/Base.h>
#include <Courier/Foundation/Component.h>
#include <Courier/Util/FixedSizeList.h>

namespace Courier {
    class Message;
    class MessageReceiver;
    class MessageReferrer;
}

namespace Courier { namespace UnitTest {
    class MessageRouterTest;
}}

namespace Courier { namespace UnitTest { namespace Helper {
    class CourierEnvironment;
}}}

namespace Courier {  namespace Internal {
    /// @addtogroup COURIER_MESSAGING
    /// @{
    /** Signature of message hook function */
    typedef bool (*MessageHookSignature)(Message *msg);

    /** MessageRouter is a static class to collect the base functionality of message routing.
    Posting a message in general means to enqueue the message to the message-queues of
    message receiving entities, like application or service. */

    class MessageRouter {
    public:
        /** Initialize message router system. */
        static void Init();

        /** Distributes an asynchronous message to the system.
            @note The caller has to take care that the message instance is all
            available through the routing process. Message instance is automatically
            destroyed, when routing process has finished.
            @param msg Message to post
            @return <em>true</em>   If message was successfully distributed,
                    <em>false</em>  message reception failed in at least one registered receiver. */
        static bool Post(Message * msg);

        /** Registers a given message receiver to the message router's static receiver list.
            @param msgReceiver Message receiver to register. */
        static void RegisterReceiver(MessageReceiver * msgReceiver);

        /** De-registers a given message receiver from the message router's static receiver list.
            @param msgReceiver Message receiver to de-register. */
        static void DeregisterReceiver(MessageReceiver * msgReceiver);

        /** Registers the given component to message receiver map entry to the message router.
            @param msgReceiver Message receiver part of the component to message receiver map entry.
            @param componentId Component ID part of the component to message receiver map entry. */
        static void RegisterComponentAtReceiver(MessageReceiver * msgReceiver, ComponentId componentId);

        /** De-registers the given component to message receiver map entry to the message router.
            @param msgReceiver Message receiver part of the component to message receiver map entry.
            @param componentId Component ID part of the component to message receiver map entry. */
        static void DeregisterComponentFromReceiver(MessageReceiver * msgReceiver, ComponentId componentId);

        /** returns the current message hook function
            Note that hooking functions GetMessageHook and SetMessageHook are not thread safe.
            @return the current message hook function pointer or 0 (null) if none set */
        static MessageHookSignature GetMessageHook();

        /** sets a new message hook function
            The message hook function will be invoked with all posted messages before message distribution.
            Note that hooking functions GetMessageHook and SetMessageHook are not thread safe. The hook function
            may be invoked from different thread context.

            If the hooking function returns true, the given message is processed (distributed) normally. Message
            distribution will be abandoned if the hooking function return false.

            @param aMessageHook the pointer to the hooking function or 0 (null) to abandon hooking.
            @return the current set hook function or 0 (null) if no hook is set. */
        static MessageHookSignature SetMessageHook(MessageHookSignature aMessageHook);

    private:
        FEATSTD_MAKE_CLASS_STATIC(MessageRouter);

        /// Test private methods.
        friend class Courier::UnitTest::Helper::CourierEnvironment;
        /// Test private methods.
        friend class Courier::UnitTest::MessageRouterTest;

        /// Component receiver map.
        static MessageReceiver ** mComponentReceiverMap;

        /// Receiver list.
        static MessageReceiver ** mReceiverList;

        /// Number of receivers registered in receiver list.
        static UInt32 mReceiverListCount;

        /** Performs sequential message distribution.
            @param msgRef MessageReferrer referring to the Message to be distributed.
            @return <em>true</em>   If message was successfully distributed,
                    <em>false</em>  message reception failed in at least one registered receiver. */
        static bool DoSequentialDistribution(const MessageReferrer & msgRef);

        /** Performs parallel message distribution.
            @param msgRef MessageReferrer referring to the Message to be distributed.
            @return <em>true</em>   If message was successfully distributed,
                    <em>false</em>  message reception failed in at least one registered receiver. */
        static bool DoParallelDistribution(const MessageReferrer & msgRef);
    };
    /// @}
}}

#endif
