//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MessageReceiver.h"

#include "MessageRouter.h"

#include <Courier/Diagnostics/Log.h>

namespace Courier {

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

    // ------------------------------------------------------------------------
    MessageReceiver::MessageReceiver() : mShutdownTriggered(false)
    {
    }

    // ------------------------------------------------------------------------
    MessageReceiver::~MessageReceiver()
    {
        Deregister();
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Activate()
    {
        // Register receiver at message router
        Register();
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Deactivate()
    {
        // Deregister receiver at message router
        Deregister();
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::SetName(const Char * name)
    {
        if (0 != name) {
            mName = name;
        } else {
            typedef char MisraChar;
            ::sprintf(static_cast<MisraChar *>(const_cast<MisraChar *>(mName.CStr())), "0x%p", this);
        }
    }

    // ------------------------------------------------------------------------
    const Char * MessageReceiver::GetName() const 
    { 
        return FeatStd::Internal::ToCStr(mName); 
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Shutdown()
    {
        // De-register from message router
        Deregister();

        // Set shutdown flag for gracefully termination
        mShutdownTriggered = true;
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Register()
    {
        // Register receiver at message router
        COURIER_LOG_INFO("Register message receiver %s", mName.CStr());
        Internal::MessageRouter::RegisterReceiver(this);
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Deregister()
    {
        // De-register receiver from message router
        COURIER_LOG_INFO("De-register message receiver %s", mName.CStr());
        Internal::MessageRouter::DeregisterReceiver(this);
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::AttachPreprocessor(MessageReceiverPreprocessor * mp)
    {
        FEATSTD_DEBUG_ASSERT(0 != mp);
        if (0 != mp) {
            mMessageReceiverPreprocessorList.Prepend(mp);
        }
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::DetachPreprocessor(MessageReceiverPreprocessor * mp)
    {
        FEATSTD_DEBUG_ASSERT(0 != mp);
        if (0 != mp) {
            (void)mMessageReceiverPreprocessorList.Remove(mp);
        }
    }

    // ------------------------------------------------------------------------
    void MessageReceiver::Preprocess()
    {
        for (MessageReceiverPreprocessorList::Iterator it = mMessageReceiverPreprocessorList.Begin(); it != mMessageReceiverPreprocessorList.End(); ++it) {
            MessageReceiverPreprocessor * mp = &*it;
            (void)mp->Process();
        }
    }
}
