//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Message.h"

#include "MessageRouter.h"
#include "MessageSubscriberListModifier.h"

#include <Courier/Platform/MessageFactory.h>

#if defined(COURIER_MESSAGE_SERIALIZATION_ENABLED)
    #include <Courier/Serialization/Serialization.h>
#endif

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
    #include <Courier/Platform/Ticks.h>
#endif

namespace Courier {

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

    static Message::MessageHookSignature gMessageHook = 0;   ///< pointer to the active message hook

    // ------------------------------------------------------------------------
    Message::Message() : mLocalSubscriberList(0), mSequenceNumber(0), mCurrentRecipientIndex(0)
    {
        *mRefCount = 0;
    }

    // ------------------------------------------------------------------------
    Message::~Message()
    {
        FEATSTD_DEBUG_ASSERT(0 == *mRefCount);
        mSequenceNumber = 0;
        mCurrentRecipientIndex = 0;
        mLocalSubscriberList = 0;
    }

    // ------------------------------------------------------------------------
    bool Message::Subscribe(ComponentId * subscriptionList, UInt8 subscriptionListSize, ComponentId componentId, ComponentId before) {
        Internal::MessageSubscriberListModifier lListModifier(
            const_cast<ComponentId *>(subscriptionList),
            subscriptionListSize);

        return lListModifier.Add(componentId, before);
    }

    // ------------------------------------------------------------------------
    bool Message::Unsubscribe(ComponentId * subscriptionList, UInt8 subscriptionListSize, ComponentId componentId) {
        Internal::MessageSubscriberListModifier lListModifier(
            const_cast<ComponentId *>(subscriptionList),
            subscriptionListSize);
        return lListModifier.Remove(componentId);
    }

    // ------------------------------------------------------------------------
    MessageSubscriberList Message::GetSubscriberList() {
        // Subscriber list is only valid in case of a subscription message
        if (!IsBroadcast()) {
            // If message's subscriber list is modifiable use local copy of subscriber list
            if (HasModifiableSubscriptionList()) {
                return MessageSubscriberList(mLocalSubscriberList,
                                             GetMetadata().mCategory.mSubscriptionListSize,
                                             mCurrentRecipientIndex);
            }
            // If message's subscriber list is NOT modifiable use static subscriber list of message class
            else {
                return MessageSubscriberList(GetMetadata().mStaticSubscriptionList,
                                             GetMetadata().mCategory.mSubscriptionListSize,
                                             mCurrentRecipientIndex);
            }
        }
        // If message is of type broadcast, return an empty list
        else {
            return MessageSubscriberList(0, 0, mCurrentRecipientIndex);
        }
    }

    // ------------------------------------------------------------------------
    bool Message::Post(bool updateSubscriptionList)
    {
        if(mSequenceNumber==0) {
            // set the sequence number if not set
            mSequenceNumber = static_cast<UInt32>(Platform::AtomicOp::Inc(GetModifiableMetadata().mSequenceNumber));
        }

        // If set and this is the initial post of the message, invoke gMessageHook.
        // Initial post means that the index of the recipient list is 0 (messages get reposted in sequential distribution).
        if ((0 != gMessageHook) && (0 == CurrentRecipient()) && (! gMessageHook(this))) {
            // if the message hook returned false, the distribution will be abandoned
            COURIER_LOG_DEBUG("MessageHook suppresses posting %s", GetName());
            return true;
        }

        // Update subscriber list before posting the message
        if (updateSubscriptionList) {
            UpdateLocalSubscriberList();
        }
        return Internal::MessageRouter::Post(this);
    }

    // ------------------------------------------------------------------------
    bool Message::PostToNextRecipient()
    {
        ++mCurrentRecipientIndex;
        return Post(false);
    }

#if defined(COURIER_IPC_ENABLED)
    // ------------------------------------------------------------------------
    bool Message::IsMarkedForLocalDistributionOnly() const {
        return (!HasSequentialDistribution()) && (!IsBroadcast()) && (0 != mCurrentRecipientIndex);
    }

    // ------------------------------------------------------------------------
    bool Message::MarkForLocalDistributionOnly() {
        bool lRc = (!HasSequentialDistribution()) && (!IsBroadcast());
        if (lRc) {
            mCurrentRecipientIndex = 0xffU;
        }
        return lRc;
    }
#endif

    // ------------------------------------------------------------------------
    void Message::InitializeLocalSubscriberList(ComponentId * mSubscribers)
    {
        if ((HasModifiableSubscriptionList()) && (!IsBroadcast())) {
            FEATSTD_DEBUG_ASSERT(0 != mSubscribers);
            mLocalSubscriberList = mSubscribers;
            ResetLocalSubscriberList();
        }
    }

    // ------------------------------------------------------------------------
    void Message::UpdateLocalSubscriberList()
    {
        if ((HasModifiableSubscriptionList()) && (!IsBroadcast())) {
            FEATSTD_DEBUG_ASSERT(0 != mLocalSubscriberList);
            if (0 != mLocalSubscriberList) {
                FeatStd::Internal::Memory::Copy(mLocalSubscriberList, GetMetadata().mStaticSubscriptionList,
                    static_cast<FeatStd::SizeType>(sizeof(ComponentId) * GetMetadata().mCategory.mSubscriptionListSize));
            }
        }
    }

    // ------------------------------------------------------------------------
    void Message::ResetLocalSubscriberList()
    {
        if ((HasModifiableSubscriptionList()) && (!IsBroadcast())) {
            FEATSTD_DEBUG_ASSERT(0 != mLocalSubscriberList);
            if (0 != mLocalSubscriberList) {
                for (FeatStd::SizeType i = 0 ; i != static_cast<FeatStd::SizeType>(GetMetadata().mCategory.mSubscriptionListSize) ; ++i)
                {
                   mLocalSubscriberList[i] = ComponentType::Invalid;
                }
            }
        }
    }

    // ------------------------------------------------------------------------
    void Message::IncRef()
    {
        (void)Platform::AtomicOp::Inc(mRefCount);
        FEATSTD_PANIC_IF((GetRefCounter() == 0), "Reference counter overflow of message %s", GetName());
    }

    // ------------------------------------------------------------------------
    void Message::DecRef()
    {
        if (Platform::AtomicOp::Dec(mRefCount) == 0) {
            COURIER_LOG_DEBUG("Delete %s", GetName()); 
            // TODO: determine cause for crash in TextBindingApplication
            Platform::MessageFactory::Destroy(this);
        }
    }

    // ------------------------------------------------------------------------
    Message::MessageHookSignature Message::GetMessageHook()
    {
        return gMessageHook;
    }

    // ------------------------------------------------------------------------
    Message::MessageHookSignature Message::SetMessageHook(MessageHookSignature messageHook)
    {
        MessageHookSignature current = gMessageHook;
        gMessageHook = messageHook;
        return current;
    }

    // ------------------------------------------------------------------------
    Message::ModifiableMetadata & Message::GetModifiableMetadata() 
    {
        static ModifiableMetadata data; 
        return data;
    }

#if defined(COURIER_MESSAGE_SERIALIZATION_ENABLED)
    // ------------------------------------------------------------------------
    const SerializationInfo * Message::GetSerializationInfo() const
    {
        static const ::Courier::SerializationInfo gInfo = { 0, 0, 0, 0 };
        return &gInfo;
    }

    // ------------------------------------------------------------------------
    bool Message::IsSerializable() const 
    {
        const SerializationInfo * info = GetSerializationInfo();
        if(info!=0) {
            return (info->mCount > 0);
        }
        return false;
    }

#else
    // ------------------------------------------------------------------------
    const SerializationInfo * Message::GetSerializationInfo() const {
        return 0;
    }

    // ------------------------------------------------------------------------
    bool Message::IsSerializable() const 
    {
        return false;
    }
#endif

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
    // ------------------------------------------------------------------------
    void Message::MonitoringData::Init() 
    { 
        if(mPostTimestamp==0) {
            mPostTimestamp = Courier::Platform::Ticks(Courier::Platform::TicksType::Now).GetTicks().Get();
            mCyclesUnprocessed = 0;
        }
    }
#endif

}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)

::FeatStd::UInt32 Message::AppendTo(::FeatStd::StringBuffer& stringBuffer) const
{
    UInt32 tcharCount = 0;
    tcharCount += stringBuffer.Append("::Courier::Message { ");
    tcharCount += stringBuffer.Append("Id = ");
    tcharCount += stringBuffer.AppendObject(GetId());
    tcharCount += stringBuffer.Append(", ModifiableSubscriptionList = ");
    tcharCount += stringBuffer.AppendObject(HasModifiableSubscriptionList());
    tcharCount += stringBuffer.Append(", SequentialDistribution = ");
    tcharCount += stringBuffer.AppendObject(HasSequentialDistribution());
    tcharCount += stringBuffer.Append(", Broadcast = ");
    tcharCount += stringBuffer.AppendObject(IsBroadcast());
    tcharCount += stringBuffer.Append(", Priority = ");
    tcharCount += stringBuffer.AppendObject(GetPriority());
    tcharCount += stringBuffer.Append(", Tag = ");
    tcharCount += stringBuffer.AppendObject(GetTag());
    tcharCount += stringBuffer.Append(", ScopeMask = ");
    tcharCount += stringBuffer.AppendObject(GetScopeMask());
    tcharCount += stringBuffer.Append(", Serializable = ");
    tcharCount += stringBuffer.AppendObject(IsSerializable());
    tcharCount += stringBuffer.Append(" }");
    return tcharCount;
}
#endif
