//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Component.h"

#include <Courier/Messaging/Message.h>
#include <Courier/Foundation/IComponentNotification.h>
#include <Courier/Diagnostics/Trace.h>
#include <Courier/Diagnostics/Log.h>
#include <FeatStd/Platform/PerfCounter.h>

#if defined(COURIER_ENHANCED_ENABLED)
    #include <Courier/DataBinding/ModelBindingSourceMap.h>
#endif
#if defined(COURIER_MESSAGE_SERIALIZATION_ENABLED)
    #include <Courier/Serialization/MessageVerification.h>
#endif

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Foundation);

    using namespace Diagnostics;
    Component * Component::mInstance[COURIER_COMPONENT_MAX_COUNT];

    Component::MessageCorruptionFct Component::mMessageCorruptionCallback = 0;

#ifdef COURIER_DATA_LOCK_ENABLED
    static Platform::CriticalSection * GetComponentDataLock()
    {
        static Platform::CriticalSection sComponentDataLock;
        return &sComponentDataLock;
    }
#endif // COURIER_DATA_LOCK_ENABLED

    // ------------------------------------------------------------------------
    void Component::AcquireDataLock() const
    {
#ifdef COURIER_DATA_LOCK_ENABLED
        // try to get the CS
        GetComponentDataLock()->Obtain();
#endif //COURIER_DATA_LOCK_ENABLED
    }

    // ------------------------------------------------------------------------
    void Component::ReleaseDataLock() const
    {
#ifdef COURIER_DATA_LOCK_ENABLED
        GetComponentDataLock()->Release();
#endif //COURIER_DATA_LOCK_ENABLED
    }

    // ------------------------------------------------------------------------
    Component::Component(ComponentId cid) : mComponentNotification(0), mComponentId(cid),
                                            mAllowMessageInterrupt(false), mMessageProcessed(false),
                                            mMinimumExecutionTime(0), mOldTick(0)
    {
        FEATSTD_COMPILETIME_ASSERT(sizeof(ComponentId) <= 4);
        FEATSTD_COMPILETIME_ASSERT(ComponentType::Invalid >= 0);
        FEATSTD_COMPILETIME_ASSERT(ComponentType::CustomerLast < ComponentType::Invalid);
        FEATSTD_DEBUG_ASSERT(cid < COURIER_COMPONENT_MAX_COUNT);

        mContinuousExecution.mValue = 0;

        FEATSTD_DEBUG_ASSERT(mInstance[cid] == 0);
        FEATSTD_LINT_NEXT_EXPRESSION(1938, "this is correct behaviour to access the static instance list here")
        mInstance[cid] = this;
    }

    // ------------------------------------------------------------------------
    Component::~Component()
    {
        mInstance[mComponentId] = 0;
        mComponentNotification = 0;
    }

    // ------------------------------------------------------------------------
    void Component::Execute()
    {
        bool contExecute = OnExecute();
        (void)Platform::AtomicOp::Set(mContinuousExecution, Platform::AtomicOp::ValueType(contExecute ? 1 : 0));
    }

    // ------------------------------------------------------------------------
    FEATSTD_LINT_NONCONST_METHOD(Courier::Component::PostProcess, "Depends on macros, can't be made const in certain configurations")
    void Component::PostProcess()
    {
        // Give binding sources the chance to do any post processing (i.e. sending ModelUpdate msg)
#if defined(COURIER_ENHANCED_ENABLED)
        if (mComponentId == Courier::ComponentType::View)
        {
            AcquireDataLock();
            Internal::ModelBindingSourceMap::DoBindingSourcePostProcessing(mComponentId);
            ReleaseDataLock();
        }
#endif
    }

    // ------------------------------------------------------------------------
    bool Component::IsContinuousExecutionEnabled()
    {
        return Platform::AtomicOp::TestAndSet(mContinuousExecution, 1, 1);
    }

    // ------------------------------------------------------------------------
    void Component::EnableContinuousExecution(bool enable)
    {
        (void)Platform::AtomicOp::Set(mContinuousExecution, Platform::AtomicOp::ValueType(enable ? 1 : 0));
    }

    // ------------------------------------------------------------------------
    void Component::SetMinExecutionCycleTime(UInt32 minimumExecutionTime, bool allowMessageInterrupt)
    {
        COURIER_LOG_INFO("Component Id(%u) MinimumExecutionCycleTime(%u ms) AllowMessageInterrupt(%s) set",
                         GetId(), minimumExecutionTime, (allowMessageInterrupt ? "ENABLED" : "DISABLED"));
        mMinimumExecutionTime = minimumExecutionTime;
        mAllowMessageInterrupt = allowMessageInterrupt;
        mOldTick = 0;
    }
    // ------------------------------------------------------------------------
    bool Component::ShallExecute(UInt32 currentTick, UInt32 & oldTick, UInt32 cycleTime)
    {
      // compute difference between current time and the last execution time point
        UInt32 diff = currentTick - oldTick;
        // if we reached the cycletime then trigger execution
        if(diff>=cycleTime) {
            // compute the cycles which may fit into the diff time.
            UInt32 cycles = diff / cycleTime;
            // usually there must be one cycle.
            COURIER_DEBUG_ASSERT(cycles>0);
            // adjust the last execution time point.
            oldTick = oldTick + (cycles * cycleTime);
            return true;
        }
        return false;
    }

    // ------------------------------------------------------------------------
    bool Component::ShallExecute(UInt32 currentTick)
    {
        bool shallExecute = true;
        if(! (mAllowMessageInterrupt && mMessageProcessed)) {
            if(mMinimumExecutionTime>0) {
                UInt32 cycleTime = (mMinimumExecutionTime * 1000) / FEATSTD_PERFCOUNTER_RESOLUTION;
                if(mOldTick==0) {
                    mOldTick = currentTick;
                }
                shallExecute = ShallExecute(currentTick,mOldTick,cycleTime);
            }
        }
        // reset the flag
        mMessageProcessed = false;
        return shallExecute;
    }

    // ------------------------------------------------------------------------
    bool Component::ProcessMessage(const Message & msg)
    {
        mMessageProcessed = true;

        COURIER_TRACE_MESSAGE(TraceId::MessageReceived, &msg);

#if defined(COURIER_IPC_ENABLED)
        if(MessageVerification::IsCorrupted(msg)) {
            if(mMessageCorruptionCallback!=0) {
                mMessageCorruptionCallback(this,msg);
            }
            COURIER_LOG_ERROR("Component Id(%u) stopped processing, corrupt Message Id (0x%08x)",this->GetId(),msg.GetId());
            // return true so that we stop message distribution
            return true;
        }
#endif

#if defined(COURIER_ENHANCED_ENABLED)
        bool lConsumed = (msg.GetTag() == MessageTag::DataBinding);
        if (lConsumed) {
            AcquireDataLock();
            lConsumed = Internal::ModelBindingSourceMap::OnDataBindingMsg(mComponentId, &msg, Candera::AbstractNodePointer());
            ReleaseDataLock();
            if (0 != mComponentNotification) {
                mComponentNotification->OnDataBindingMsg(mComponentId, &msg);
            }
        }
#else
        bool lConsumed = false;
#endif

        // Only forward message, if it was not already consumed by data binding
        if (!lConsumed) {
            lConsumed = OnMessage(msg);
        }

        // Enforce calling of Execute method
        EnableContinuousExecution(true);

        return lConsumed;
    }
}
