//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Diagnostics_ContextCheck_h)
#define Courier_Diagnostics_ContextCheck_h

#include <Courier/Base.h>
#include <Courier/Platform/Thread.h>
#include <Courier/Foundation/ComponentType.h>
#include <Courier/Util/Traits.h>

#if defined(COURIER_DISABLE_CONTEXT_AFFINITY_CHECKS)
#   if defined(COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS)
#       undef COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS
#   endif
#elif !defined(COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS)
    // enable automatically only in Win32 debug builds
#   if defined(COURIER_DEBUG) && defined(FEATSTD_OS_WIN32)
#       define COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS
#   endif
#endif

#if defined(COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS)
    namespace Courier { namespace Internal {
    /// @addtogroup COURIER_DIAGNOSTICS
    /// @{
        // ------------------------------------------------------------------------
        // unsynchronized types macros

        /** defines a type to be unsynchronized - thread safe
            The macro causes the definition of a member variable storing the thread id that
            creates an instance of the type. COURIER_CHECK_OBJECT_AFFINITY() calls in the
            types member function subsequently will check if the member function has been
            invoked by the correct thread context.
            The macro should be declared either in protected or private section of the
            class / structure. */
        #define COURIER_UNSYNCHRONIZED_TYPE(typeName) \
            mutable Courier::Internal::AffinityObject mZAffinityObject_

        /** Checks if a member function of an unsynchronized type is invoked in the correct context
            every member function in a type decorated with COURIER_UNSYNCHRONIZED_TYPE() should add
            this check in the very beginning of the function code */
        #define COURIER_CHECK_OBJECT_AFFINITY() \
            FEATSTD_DEBUG_ASSERT(CheckObjectThreadAffinity(mZAffinityObject_, Courier::Internal::IsConstObject(this)))

        /** Transfers the ownership of an unsynchronized object to the specified context
            Transfers an instance that has been created in one context to execute in a different
            context.
            @param unsyncedObject an object which type was decorated with COURIER_UNSYNCHRONIZED_TYPE
            @param contextName the name of the context the object shall be transferred to. */
        #define COURIER_TRANSFER_OBJECT_AFFINITY(unsyncedObject, contextName) \
            Courier::Internal::GetAffinityObject(unsyncedObject).TransferAffinity(contextName::GetThreadId())

        /** Transfers the ownership of an unsynchronized object to the current context
            Transfers an instance that has been created in one context to execute in the current
            context.
            @param unsyncedObject an instance of a type decorated with COURIER_UNSYNCHRONIZED_TYPE */
        #define COURIER_TRANSFER_OBJECT_AFFINITY_TO_CURRENT(unsyncedObject) \
            COURIER_TRANSFER_OBJECT_AFFINITY(unsyncedObject, Courier::Internal::CurrentThreadContext)

        /** Orchestrate the context of an unsyncronized type that is composed from other unsychronized types
            pass the name of all unsycnronized member to the macro.

            *Note* by aggregating unsyncronized objects the life time of these objects is bound too.
            This means that all aggregated object must be destroyed together. To remove a specific
            unsynchronized object use COURIER_REMOVE_FROM_AFFINITY_AGGREGATION */
        #define COURIER_AGGREGATE_OBJECT_AFFINITY(...) \
            Courier::Internal::LinkAffinityObjects(mZAffinityObject_, __VA_ARGS__)

        /** remove the given unsychronized object from the aggregation
            @param unsyncedObject the unsynchronized object to remove from the aggregation */
        #define COURIER_REMOVE_FROM_AFFINITY_AGGREGATION(unsyncedObject) \
            Courier::Internal::GetAffinityObject(unsyncedObject).Unlink()

        /** Mark all given unsyncronized objects as externally synced.
            An unsyncronized object may be used by different thread context if an external entity
            cares for synchronized access to the object. Such objects must be declared as externally
            synchronized - meaning that multiple threads can access the object. */
        #define COURIER_DECLARE_EXTERNALLY_SYNCED(...) \
            Courier::Internal::SetAffinityObjectExternallySynced(__VA_ARGS__);

        /** Sets the shared read only state of the given object
            @param unsyncedObject an instance of a type decorated with COURIER_UNSYNCHRONIZED_TYPE
            @param sharedReadOnly true if the object should enter the state shared read only, false otherwise */
        #define COURIER_DECLARE_SHARED_READONLY(unsyncedObject, sharedReadOnly) \
            Courier::Internal::GetAffinityObject(unsyncedObject).SetSharedReadOnly(sharedReadOnly)

        // ------------------------------------------------------------------------
        // static context binding

        /** Defines the context for the current scope.
            Defines context that can be checked with COURIER_CHECK_SCOPE_AFFINITY.
            accepts a list of context names that are valid for the current scope */
        #define COURIER_DEFINE_SCOPE_AFFINITY(...) \
            typedef Courier::Internal::ThreadContextVerifier<__VA_ARGS__> ScopeAffinity

        /** Checks if the current context matches the context defined by COURIER_DEFINE_SCOPE_AFFINITY for the current scope. */
        #define COURIER_CHECK_SCOPE_AFFINITY() \
            FEATSTD_DEBUG_ASSERT(ScopeAffinity::Check())

        /** checks if the current context matches one of the given context
            receives a list of context that are valid for this function */
        #define COURIER_EXPECT_CONTEXT(...) \
            FEATSTD_DEBUG_ASSERT(Courier::Internal::ThreadContextVerifier< __VA_ARGS__ >::Check())

        // ------------------------------------------------------------------------
        // context definition

        /** defines a thread context with the given name
            The context name will be used to define a type and thus must not conflict with other
            type names.
            @param contextName the name of the context */
        #define COURIER_DEFINE_THREAD_CONTEXT(contextName) \
            struct FEATSTD_CONCAT2(ThreadContextType, contextName) { }; \
            typedef Courier::Internal::ThreadContext< FEATSTD_CONCAT2(ThreadContextType, contextName) > contextName

        /** Set the thread id of the given context to the specified thread id
            @param contextName the name of the context
            @param threadId the thread id of the context */
        #define COURIER_SET_CONTEXT_THREAD_ID(contextName, threadId) \
            contextName::SetThreadId(threadId)

        /** Set the thread id of the given context to the current thread id
            @param contextName the name of the context */
        #define COURIER_SET_CONTEXT_THREAD_ID_TO_CURRENT(contextName) \
            COURIER_SET_CONTEXT_THREAD_ID(contextName, Courier::Platform::Thread::GetCurrentId())

        // ------------------------------------------------------------------------
        /** class ThreadContext defines for a given (thread) type a context class.
            COURIER_DEFINE_THREAD_CONTEXT uses this template to generate for each thread context
            a unique type that stores the thread id
            @tparam T type specific for a thread context */
        template<typename T> class ThreadContext {
            public:
                /** returns the context thread id
                    @return thread id */
                static inline UInt32 GetThreadId() {
                    return mThreadId;
                }

                /** set the context thread id
                    @param value the id of the thread associated with this context */
                static inline void SetThreadId(UInt32 value) {
                    mThreadId = value;
                    mIsSet = true;
                }

                /** Checks if the given thread id matches the context thread id
                    @param threadId the thread id to check
                    @return true if the thread id matches, false otherwise */
                static inline bool Check(UInt32 threadId) {
                    return (!mIsSet) || (mThreadId == threadId);
                }

            private:
                static bool mIsSet;
                static UInt32 mThreadId;
        };

        template<typename T> UInt32 ThreadContext<T>::mThreadId;
        template<typename T> bool ThreadContext<T>::mIsSet = false;

        // ------------------------------------------------------------------------

        /** thread context class for the current thread */
        struct CurrentThreadContext {
            /** returns the context thread id
                @return thread id */
            static inline UInt32 GetThreadId() {
                return Platform::Thread::GetCurrentId();
            }

            /** Checks if the given thread id matches the context thread id
                @param threadId the thread id to check
                @return true if the thread id matches, false otherwise */
            static inline bool Check(UInt32 threadId) {
                return Platform::Thread::GetCurrentId() == threadId;
            }
        };

        // ------------------------------------------------------------------------

        /** AnyThreadContext is a dummy thread context class.
            The class sole purpose is to be the default type the type params of
            Courier::Internal::ThreadContextVerifier */
        struct AnyThreadContext {
            static inline bool Check(UInt32 threadId) {
                FEATSTD_UNUSED(threadId);
                return false;
            }
        };

        // ------------------------------------------------------------------------

        /** used by macro COURIER_UNSYNCHRONIZED_TYPE
            Declares a struct / class member that stores the owning context thread id.
            Macros COURIER_CHECK_OBJECT_AFFINITY, COURIER_TRANSFER_OBJECT_AFFINITY etc.
            use this member to read and write the objects owning thread id. */
        class AffinityObject {
            public:
                /** ctor - initializes the thread affinity to the current thread */
                AffinityObject();

                /** checks if the object is correctly destroyed by the affine thread */
                ~AffinityObject();

                /** returns the thread id of the thread affine to this object
                    @return affine thread id */
                UInt32 GetThreadId() const {
                    return Resolve().mData.mThreadId;
                }

                /** Returns if this affinity object is linked or not
                    @return true if the object is linked, false otherwise */
                bool IsLinked() const {
                    return mIsLinked;
                }

                /** Returns the linked affinity object
                    @return if the object is linked, the linked affinity object. or this if unlinked. */
                const AffinityObject& ResolveLinkedObject() const;

                /** Returns if the object is synchronized externally and therefore is thread safe.
                    @return true if the object is externally synced, false otherwise */
                bool IsExternallySynced() const {
                    return Resolve().mIsSynced;
                }

                /** Marks the object to be externally synced.
                    External synchronization means that the objects thread safety is managed
                    by external means.
                    @param isExternallySynced the new externally linked state */
                void SetIsExternallySynced(bool isExternallySynced) {
                    Resolve().mIsSynced = isExternallySynced;
                }

                /** returns true if the object has been marked to be shared read only.
                    @return true if object is shared read only, false otherwise */
                bool IsSharedReadOnly() const {
                    return Resolve().mSharedReadOnly;
                }

                /** Set the shared read only state fir the object
                    @param sharedReadOnly the new shared read only state of the object */
                void SetSharedReadOnly(bool sharedReadOnly) {
                    Resolve().mSharedReadOnly = sharedReadOnly;
                }

                /** transfers thread affinity to the specified thread
                    @param threadId the new affine thread id */
                void TransferAffinity(UInt32 threadId) {
                    Resolve().mData.mThreadId = threadId;
                }

                /** links the affinity object to the given affinity object
                    Linking affinity objects means that they share the thread id they are affine to.
                    When the affine thread is changed with TransferAffinity, the transfer applies to
                    all linked affinity objects.
                    This is how unsyncronized types aggregated from other unsynchronized types can
                    share the affine thread id.
                    @param affinityObject the affinity object a link should be established to
                    @return the resolved linked affinity object */
                AffinityObject& LinkTo(AffinityObject &affinityObject);

                /** Unlink the affinity object from the aggregation */
                void Unlink();

            private:
                bool mIsLinked;
                bool mIsSynced;
                bool mSharedReadOnly;
                union {
                    AffinityObject *mLinkedAffinityObject;
                    UInt32 mThreadId;
                } mData;

                /** Returns either this if not linked or the linked affinity object
                    @return an affinity object (either this or the linked object) */
                const AffinityObject& Resolve() const {
                    return ResolveLinkedObject();
                }

                /** Returns either this if not linked or the linked affinity object
                    @return an affinity object (either this or the linked object) */
                AffinityObject& Resolve();
        };

        // ------------------------------------------------------------------------

        /// @{
        /** Used to access affinity objects disregarding their access level
            COURIER_UNSYNCHRONIZED_TYPE will define automatically the AffinityObjectAccessor as
            friend of the class hosting the affinity object
            @tparam T The type declared as COURIER_UNSYNCHRONIZED_TYPE */
        template<typename T> struct AffinityObjectAccessor {
            /** return the objects affinity object
                @param object the object with an affinity object
                @return the objects affinity object */
            static inline AffinityObject& Get(T &object) {
                return object.mZAffinityObject_;
            }
        };

        template<> struct AffinityObjectAccessor<AffinityObject> {
            /** return the objects affinity object
                @param object the object with an affinity object
                @return the objects affinity object */
            static inline AffinityObject& Get(AffinityObject &object) {
                return object;
            }
        };
        /// @}

        /** Helper function to get the objects affinity object through the AffinityObjectAccessor */
        template<typename T> inline AffinityObject& GetAffinityObject(T &object) {
            return AffinityObjectAccessor<T>::Get(object);
        }

        // ------------------------------------------------------------------------

        template<typename T> struct AffinityObjectLinker {
            static inline AffinityObject& Link(T &object) {
                return GetAffinityObject(object);
            }
        };

        template<typename T, UInt32 N> struct AffinityObjectLinker<T[N]> {
            static AffinityObject& Link(T object[N]) {
                for (UInt32 i = 1; i < N; ++i) {
                    GetAffinityObject(object[i]).LinkTo(GetAffinityObject(object[0]));
                }
                return GetAffinityObject(object[0]);
            }
        };

        /// @{
        /** LinkAffinityObjects links the given affinity object with the affinity objects in the given objects a1 ... a5 */
        template<typename A0, typename A1> AffinityObject& LinkAffinityObjects(A0 &a0, A1 &a1)
        {
            return AffinityObjectLinker<A1>::Link(a1).LinkTo(AffinityObjectLinker<A0>::Link(a0));
        }

        template<typename A0, typename A1, typename A2>
        AffinityObject& LinkAffinityObjects(A0 &a0, A1 &a1, A2 &a2)
        {
            return LinkAffinityObjects(LinkAffinityObjects(a0, a1), a2);
        }

        template<typename A0, typename A1, typename A2, typename A3>
        AffinityObject& LinkAffinityObjects(A0 &a0, A1 &a1, A2 &a2, A3 &a3)
        {
            return LinkAffinityObjects(LinkAffinityObjects(a0, a1, a2), a3);
        }

        template<typename A0, typename A1, typename A2, typename A3, typename A4>
        AffinityObject& LinkAffinityObjects(A0 &a0, A1 &a1, A2 &a2, A3 &a3, A4 &a4)
        {
            return LinkAffinityObjects(LinkAffinityObjects(a0, a1, a2, a3), a4);
        }

        template<typename A0, typename A1, typename A2, typename A3, typename A4, typename A5>
        AffinityObject& LinkAffinityObjects(A0 &a0, A1 &a1, A2 &a2, A3 &a3, A4 &a4, A5 &a5)
        {
            return LinkAffinityObjects(LinkAffinityObjects(a0, a1, a2, a3, a4), a5);
        }
        /// @}

        // ------------------------------------------------------------------------

        /// @{
        /** Tags all given unsynchronized objects to be externally synced.
            The given objects will be automatically linked too. */
        template<typename A1> void SetAffinityObjectExternallySynced(A1 &a1)
        {
            AffinityObjectLinker<A1>::Link(a1).SetIsExternallySynced(true);
        }

        template<typename A1, typename A2>
        void SetAffinityObjectExternallySynced(A1 &a1, A2 &a2)
        {
            (void) LinkAffinityObjects(a1, a2).SetIsExternallySynced(true);
        }

        template<typename A1, typename A2, typename A3>
        void SetAffinityObjectExternallySynced(A1 &a1, A2 &a2, A3 &a3)
        {
            (void) LinkAffinityObjects(a1, a2, a3).SetIsExternallySynced(true);
        }

        template<typename A1, typename A2, typename A3, typename A4>
        void SetAffinityObjectExternallySynced(A1 &a1, A2 &a2, A3 &a3, A4 &a4)
        {
            (void) LinkAffinityObjects(a1, a2, a3, a4).SetIsExternallySynced(true);
        }

        template<typename A1, typename A2, typename A3, typename A4, typename A5>
        void SetAffinityObjectExternallySynced(A1 &a1, A2 &a2, A3 &a3, A4 &a4, A5 &a5)
        {
            (void) LinkAffinityObjects(a1, a2, a3, a4, a5).SetIsExternallySynced(true);
        }

        /// @}

        // ------------------------------------------------------------------------

        /** Helper class to check current thread id against the specified thread context types
            @tparam T0 thread context type 0
            @tparam T1 thread context type 1
            @tparam T2 thread context type 2
            @tparam T3 thread context type 3
            @tparam T4 thread context type 4 */
        template<typename T0,
                 typename T1 = AnyThreadContext,
                 typename T2 = AnyThreadContext,
                 typename T3 = AnyThreadContext,
                 typename T4 = AnyThreadContext
                >
        struct ThreadContextVerifier {
            typedef T0 Context0;
            typedef T1 Context1;
            typedef T2 Context2;
            typedef T3 Context3;
            typedef T4 Context4;

            /** Checks if the current thread id is affine to the specified thread context types */
            static bool Check() {
                UInt32 currentThreadId = Platform::Thread::GetCurrentId();
                return Context0::Check(currentThreadId) ||
                       Context1::Check(currentThreadId) ||
                       Context2::Check(currentThreadId) ||
                       Context3::Check(currentThreadId) ||
                       Context4::Check(currentThreadId);
            }
        };

        // ------------------------------------------------------------------------

        /** Helper function to check an objects thread affinity
            Used by COURIER_CHECK_OBJECT_AFFINITY
            @param affinityObject the affinityObject of the object
            @param isConst true if the object is const (call invoked in context of a const method)
            @return true if affinity is correct, false otherwise */
        extern bool CheckObjectThreadAffinity(const AffinityObject &affinityObject, bool isConst);

        /// @{
        /** template function to determine if the given object is declared const
            used for shared read only objects. Non-const member functions of unsynced objects
            with shared read only state must not be invoked. */
        template<typename T> inline bool IsConstObject(T *) {
            return false;
        }

        template<typename T> inline bool IsConstObject(const T *) {
            return true;
        }
        /// @}

        /** Helper class to store the thread id for each Courier component */
        class ComponentThreadContextPool {
            public:
                /** returns the thread id for the given component
                    @param componentId the id of the component (one of Courier::ComponentType::Enum)
                    @return the components thread id */
                static UInt32 GetThreadId(ComponentId componentId) {
                    return mComponentThreadId[componentId ];
                }

                /** set the thread id for the given component to the given thread id
                    @param componentId the id of the component (one of Courier::ComponentType::Enum)
                    @param threadId the id of the thread affine to the component */
                static void SetThreadId(ComponentId componentId, UInt32 threadId);

                /** returns if the thread id for the given component has been set yet
                    @param componentId the id of the component (one of Courier::ComponentType::Enum)
                    @return true if thread id for the component has been set, false otherwise. */
                static bool IsThreadIdSet(ComponentId componentId) {
                    return mComponentThreadIdSet[componentId];
                }

                /** check the thread id defined for the given component against the given thread id
                    @param componentId the id of the component (one of Courier::ComponentType::Enum)
                    @param threadId to check against
                    @return true if component threadId matches the given thread id, false otherwise */
                static bool Check(ComponentId componentId , UInt32 threadId) {
                    return (!mComponentThreadIdSet[componentId]) || (mComponentThreadId[componentId] == threadId);
                }

            private:
                static bool mComponentThreadIdSet[COURIER_COMPONENT_MAX_COUNT];
                static UInt32 mComponentThreadId[COURIER_COMPONENT_MAX_COUNT];
        };

    /// @}
    }}

    namespace Courier {
    /// @addtogroup COURIER_DIAGNOSTICS
    /// @{
        /** Set the thread id of the given component to the specified thread id
            @param componentId one of Courier::componentId ::Enum
            @param threadId the thread id of the context */
        #define COURIER_SET_COMPONENT_THREAD_ID(componentId, threadId) \
            Courier::Internal::ComponentThreadContextPool::SetThreadId(componentId, threadId)

        /** Set the thread id of the given component to the current thread id
            @param componentId one of Courier::ComponentId */
        #define COURIER_SET_COMPONENT_THREAD_ID_TO_CURRENT(componentId) \
            COURIER_SET_COMPONENT_THREAD_ID(componentId, Courier::Platform::Thread::GetCurrentId())

        // ------------------------------------------------------------------------

        /** thread context class for a specific component
            @tparam T the component type (one of Courier::componentId ::Enum) */
        template<ComponentId componentId > class ComponentThreadContext {
            public:
                /** returns the context thread id
                    @return thread id */
                static inline UInt32 GetThreadId() {
                    return Internal::ComponentThreadContextPool::GetThreadId(componentId );
                }

                /** set the context thread id
                    @param value the id of the thread associated with this context */
                static inline void SetThreadId(UInt32 value) {
                    Internal::ComponentThreadContextPool::SetThreadId(componentId , value);
                }

                /** Checks if the given thread id matches the context thread id
                    @param threadId the thread id to check
                    @return true if the thread id matches, false otherwise */
                static inline bool Check(UInt32 threadId) {
                    return Internal::ComponentThreadContextPool::Check(componentId , threadId);
                }
        };

        // the context Courier::Init will be invoked with
        COURIER_DEFINE_THREAD_CONTEXT(MainContext);

    /// @}
    }

#else
    /// @addtogroup COURIER_DIAGNOSTICS
    /// @{
    #define COURIER_UNSYNCHRONIZED_TYPE(typeName) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_CHECK_OBJECT_AFFINITY() COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_TRANSFER_OBJECT_AFFINITY(unsyncedObject, contextName) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_TRANSFER_OBJECT_AFFINITY_TO_CURRENT(unsyncedObject) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_DEFINE_SCOPE_AFFINITY(...) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_EXPECT_CONTEXT(...) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_CHECK_SCOPE_AFFINITY() COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_DEFINE_THREAD_CONTEXT(contextName)
    #define COURIER_SET_CONTEXT_THREAD_ID(contextName, threadId) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_SET_CONTEXT_THREAD_ID_TO_CURRENT(contextName) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_SET_COMPONENT_CONTEXT_THREAD_ID(componentId , threadId) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_SET_COMPONENT_CONTEXT_THREAD_ID_TO_CURRENT(componentId ) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_AGGREGATE_OBJECT_AFFINITY(...) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_REMOVE_FROM_AFFINITY_AGGREGATION(unsyncedObject) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_DECLARE_EXTERNALLY_SYNCED(...) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_DECLARE_SHARED_READONLY(unsyncedObject, sharedReadOnly) COURIER_EXPAND_TO_EMPTY_MACRO()

    #define COURIER_SET_COMPONENT_THREAD_ID(componentId, threadId) COURIER_EXPAND_TO_EMPTY_MACRO()
    #define COURIER_SET_COMPONENT_THREAD_ID_TO_CURRENT(componentId, threadId) COURIER_EXPAND_TO_EMPTY_MACRO()
    /// @}
#endif  // COURIER_ENABLE_CONTEXT_AFFINITY_CHECKS
#endif
