//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_StdTypeConverter_h)
#define Courier_DataBinding_StdTypeConverter_h

#include <Courier/Util/Limits.h>
#include <Courier/Util/Traits.h>

namespace Courier {

    namespace Internal {
        /// @addtogroup COURIER_DATABINDING
        /// @{
        namespace DataBinding {
            /** generic type conversion functions for unsigned integral types
            @param LargeType unsigned type with sizeof(LargeType) > sizeof(SmallType)
            @param SmallType unsigned type with sizeof(LargeType) > sizeof(SmallType) */
            template<typename LargeType, typename SmallType> struct UnsignedIntegralTypeConverter {
                /** Conversion from SmallType to LargeType
                @param lt receives value
                @param st gives value
                @return true if conversion succeeded, otherwise false */
                //! [COURIER_COMPILETIME_ASSERT]
                static inline bool SmallToLarge(LargeType &lt, SmallType st) {
                    FEATSTD_COMPILETIME_ASSERT(sizeof(LargeType) >= sizeof(SmallType));
                    lt = st;
                    return true;
                }
                //! [COURIER_COMPILETIME_ASSERT]

                /** Conversion from LargeType to SmallType
                @param st receives value
                @param lt gives value
                @return true if conversion succeeded, otherwise false */
                static inline bool LargeToSmall(SmallType &st, LargeType lt) {
                    FEATSTD_COMPILETIME_ASSERT(sizeof(LargeType) >= sizeof(SmallType));
                    if (lt > LargeType(Internal::Limits<SmallType>::Max())) {
                        return false;
                    }
                    st = SmallType(lt);
                    return true;
                }
            };

            /** generic type conversion functions for signed integral types
            @param LargeType unsigned type with sizeof(LargeType) > sizeof(SmallType)
            @param SmallType unsigned type with sizeof(LargeType) > sizeof(SmallType) */
            template<typename LargeType, typename SmallType> struct SignedIntegralTypeConverter {
                /** Conversion from SmallType to LargeType
                @param lt receives value
                @param st gives value
                @return true if conversion succeeded, otherwise false */
                static inline bool SmallToLarge(LargeType &lt, SmallType st) {
                    FEATSTD_COMPILETIME_ASSERT(sizeof(LargeType) >= sizeof(SmallType));
                    lt = st;
                    return true;
                }

                /** Conversion from LargeType to SmallType
                @param st receives value
                @param lt gives value
                @return true if conversion succeeded, otherwise false */
                static inline bool LargeToSmall(SmallType &st, LargeType lt) {
                    FEATSTD_COMPILETIME_ASSERT(sizeof(LargeType) >= sizeof(SmallType));
                    if ((lt < Internal::Limits<SmallType>::Min()) || (lt > Internal::Limits<SmallType>::Max())) {
                        return false;
                    }
                    st = SmallType(lt);
                    return true;
                }
            };

            /** generic type conversion functions for signed integral types
            @param LargeType unsigned type with sizeof(LargeType) > sizeof(SmallType)
            @param SmallType unsigned type with sizeof(LargeType) > sizeof(SmallType) */
            template<typename Signed, typename Unsigned> struct MixedSignIntegralTypeConverter {
                /** Conversion from unsigned to signed
                @param s receives value
                @param us gives value
                @return true if conversion succeeded, otherwise false */
                static inline bool UnsignedToSigned(Signed &s, Unsigned us) {
                    typedef typename Internal::MakeUnsigned<Signed>::Type UnsignedOfSigned;
                    if (us > UnsignedOfSigned(Internal::Limits<Signed>::Max())) {
                        return false;
                    }
                    s = Signed(UnsignedOfSigned(us));
                    return true;
                }

                /** Conversion from signed to unsigned
                @param us receives value
                @param s gives value
                @return true if conversion succeeded, otherwise false */
                static inline bool SignedToUnsigned(Unsigned &us, Signed s) {
                    typedef typename Internal::MakeUnsigned<Signed>::Type UnsignedOfSigned;
                    typedef typename Internal::MakeSigned<Unsigned>::Type SignedOfUnsigned;
                    if ((s < 0) || (UnsignedOfSigned(s) > Internal::Limits<Unsigned>::Max())) {
                        return false;
                    }
                    us = Unsigned(SignedOfUnsigned(s));
                    return true;
                }
            };
        }
    }

    /** registers standard built-in type converters
    type converters for (X and Y one of 8, 16, and 32
    - UIntX to UIntY types
    - IntX to UIntY types */
    void RegisterStdTypeConverters();

    //@}
}   // namespace
#endif // Courier_DataBinding_StdTypeConverter_h
