//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StdTypeConverter.h"

#include <Courier/Diagnostics/Log.h>
#include <FeatStd/Util/Variant.h>
#include <FeatStd/Util/String.h>
#include <FeatStd/Util/VariantTypeConverter.h>
#include <Courier/DataBinding/DataItemType.h>

//COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

namespace Courier {

#define COURIER_DECL_FLOAT_TYPECONV(T, FloatType) \
    static bool TypeConvert(FloatType &dst, T src) \
    { \
        static_cast<void>(FeatStd::Internal::VariantTypeConverterSelector<T, FloatType>::StaticCast::TryConvert(src, dst)); \
        return true; \
    } \
    static bool TypeConvert(T &dst, FloatType src) \
    { \
        static_cast<void>(FeatStd::Internal::VariantTypeConverterSelector<FloatType, T>::StaticCast::TryConvert(src, dst)); \
        return true; \
    }

#define COURIER_DECL_FLOAT_BOOL_TYPECONV(FloatType, BoolType, Threashold, TrueValue, FalseValue) \
    static bool TypeConvert(FloatType &dst, BoolType src) \
    { \
        dst = src ? TrueValue : FalseValue; \
        return true; \
    } \
    static bool TypeConvert(BoolType &dst, FloatType src) \
    { \
        dst = (src >= Threashold); \
        return true; \
    }

#define COURIER_DECL_BOOL_TYPECONV(T) \
    static bool TypeConvert(bool &dst, T src) \
    { \
        dst = src != 0; \
        return true; \
    } \
    static bool TypeConvert(T &dst, bool src) \
    { \
        dst = T(src); \
        return true; \
    }

    template<typename T>
    struct TypeConvertStringHelper {
        static bool Convert(T&, const FeatStd::String&)
        {
            return false;
        }
    };

#define COURIER_DECL_STRING_TYPECONV2(T, TypeName) \
    template<> \
    struct TypeConvertStringHelper<T> \
    { \
        static bool Convert(T &dst, const FeatStd::String& src) \
        { \
            dst = FeatStd::Variant(src).FEATSTD_CONCAT2(Get, TypeName)(); \
            return true; \
        } \
    }; \
    static bool TypeConvert(FeatStd::String &dst, T src) \
    { \
        dst = FeatStd::Variant(src).GetString(); \
        return true; \
    } \
    static bool TypeConvert(T &dst, const FeatStd::String& src) \
    { \
        return TypeConvertStringHelper<T>::Convert(dst, src); \
    }

#define COURIER_DECL_STRING_TYPECONV(T) COURIER_DECL_STRING_TYPECONV2(T, T)

#define COURIER_DEF_Variant_TYPECONV(T) \
    static inline bool TypeConvert(T &dst, const FeatStd::Variant& src); \
    static inline bool TypeConvert(FeatStd::Variant &dst, T src);

#define COURIER_DECL_Variant_TYPECONV(T) \
    static inline bool TypeConvert(T &dst, const FeatStd::Variant& src) \
    { \
        return TypeConvertFromVariant<T>(dst, src); \
    } \
    static inline bool TypeConvert(FeatStd::Variant &dst, T src) \
    { \
        return TypeConvertToVariant<T>(dst, src); \
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, UInt32)
    static inline bool TypeConvert(UInt64 &dst, UInt32 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt32>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt32 &dst, UInt64 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt32>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, UInt16)
    static inline bool TypeConvert(UInt64 &dst, UInt16 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt16>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt16 &dst, UInt64 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt16>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, UInt8)
    static inline bool TypeConvert(UInt64 &dst, UInt8 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt8 &dst, UInt64 src)
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt64, UInt8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Int64)
    static inline bool TypeConvert(UInt64 &dst, Int64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt64>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int64 &dst, UInt64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt64>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Int32)
    static inline bool TypeConvert(UInt64 &dst, Int32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt64>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int32 &dst, UInt64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt64>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Int16)
    static inline bool TypeConvert(UInt64 &dst, Int16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt64>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int16 &dst, UInt64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt64>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Int8)
    static inline bool TypeConvert(UInt64 &dst, Int8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int8, UInt64>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int8 &dst, UInt64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int8, UInt64>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Float)
    COURIER_DECL_FLOAT_TYPECONV(UInt64, Float)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, Double)
    COURIER_DECL_FLOAT_TYPECONV(UInt64, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt64, bool)
    COURIER_DECL_BOOL_TYPECONV(UInt64)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Int64)
    static inline bool TypeConvert(UInt32 &dst, Int64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt32>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int64 &dst, UInt32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt32>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, UInt16)
    static inline bool TypeConvert(UInt32 &dst, UInt16 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt32, UInt16>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt16 &dst, UInt32 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt32, UInt16>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, UInt8)
    static inline bool TypeConvert(UInt32 &dst, UInt8 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt32, UInt8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt8 &dst, UInt32 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt32, UInt8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Int32)
    static inline bool TypeConvert(UInt32 &dst, Int32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int32 &dst, UInt32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Int16)
    static inline bool TypeConvert(UInt32 &dst, Int16 src)
    {
        Int32 v = 0;
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int16>::SmallToLarge(v, src) &&
            Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::SignedToUnsigned(dst, v);
    }

    static inline bool TypeConvert(Int16 &dst, UInt32 src)
    {
        Int32 v = 0;
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::UnsignedToSigned(v, src) &&
            Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int16>::LargeToSmall(dst, v);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Int8)
    static inline bool TypeConvert(UInt32 &dst, Int8 src)
    {
        Int32 v = 0;
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int8>::SmallToLarge(v, src) &&
            Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::SignedToUnsigned(dst, v);
    }

    static inline bool TypeConvert(Int8 &dst, UInt32 src)
    {
        Int32 v = 0;
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt32>::UnsignedToSigned(v, src) &&
            Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int8>::LargeToSmall(dst, v);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Float)
    COURIER_DECL_FLOAT_TYPECONV(UInt32, Float)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, Double)
    COURIER_DECL_FLOAT_TYPECONV(UInt32, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt32, bool)
    COURIER_DECL_BOOL_TYPECONV(UInt32)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Int64)
    static inline bool TypeConvert(UInt16 &dst, Int64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt16>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int64 &dst, UInt16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt16>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, UInt8)
    static inline bool TypeConvert(UInt16 &dst, UInt8 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt16, UInt8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(UInt8 &dst, UInt16 src) 
    {
        return Internal::DataBinding::UnsignedIntegralTypeConverter<UInt16, UInt8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Int32)
    static inline bool TypeConvert(UInt16 &dst, Int32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt16>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int32 &dst, UInt16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt16>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Int16)
    static inline bool TypeConvert(UInt16 &dst, Int16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt16>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int16 &dst, UInt16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt16>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Int8)
    static inline bool TypeConvert(UInt16 &dst, Int8 src)
    {
        Int16 v = 0;
        return Internal::DataBinding::SignedIntegralTypeConverter<Int16, Int8>::SmallToLarge(v, src) &&
            Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt16>::SignedToUnsigned(dst, v);
    }

    static inline bool TypeConvert(Int8 &dst, UInt16 src)
    {
        Int16 v = 0;
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt16>::UnsignedToSigned(v, src) &&
            Internal::DataBinding::SignedIntegralTypeConverter<Int16, Int8>::LargeToSmall(dst, v);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Float)
    COURIER_DECL_FLOAT_TYPECONV(UInt16, Float)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Double)
    COURIER_DECL_FLOAT_TYPECONV(UInt16, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, bool)
    COURIER_DECL_BOOL_TYPECONV(UInt16)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt16, Int64)
    static inline bool TypeConvert(UInt8 &dst, Int64 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt8>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int64 &dst, UInt8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int64, UInt8>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, Int32)
    static inline bool TypeConvert(UInt8 &dst, Int32 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt8>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int32 &dst, UInt8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int32, UInt8>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, Int16)
    static inline bool TypeConvert(UInt8 &dst, Int16 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt8>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int16 &dst, UInt8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int16, UInt8>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, Int8)
    static inline bool TypeConvert(UInt8 &dst, Int8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int8, UInt8>::SignedToUnsigned(dst, src);
    }

    static inline bool TypeConvert(Int8 &dst, UInt8 src)
    {
        return Internal::DataBinding::MixedSignIntegralTypeConverter<Int8, UInt8>::UnsignedToSigned(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, Float)
    COURIER_DECL_FLOAT_TYPECONV(UInt8, Float)

        // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, Double)
    COURIER_DECL_FLOAT_TYPECONV(UInt8, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(UInt8, bool)
    COURIER_DECL_BOOL_TYPECONV(UInt8)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, Int32)
    static inline bool TypeConvert(Int64 &dst, Int32 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int32>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int32 &dst, Int64 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int32>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, Int16)
    static inline bool TypeConvert(Int64 &dst, Int16 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int16>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int16 &dst, Int64 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int16>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, Int8)
    static inline bool TypeConvert(Int64 &dst, Int8 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int8 &dst, Int64 src)
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int64, Int8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, Float)
    COURIER_DECL_FLOAT_TYPECONV(Int64, Float)

        // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, Double)
    COURIER_DECL_FLOAT_TYPECONV(Int64, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int64, bool)
    COURIER_DECL_BOOL_TYPECONV(Int64)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int32, Int16)
    static inline bool TypeConvert(Int32 &dst, Int16 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int16>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int16 &dst, Int32 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int16>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int32, Int8)
    static inline bool TypeConvert(Int32 &dst, Int8 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int8 &dst, Int32 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int32, Int8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int32, Float)
    COURIER_DECL_FLOAT_TYPECONV(Int32, Float)

        // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int32, Double)
    COURIER_DECL_FLOAT_TYPECONV(Int32, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int32, bool)
    COURIER_DECL_BOOL_TYPECONV(Int32)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int16, Int8)
    static inline bool TypeConvert(Int16 &dst, Int8 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int16, Int8>::SmallToLarge(dst, src);
    }

    static inline bool TypeConvert(Int8 &dst, Int16 src) 
    {
        return Internal::DataBinding::SignedIntegralTypeConverter<Int16, Int8>::LargeToSmall(dst, src);
    }

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int16, Float)
    COURIER_DECL_FLOAT_TYPECONV(Int16, Float)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int16, Double)
    COURIER_DECL_FLOAT_TYPECONV(Int16, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int16, bool)
    COURIER_DECL_BOOL_TYPECONV(Int16)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int8, Float)
    COURIER_DECL_FLOAT_TYPECONV(Int8, Float)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int8, Double)
    COURIER_DECL_FLOAT_TYPECONV(Int8, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Int8, bool)
    COURIER_DECL_BOOL_TYPECONV(Int8)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Float, Double)
    COURIER_DECL_FLOAT_TYPECONV(Float, Double)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Float, bool)
    COURIER_DECL_FLOAT_BOOL_TYPECONV(Float, bool, 0.5F, 1.0F, 0.0F)

    // ------------------------------------------------------------------------
    /// Type conversion functions for DefConverter(Double, bool)
    COURIER_DECL_FLOAT_BOOL_TYPECONV(Double, bool, 0.5, 1.0, 0.0)

    COURIER_DECL_STRING_TYPECONV(UInt64)
    COURIER_DECL_STRING_TYPECONV(UInt32)
    COURIER_DECL_STRING_TYPECONV(UInt16)
    COURIER_DECL_STRING_TYPECONV(UInt8)
    COURIER_DECL_STRING_TYPECONV(Int64)
    COURIER_DECL_STRING_TYPECONV(Int32)
    COURIER_DECL_STRING_TYPECONV(Int16)
    COURIER_DECL_STRING_TYPECONV(Int8)
    COURIER_DECL_STRING_TYPECONV2(bool, Bool)
    COURIER_DECL_STRING_TYPECONV(Float)
    COURIER_DECL_STRING_TYPECONV(Double)

    static bool TypeConvert(FeatStd::String &dst, const FeatStd::Variant& src)
    {
        dst = src.GetString();
        return true;
    }

    static bool TypeConvert(FeatStd::Variant& dst, const FeatStd::String& src)
    {
        static_cast<void>(dst.SetValue(FeatStd::Variant(src)));
        return true;
    }

    COURIER_DEF_Variant_TYPECONV(UInt64)
    COURIER_DEF_Variant_TYPECONV(UInt32)
    COURIER_DEF_Variant_TYPECONV(UInt16)
    COURIER_DEF_Variant_TYPECONV(UInt8)
    COURIER_DEF_Variant_TYPECONV(Int64)
    COURIER_DEF_Variant_TYPECONV(Int32)
    COURIER_DEF_Variant_TYPECONV(Int16)
    COURIER_DEF_Variant_TYPECONV(Int8)
    COURIER_DEF_Variant_TYPECONV(bool)
    COURIER_DEF_Variant_TYPECONV(Float)
    COURIER_DEF_Variant_TYPECONV(Double)

    template <typename T1, typename T2, FeatStd::UInt isSameType>
    struct TTypeConvertHelper;

    template <typename T1, typename T2>
    struct TTypeConvertHelper<T1, T2, true> {
        static bool Convert(T1& v1, const T2& v2) {
            v1 = v2;
            return true;
        }
    };

    template <typename T1, typename T2>
    struct TTypeConvertHelper<T1, T2, false> {
        static bool Convert(T1& v1, const T2& v2) {
            return ::Courier::TypeConvert(v1, v2);
        }
    };
    namespace {
        template <typename T1, typename T2>
        bool TTypeConvert(T1& v1, const T2& v2)
        {
            return TTypeConvertHelper< T1, T2, Courier::Internal::IsSameType< T1, T2 >::Value >::Convert(v1, v2);
        } 
    }
    template <typename T>
    bool TypeConvertFromVariant(T& dst, const FeatStd::Variant& src)
    {
        bool result = false;
        switch (src.GetType()) {
        case FeatStd::VariantValueType::FloatValue:
            result = TTypeConvert(dst, src.GetFloat());
            break;
        case FeatStd::VariantValueType::DoubleValue:
            result = TTypeConvert(dst, src.GetDouble());
            break;
        case FeatStd::VariantValueType::Int8Value:
            result = TTypeConvert(dst, src.GetInt8());
            break;
        case FeatStd::VariantValueType::Int16Value:
            result = TTypeConvert(dst, src.GetInt16());
            break;
        case FeatStd::VariantValueType::Int32Value:
            result = TTypeConvert(dst, src.GetInt32());
            break;
        case FeatStd::VariantValueType::Int64Value:
            result = TTypeConvert(dst, src.GetInt64());
            break;
        case FeatStd::VariantValueType::UInt8Value:
            result = TTypeConvert(dst, src.GetUInt8());
            break;
        case FeatStd::VariantValueType::UInt16Value:
            result = TTypeConvert(dst, src.GetUInt16());
            break;
        case FeatStd::VariantValueType::UInt32Value:
            result = TTypeConvert(dst, src.GetUInt32());
            break;
        case FeatStd::VariantValueType::UInt64Value:
            result = TTypeConvert(dst, src.GetUInt64());
            break;
        case FeatStd::VariantValueType::BoolValue:
            result = TTypeConvert(dst, src.GetFloat());
            break;
        case FeatStd::VariantValueType::StringValue:
            result = TTypeConvert(dst, src.GetString());
            break;
        default:
            break;
        }
        return result;
    }

    template <typename T>
    bool TypeConvertToVariant(FeatStd::Variant& dst, const T& src)
    {
        bool result = false;
        switch (dst.GetType()) {
        case FeatStd::VariantValueType::FloatValue: {
            FeatStd::Float value = 0.0F;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::DoubleValue: {
            FeatStd::Double value = 0.0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::Int8Value: {
            FeatStd::Int8 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::Int16Value: {
            FeatStd::Int16 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::Int32Value: {
            FeatStd::Int32 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::Int64Value: {
            FeatStd::Int64 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::UInt8Value: {
            FeatStd::UInt8 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::UInt16Value: {
            FeatStd::UInt16 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::UInt32Value: {
            FeatStd::UInt32 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::UInt64Value: {
            FeatStd::UInt64 value = 0;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::BoolValue: {
            bool value = false;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        case FeatStd::VariantValueType::StringValue: {
            FeatStd::String value;
            if (TTypeConvert(value, src)) {
                dst = FeatStd::Variant(value);
                result = true;
            }
            else {
                result = false;
            }
            break;
        }
        default:
            break;
        }
        return result;
    }

    COURIER_DECL_Variant_TYPECONV(UInt64)
    COURIER_DECL_Variant_TYPECONV(UInt32)
    COURIER_DECL_Variant_TYPECONV(UInt16)
    COURIER_DECL_Variant_TYPECONV(UInt8)
    COURIER_DECL_Variant_TYPECONV(Int64)
    COURIER_DECL_Variant_TYPECONV(Int32)
    COURIER_DECL_Variant_TYPECONV(Int16)
    COURIER_DECL_Variant_TYPECONV(Int8)
    COURIER_DECL_Variant_TYPECONV(bool)
    COURIER_DECL_Variant_TYPECONV(Float)
    COURIER_DECL_Variant_TYPECONV(Double)

    FEATSTD_LINT_NEXT_EXPRESSION(1960, "the macro COURIER_DECL_BOOL_TYPECONV is a local macro")
#undef COURIER_DECL_BOOL_TYPECONV
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "the macro COURIER_DECL_FLOAT_TYPECONV is a local macro")
#undef COURIER_DECL_FLOAT_TYPECONV
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "the macro COURIER_DECL_FLOAT_BOOL_TYPECONV is a local macro")
#undef COURIER_DECL_FLOAT_BOOL_TYPECONV
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "the macro COURIER_DECL_Variant_TYPECONV is a local macro")
#undef COURIER_DECL_Variant_TYPECONV

} // namespace

// Due to the fact that TypeConvert function is further referenced in the below header it must be defined first as above to avoid compile warnings in gcc
// ToDo: split StdTypeConverter in two files: one with the above definitions for TypeConvert and one with the RegisterStdTypeConverters below
#include <Courier/DataBinding/TypeConverter.h>

namespace Courier {
    #define DefConverter(l, r) \
        static TypeConverter< l , r > FEATSTD_CONCAT3(l, To, r); \
        FEATSTD_UNUSED(FEATSTD_CONCAT3(l, To, r))

    // ------------------------------------------------------------------------
    void RegisterStdTypeConverters()
    {
        // force usage of TypeConvert(UInt64,UIn64)
        UInt64 v1 = 0;
        UInt64 v2 = 0;
        static_cast<void>(TTypeConvert(v1, v2));
        static bool enlisted = false;
        if (enlisted) {
            return;
        }
        enlisted = true;

        using FeatStd::Variant;
        using FeatStd::String;
        // UInt64
        // DefConverter(UInt64, UInt64); = > direct convert
        DefConverter(UInt64, UInt32);
        DefConverter(UInt64, UInt16);
        DefConverter(UInt64, UInt8);
        DefConverter(UInt64, Int64);
        DefConverter(UInt64, Int32);
        DefConverter(UInt64, Int16);
        DefConverter(UInt64, Int8);
        DefConverter(UInt64, Float);
        DefConverter(UInt64, Double);
        DefConverter(UInt64, bool);
        DefConverter(UInt64, Variant);
        DefConverter(UInt64, String);

        // UInt32
        // DefConverter(UInt32, UInt64); => DefConverter(UInt64, UInt32)
        // DefConverter(UInt32, UInt32); => direct convert
        DefConverter(UInt32, UInt16);
        DefConverter(UInt32, UInt8);
        DefConverter(UInt32, Int64);
        DefConverter(UInt32, Int32);
        DefConverter(UInt32, Int16);
        DefConverter(UInt32, Int8);
        DefConverter(UInt32, Float);
        DefConverter(UInt32, Double);
        DefConverter(UInt32, bool);
        DefConverter(UInt32, Variant);
        DefConverter(UInt32, String);

        // UInt16
        // DefConverter(UInt16, UInt64); => DefConverter(UInt64, UInt16)
        // DefConverter(UInt16, UInt32); => DefConverter(UInt32, UInt16)
        // DefConverter(UInt16, UInt16); => direct convert
        DefConverter(UInt16, UInt8);
        DefConverter(UInt16, Int64);
        DefConverter(UInt16, Int32);
        DefConverter(UInt16, Int16);
        DefConverter(UInt16, Int8);
        DefConverter(UInt16, Float);
        DefConverter(UInt16, Double);
        DefConverter(UInt16, bool);
        DefConverter(UInt16, Variant);
        DefConverter(UInt16, String);

        // UInt8
        // DefConverter(UInt8, UInt64); => DefConverter(UInt64, UInt8)
        // DefConverter(UInt8, UInt32); => DefConverter(UInt32, UInt8)
        // DefConverter(UInt8, UInt16); => DefConverter(UInt16, UInt8)
        // DefConverter(UInt8, UInt8);  => direct convert
        DefConverter(UInt8, Int64);
        DefConverter(UInt8, Int32);
        DefConverter(UInt8, Int16);
        DefConverter(UInt8, Int8);
        DefConverter(UInt8, Float);
        DefConverter(UInt8, Double);
        DefConverter(UInt8, bool);
        DefConverter(UInt8, Variant);
        DefConverter(UInt8, String);

        // Int64
        // DefConverter(Int64, UInt64); => DefConverter(UInt64, Int64)
        // DefConverter(Int64, UInt32); => DefConverter(UInt32, Int64)
        // DefConverter(Int64, UInt16); => DefConverter(UInt16, Int64)
        // DefConverter(Int64, UInt8);  => DefConverter(UInt8,  Int64)
        // DefConverter(Int64, Int64);  => direct convert
        DefConverter(Int64, Int32);
        DefConverter(Int64, Int16);
        DefConverter(Int64, Int8);
        DefConverter(Int64, Float);
        DefConverter(Int64, Double);
        DefConverter(Int64, bool);
        DefConverter(Int64, Variant);
        DefConverter(Int64, String);

        // Int32
        // DefConverter(Int32, UInt64); => DefConverter(UInt64, Int32)
        // DefConverter(Int32, UInt32); => DefConverter(UInt32, Int32)
        // DefConverter(Int32, UInt16); => DefConverter(UInt16, Int32)
        // DefConverter(Int32, UInt8);  => DefConverter(UInt8,  Int32)
        // DefConverter(Int32, Int64);  => DefConverter(Int64,  Int32)
        // DefConverter(Int32, Int32);  => direct convert
        DefConverter(Int32, Int16);
        DefConverter(Int32, Int8);
        DefConverter(Int32, Float);
        DefConverter(Int32, Double);
        DefConverter(Int32, bool);
        DefConverter(Int32, Variant);
        DefConverter(Int32, String);

        // Int16
        // DefConverter(Int16, UInt64); => DefConverter(UInt64, Int16)
        // DefConverter(Int16, UInt32); => DefConverter(UInt32, Int16)
        // DefConverter(Int16, UInt16); => DefConverter(UInt16, Int16)
        // DefConverter(Int16, UInt8);  => DefConverter(UInt8,  Int16)
        // DefConverter(Int16, Int64);  => DefConverter(Int64,  Int16)
        // DefConverter(Int16, Int32);  => DefConverter(Int32,  Int16)
        // DefConverter(Int16, Int16);  => direct convert
        DefConverter(Int16, Int8);
        DefConverter(Int16, Float);
        DefConverter(Int16, Double);
        DefConverter(Int16, bool);
        DefConverter(Int16, Variant);
        DefConverter(Int16, String);

        // Int8
        // DefConverter(Int8, UInt64); => DefConverter(UInt64, Int8)
        // DefConverter(Int8, UInt32); => DefConverter(UInt32, Int8)
        // DefConverter(Int8, UInt16); => DefConverter(UInt16, Int8)
        // DefConverter(Int8, UInt8);  => DefConverter(UInt8,  Int8)
        // DefConverter(Int8, Int64);  => DefConverter(Int64,  Int8)
        // DefConverter(Int8, Int32);  => DefConverter(Int32,  Int8)
        // DefConverter(Int8, Int16);  => DefConverter(Int16,  Int8)
        // DefConverter(Int8, Int8);   => direct convert
        DefConverter(Int8, Float);
        DefConverter(Int8, Double);
        DefConverter(Int8, bool);
        DefConverter(Int8, Variant);
        DefConverter(Int8, String);

        // Float
        // DefConverter(Float, UInt64); => DefConverter(UInt64, Float)
        // DefConverter(Float, UInt32); => DefConverter(UInt32, Float)
        // DefConverter(Float, UInt16); => DefConverter(UInt16, Float)
        // DefConverter(Float, UInt8);  => DefConverter(UInt8,  Float)
        // DefConverter(Float, Int64);  => DefConverter(Int64,  Float)
        // DefConverter(Float, Int32);  => DefConverter(Int32,  Float)
        // DefConverter(Float, Int16);  => DefConverter(Int16,  Float)
        // DefConverter(Float, Int8);   => DefConverter(Int8,   Float)
        // DefConverter(Float, Float);  => direct convert
        DefConverter(Float, Double);
        DefConverter(Float, bool);
        DefConverter(Float, Variant);
        DefConverter(Float, String);

        // Double
        // DefConverter(Double, UInt64); => DefConverter(UInt64, Double)
        // DefConverter(Double, UInt32); => DefConverter(UInt32, Double)
        // DefConverter(Double, UInt16); => DefConverter(UInt16, Double)
        // DefConverter(Double, UInt8);  => DefConverter(UInt8,  Double)
        // DefConverter(Double, Int64);  => DefConverter(Int64,  Double)
        // DefConverter(Double, Int32);  => DefConverter(Int32,  Double)
        // DefConverter(Double, Int16);  => DefConverter(Int16,  Double)
        // DefConverter(Double, Int8);   => DefConverter(Int8,   Double)
        // DefConverter(Double, Float);  => DefConverter(Float,  Double)
        // DefConverter(Double, Double); => direct convert
        DefConverter(Double, bool);
        DefConverter(Double, Variant);
        DefConverter(Double, String);

        // bool
        // DefConverter(bool, UInt64); => DefConverter(UInt64, bool)
        // DefConverter(bool, UInt32); => DefConverter(UInt32, bool)
        // DefConverter(bool, UInt16); => DefConverter(UInt16, bool)
        // DefConverter(bool, UInt8);  => DefConverter(UInt8,  bool)
        // DefConverter(bool, Int64);  => DefConverter(Int64,  bool)
        // DefConverter(bool, Int32);  => DefConverter(Int32,  bool)
        // DefConverter(bool, Int16);  => DefConverter(Int16,  bool)
        // DefConverter(bool, Int8);   => DefConverter(Int8,   bool)
        // DefConverter(bool, Float);  => DefConverter(Float,  bool)
        // DefConverter(bool, Double); => DefConverter(Double, bool)
        // DefConverter(bool, bool);   => direct convert
        DefConverter(bool, Variant);
        DefConverter(bool, String);

        // Variant
        // DefConverter(Variant, UInt64);  => DefConverter(UInt64, Variant)
        // DefConverter(Variant, UInt32);  => DefConverter(UInt32, Variant)
        // DefConverter(Variant, UInt16);  => DefConverter(UInt16, Variant)
        // DefConverter(Variant, UInt8);   => DefConverter(UInt8,  Variant)
        // DefConverter(Variant, Int64);   => DefConverter(Int64,  Variant)
        // DefConverter(Variant, Int32);   => DefConverter(Int32,  Variant)
        // DefConverter(Variant, Int16);   => DefConverter(Int16,  Variant)
        // DefConverter(Variant, Int8);    => DefConverter(Int8,   Variant)
        // DefConverter(Variant, Float);   => DefConverter(Float,  Variant)
        // DefConverter(Variant, Double);  => DefConverter(Double, Variant)
        // DefConverter(Variant, bool);    => DefConverter(bool,   Variant)
        // DefConverter(Variant, Variant); => direct convert
        DefConverter(Variant, String);

        // Variant
        // DefConverter(String, UInt64);  => DefConverter(UInt64, String)
        // DefConverter(String, UInt32);  => DefConverter(UInt32, String)
        // DefConverter(String, UInt16);  => DefConverter(UInt16, String)
        // DefConverter(String, UInt8);   => DefConverter(UInt8,  String)
        // DefConverter(String, Int64);   => DefConverter(Int64,  String)
        // DefConverter(String, Int32);   => DefConverter(Int32,  String)
        // DefConverter(String, Int16);   => DefConverter(Int16,  String)
        // DefConverter(String, Int8);    => DefConverter(Int8,   String)
        // DefConverter(String, Float);   => DefConverter(Float,  String)
        // DefConverter(String, Double);  => DefConverter(Double, String)
        // DefConverter(String, bool);    => DefConverter(bool,   String)
        // DefConverter(String, Variant); => DefConverter(bool,   String)
        // DefConverter(String, String);  => direct convert
    }
}   // namespace
