//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ModelBindingSourceMap_h)
#define Courier_DataBinding_ModelBindingSourceMap_h

#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/Foundation/Component.h>
#include <Courier/DataBinding/ModelBindingSource.h>
#include <Courier/Messaging/Message.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>

namespace Courier { namespace Internal { namespace DataBinding {
    /**
        <summary>
            Component data stores DataBinding related data for a component. Every component using DataBinding has a
            corresponding ComponentData object.
        </summary>
     */
    class ComponentData {
        public:
            ComponentData();

            /** return the number of binding sources in this component data*/
            BindingSourceIndex GetBindingSourceCount() const {
                return mBindingSourceCount;
            }

            /**
                <summary>Query if this object is valid (Setup has been invoked)</summary>
                <returns>true if valid, false if not.</returns>
             */
            bool IsValid() const {
                return mBindingSources != 0;
            }

            /**
                <summary>Gets the binding source with the given binding source index.</summary>
                <param name="idx">The index of the binding source.</param>
                <returns>
                    null if index out of bound or no binding source is defined for the component, else the binding
                    source.
                </returns>
             */
            Internal::ModelBindingSource* GetBindingSource(BindingSourceIndex idx) const {
                FEATSTD_DEBUG_ASSERT(idx < mBindingSourceCount);
                return (idx < mBindingSourceCount) ? mBindingSources[idx] : 0;
            }

            /**
                <summary>Setup the component data object with the given array of binding source objects.</summary>
                <param name="cid">The associated component id.</param>
                <param name="bs">An array of pointers to binding source objects.</param>
                <param name="bsCount">The number of binding source objects.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool Setup(ComponentId cid, Internal::ModelBindingSource **bs, BindingSourceIndex bsCount);

            /**
                <summary>Locates the binding source owning the data item with the given data item key.</summary>
                <param name="itemKey">The data item key.</param>
                <returns>null if it fails (the item key is not present in the binding source), else.</returns>
             */
            Internal::ModelBindingSource* Locate(DataItemKey itemKey) const;

        private:
            BindingSourceIndex mBindingSourceCount;
            Internal::ModelBindingSource **mBindingSources;
    };

/// @}
}}}

namespace Courier { namespace Internal {
/// @addtogroup COURIER_DATABINDING
/// @{
    class DataContext
    {
    public:
        FEATSTD_RTTI_DECLARATION();
        FEATSTD_TYPEDEF_SHARED_POINTER(DataContext);

        static const SharedPointer& Get(const Candera::AbstractNodePointer& node);
        static void Set(const Candera::AbstractNodePointer& node, const SharedPointer& dataContext);
        static bool IsSet(const Candera::AbstractNodePointer& node);
        static void Clear(const Candera::AbstractNodePointer& node);

        DataContext()
        {
        }

        virtual ~DataContext()
        {
        }

        virtual const Candera::AbstractNodePointer& GetNode() const;
        virtual const Courier::DataContextItem::SharedPointer& GetData(Courier::DataItemKey itemKey) const;
        virtual void SetData(const Courier::DataItemValue& dataItemValue);
        virtual ModelBindingSource* GetBindingSource(DataItemKey itemKey, const Candera::AbstractNodePointer& node);

    private:
        FEATSTD_SHARED_POINTER_DECLARATION();
    };

    /** maintains the binding sources of the different Components */
    class ModelBindingSourceMap {
        public:
            /** handle DataBinding message for the given component
                @param cid component id
                @param msg the message object (must be tagged with DataBinding)
                @return true if message was consumed, false otherwise */
            static bool OnDataBindingMsg(ComponentId cid, const Message *msg, const Candera::AbstractNodePointer& node);

            /** find the ModelBindingSource with the given id for the given component
                @param cid component id
                @param itemKey the id of the ModelBindingSource
                @param node the optional potential data context node.
                @return pointer to ModelBindingSource or 0 (null) if the binding source was not registered for the component */
            static ModelBindingSource* Locate(ComponentId cid, DataItemKey itemKey, const Candera::AbstractNodePointer& node);

            /**
                <summary>Gets the binding source with the given index.</summary>
                <param name="cid">The component id.</param>
                <param name="mapIdx">Zero-based index of the binding source.</param>
                <returns>null if it fails, else the binding source.</returns>
             */
            static ModelBindingSource* GetBindingSource(ComponentId cid, BindingSourceIndex mapIdx);

            /** registers the given bindings sources to the specified component
                registration may only done once in Generated::InitDataBindingGenerated
                @param cid the component id to register the bindings sources for
                @param bindingSources an array of pointers to ModelBindingSource objects
                @param bsCount the number of elements in the array bindingSources
                @return true if succeeded, false if parameter error */
            static bool Register(ComponentId cid, ModelBindingSource *bindingSources[], BindingSourceIndex bsCount);

            /**
                <summary>Executes the binding source post processing operation.</summary>
                <param name="cid"> The id of the component subscribed to the binding sources. </param>
             */
            static void DoBindingSourcePostProcessing(ComponentId cid);

        private:
            FEATSTD_MAKE_CLASS_STATIC(ModelBindingSourceMap);

            friend class Component;
            friend class Internal::ModelBindingSource;
    };

//@}
}}   // namespace

namespace Courier {
    inline bool Register(ComponentId cid, Internal::ModelBindingSource *bindingSources[], BindingSourceIndex bindingSourceCount) {
        return Internal::ModelBindingSourceMap::Register(cid, bindingSources, bindingSourceCount);
    }
}

#endif // Courier_DataBinding_ModelBindingSourceMap_h
