//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Diagnostics/Log.h>
#include <Courier/Util/BinarySearch.h>
#include "ModelBindingSourceMap.h"
#include "DataBindingMsg.h"
#include "AsyncModelBindingSource.h"
#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#endif
#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/Node.h>
#endif

namespace Courier { namespace Internal { namespace DataBinding {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

    using namespace Internal;

    // ------------------------------------------------------------------------
    ComponentData::ComponentData() : mBindingSourceCount(0), mBindingSources(0)
    {
    }

    // ------------------------------------------------------------------------
    bool ComponentData::Setup(ComponentId cid, ModelBindingSource **bs, BindingSourceIndex bsCount)
    {
        FEATSTD_DEBUG_ASSERT((mBindingSources == 0) && (bs != 0) && (bsCount > 0));
        FEATSTD_DEBUG_ASSERT(cid < COURIER_COMPONENT_MAX_COUNT);

        bool result = cid < COURIER_COMPONENT_MAX_COUNT ;
        if (result && (bs != 0) && (bsCount != 0)) {
            mBindingSources = bs;
            mBindingSourceCount = bsCount;

            for (BindingSourceIndex i = 0; i < bsCount; ++i) {
                if (mBindingSources[i] != 0) {
                    mBindingSources[i]->SetBindingSourceAffinity(cid);
                }
            }
        }

        return result;
    }

    // ------------------------------------------------------------------------
    ModelBindingSource* ComponentData::Locate(DataItemKey itemKey) const
    {
        DataItemKey bsKey = DataItemAccessor::GetBindingSourceIndex(itemKey);
        FEATSTD_DEBUG_ASSERT(bsKey < mBindingSourceCount);
        return ((mBindingSources != 0) && (bsKey < mBindingSourceCount)) ? mBindingSources[bsKey] : 0;
    }

    // ------------------------------------------------------------------------
    static ComponentData* Data(ComponentId cid)
    {
        FEATSTD_DEBUG_ASSERT(cid < COURIER_COMPONENT_MAX_COUNT);

        static ComponentData data[COURIER_COMPONENT_MAX_COUNT];
        return (cid < COURIER_COMPONENT_MAX_COUNT) ? (data + cid) : 0;
    }

    // ------------------------------------------------------------------------
    static inline bool IsValid(const ComponentData *cd)
    {
        return (cd != 0) && cd->IsValid();
    }
}}}

namespace Candera {
    namespace MetaInfo {
        namespace Internal {
            template<> struct DataType<Courier::Internal::DataContext::SharedPointer > {
                static inline bool ConvertToString(const Courier::Internal::DataContext::SharedPointer& /*val*/, Char* /*buf*/, UInt /*size*/) {
                    return false;
                }

                static inline bool ConvertFromString(Courier::Internal::DataContext::SharedPointer& /*val*/, const Char* /*buf*/) {
                    return false;
                }

#ifdef CANDERA_META_DESCRIPTION
                static inline const Char* GetEditor() {
                    return 0;
                }
#endif
            };
        }
    }
}
namespace Courier { namespace Internal {
    using namespace DataBinding;

    class DataContextProperties
    {
    public:
        static void SetDataContext(Candera::CanderaObject& node, const DataContext::SharedPointer& dataContext)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(DataContext), dataContext));
        }

        static const DataContext::SharedPointer& GetDataContext(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(DataContext));
        }

        static bool IsDataContextSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(DataContext));
        }

        static void ClearDataContext(Candera::CanderaObject& node)
        {
            static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(DataContext)));
        }

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
            const Candera::CanderaObject* object = static_cast<const Candera::CanderaObject*>(host);
#ifdef CANDERA_2D_ENABLED
            const Candera::Node2D* node2D = Candera::Dynamic_Cast<const Candera::Node2D*>(object);
            if (0 != node2D) {
                return node2D->GetParent();
            }
#endif
#ifdef CANDERA_3D_ENABLED
            const Candera::Node* node = Candera::Dynamic_Cast<const Candera::Node*>(object);
            if (0 != node) {
                return node->GetParent();
            }
#endif
            return 0;
        }

        static const DataContext::SharedPointer& DefaultDataContext() {
            static DataContext::SharedPointer s_default;
            return s_default;
        }

        CdaDynamicPropertiesDefinition(DataContextProperties, Candera::DynamicProperties::DynamicPropertyHost);
            CdaDynamicProperty(DataContext, Courier::Internal::DataContext::SharedPointer);
                CdaDynamicPropertyDefaultValue(DefaultDataContext())
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    FEATSTD_RTTI_BASECLASS_DEFINITION(DataContext)

    const DataContext::SharedPointer& DataContext::Get(const Candera::AbstractNodePointer& node)
    {
        return node.IsValid() ? DataContextProperties::GetDataContext(*node.ToCanderaObject()) : DataContextProperties::DefaultDataContext();
    }

    void DataContext::Set(const Candera::AbstractNodePointer& node, const DataContext::SharedPointer& dataContext)
    {
        if (node.IsValid()) {
            DataContextProperties::SetDataContext(*node.ToCanderaObject(), dataContext);
        }
    }

    bool DataContext::IsSet(const Candera::AbstractNodePointer& node)
    {
        return node.IsValid() ? DataContextProperties::IsDataContextSet(*node.ToCanderaObject()) : false;
    }

    void DataContext::Clear(const Candera::AbstractNodePointer& node)
    {
        if (node.IsValid()) {
            DataContextProperties::ClearDataContext(*node.ToCanderaObject());
        }
    }

    const Candera::AbstractNodePointer& DataContext::GetNode() const
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Candera::AbstractNodePointer, s_dummy);
        return s_dummy;
    }

    const Courier::DataContextItem::SharedPointer& DataContext::GetData(Courier::DataItemKey /*itemKey*/) const
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Courier::DataContextItem::SharedPointer, s_dummy);
        return s_dummy;
    }

    void DataContext::SetData(const Courier::DataItemValue& /*dataItemValue*/)
    {
    }

    ModelBindingSource* DataContext::GetBindingSource(DataItemKey /*itemKey*/, const Candera::AbstractNodePointer& /*node*/)
    {
        return 0;
    }

    // ------------------------------------------------------------------------
    bool ModelBindingSourceMap::Register(ComponentId cid, ModelBindingSource **bindingSources, BindingSourceIndex bsCount)
    {
        FEATSTD_DEBUG_ASSERT((bindingSources != 0) && (bsCount > 0));

        ComponentData *cd = Data(cid);
        // do not accept double registrations
        return (bindingSources != 0) &&
               (bsCount > 0) &&
               (cd != 0) &&
               (! cd->IsValid()) &&
               cd->Setup(cid, bindingSources, bsCount);
    }

    // ------------------------------------------------------------------------
    bool ModelBindingSourceMap::OnDataBindingMsg(ComponentId cid, const Message *msg, const Candera::AbstractNodePointer& node)
    {
        FEATSTD_DEBUG_ASSERT((msg != 0) && (msg->GetTag() == MessageTag::DataBinding));

        ComponentData *cd = Data(cid);
        if ((! IsValid(cd)) || (msg == 0) || (msg->GetTag() != MessageTag::DataBinding)) {
            return false;
        }

        switch(msg->GetId()) {
            case AbstractDataItemMsg::ID: {
                const AbstractDataItemMsg *m = message_cast<const AbstractDataItemMsg*>(msg);
                COURIER_LOG_DEBUG("received DataItemMsg for BindingSource %u (rev %u)", m->GetItemKey(), m->GetDataRevision());

                ModelBindingSource *bs = Locate(cid, DataItemAccessor::Root(m->GetItemKey()), node);
                if (bs != 0) {
                    bs->ProcessDataItemMsg(m);
                }
                break;
            }

            case ListEventMsg::ID: {
                const ListEventMsg *m = message_cast<const ListEventMsg*>(msg);
                COURIER_LOG_DEBUG("received ListEventMsg for BindingSource %u (rev %u)", m->GetItemKey(), m->GetDataRevision());

                ModelBindingSource *bs = Locate(cid, DataItemAccessor::Root(m->GetItemKey()), node);
                if (bs != 0) {
                    bs->ProcessListEventMsg(m);
                }
                break;
            }

            case ReleaseDataItemMsg::ID: {
                // trigger release in all binding sources of the component
                // start from 1 as binding source index 0 is reserved for invalid binding
                // source and is always 0
                for (BindingSourceIndex i = 1; i < cd->GetBindingSourceCount(); ++i) {
                    ModelBindingSource *bs = cd->GetBindingSource(i);
                    if (bs != 0) {
                        // trigger release of message reference
                        bs->ProcessReleaseDataItemMsgReference();
                    }
                }
                break;
            }

            default:
                COURIER_LOG_ERROR("unhandled DataBinding msg");
                FEATSTD_DEBUG_FAIL();
                break;
        }

        return true;
    }

    // ------------------------------------------------------------------------
    ModelBindingSource* ModelBindingSourceMap::Locate(ComponentId cid, DataItemKey itemKey, const Candera::AbstractNodePointer& node)
    {
        ComponentData *cd = Data(cid);
        if (IsValid(cd)) {
            if (cid == ComponentType::View) {
                Candera::AbstractNodePointer current = node;
                while (current.IsValid()) {
                    DataContext::SharedPointer dataContext = DataContext::Get(current);
                    if (!dataContext.PointsToNull()) {
                        ModelBindingSource* bs = dataContext->GetBindingSource(itemKey, node);
                        if (0 != bs) {
                            return bs;
                        }
                    }
                    current = current.GetParent();
                }
            }
            return cd->Locate(itemKey);
        }
        return 0;
    }

    // ------------------------------------------------------------------------
    void ModelBindingSourceMap::DoBindingSourcePostProcessing(ComponentId cid)
    {
        ComponentData *cd = Data(cid);
        if (!IsValid(cd)) {
            return;
        }

        // ignoring return value, nothing can be done here and trace is done in PostCachedUpdateModelMsg
        (void) AsyncModelBindingSource::PostCachedUpdateModelMsg();
    }

    // ------------------------------------------------------------------------
    ModelBindingSource* ModelBindingSourceMap::GetBindingSource(ComponentId cid, BindingSourceIndex mapIdx)
    {
        ComponentData *cd = Data(cid);
        return IsValid(cd) ? cd->GetBindingSource(mapIdx) : 0;
    }
}}   // namespace
