//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ListPropertyType_h)
#define Courier_DataBinding_ListPropertyType_h

#if defined(CANDERA_META_DESCRIPTION)
#   include <string>
#endif

#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <Candera/System/MetaInfo/DataType.h>
#include <Courier/DataBinding/SharedAsyncListInterface.h>
#include <Courier/DataBinding/GenericObjectList.h>
#include <Courier/DataBinding/ListEventHandler.h>
#include <Courier/Visualization/FrameworkWidget.h>

namespace Courier {
/// @addtogroup COURIER_DATABINDING
/// @{
    // ------------------------------------------------------------------------
    /**
        <summary>Generic base class for list widget properties.</summary>
        <seealso cref="AsyncListInterface"/>
     */
    class GenericListPropertyType : public AsyncListInterface
    {
        public:
            GenericListPropertyType();
            explicit GenericListPropertyType(const SharedPointer<SharedAsyncListInterface> &list);
            GenericListPropertyType(const GenericListPropertyType &prop);

            /**
                <summary>Query if property is connected to a (asynchronous) list.</summary>
                <returns>true if connected, false if not.</returns>
             */
            bool IsValid() const
            {
                return mList != 0;
            }

            /**
                <summary>Attach to the given list object.</summary>
                <param name="listObject">The list object.</param>
             */
            void AttachTo(const SharedPointer<SharedAsyncListInterface> &listObject)
            {
                mList = listObject;
            }

            SharedPointer<SharedAsyncListInterface> GetAttachedList() const
            {
                return mList;
            }

            bool operator==(const GenericListPropertyType& genericListPropertyType) const
            {
                return mList == genericListPropertyType.mList;
            }

            bool operator!=(const GenericListPropertyType& genericListPropertyType) const
            {
                return mList != genericListPropertyType.mList;
            }

            /**
                <summary>Set the callback function to be invoked when a list event happens.</summary>
                <param name="object">[in,out] The callback object or 0 (null).</param>
                <param name="CallbackFn">
                    [in,out] pointer to the member function of object.
                </param>
             */
            template<typename T> void SetEventCallback(T *object, void (T::*CallbackFn)(const ListEvent &listEvent))
            {
                if (!mList.PointsToNull()) {
                    mList->SetEventCallback(object, CallbackFn);
                }
            }

            virtual DataItemTypeId ListItemTypeId() const override;
            virtual FeatStd::SizeType Count() const override;
            virtual FeatStd::SizeType Capacity() const override;
            virtual bool EndOfList(FeatStd::SizeType index) const override;
            virtual FeatStd::SizeType FragmentStartIndex() const override;
            virtual FeatStd::SizeType FragmentItemCount() const override;
            virtual bool Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool updateOnly = false) override;
            virtual bool RequestChange(FeatStd::SizeType idx, const DataItemValue &item) override;
            virtual bool RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item) override;
            virtual bool RequestAdd(FeatStd::SizeType idx, const DataItemValue &item) override;
            virtual bool RequestRemove(FeatStd::SizeType idx) override;
            virtual bool RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src) override;
            virtual bool RequestClear() override;
            virtual bool RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems) override;
            virtual UInt32 Flags() const override;

            /**
                <summary>Executes the ListEvent - results in the invocation of the event handler callback.</summary>
                <param name="listEvent">The list event.</param>
             */
            void Invoke(const ListEvent &listEvent)
            {
                if (!mList.PointsToNull()) {
                    mList->InvokeListEventCallback(listEvent);
                }
            }

        protected:
            virtual void OnListEvent(const ListEvent &) { }

        private:
            SharedPointer<SharedAsyncListInterface> mList;  ///< reference counted list object
    };

    // ------------------------------------------------------------------------
    /**
        <summary>Typed concrete list property type.</summary>
        <seealso cref="GenericListPropertyType"/>
     */
    template<typename T> class ListPropertyType : public GenericListPropertyType {
        typedef GenericListPropertyType Base;
        public:
            FEATSTD_LINT_NEXT_EXPRESSION(1516, "intented hiding of base class type")
            typedef T ListItemType;

            ListPropertyType()
            {
            }

            explicit ListPropertyType(const SharedPointer<SharedAsyncListInterface> &listObject) :
                GenericListPropertyType(listObject)
            {
            }

            /**
                <summary>Typed version of AsyncListInterface::RequestChange.</summary>
                <param name="idx">The index.</param>
                <param name="item">The item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool RequestChange(FeatStd::SizeType idx, const T &item)
            {
                return Base::RequestChange(idx, DataItemValue(&item));
            }

            /**
                <summary>Typed version of AsyncListInterface::RequestAdd.</summary>
                <param name="idx">The index.</param>
                <param name="item">The item.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool RequestAdd(FeatStd::SizeType idx, const T &item)
            {
                return Base::RequestAdd(idx, DataItemValue(&item));
            }

    protected:
            using GenericListPropertyType::RequestAdd;
            using GenericListPropertyType::RequestChange;
    };

    typedef ListPropertyType<Courier::DataContextItem::SharedPointer> DynamicListPropertyType;

//@}
}

namespace Courier { namespace Internal { namespace DataBinding {
/// @addtogroup COURIER_DATABINDING
/// @{
#if defined(CANDERA_META_DESCRIPTION)
    std::string ComposeListPropertyEditorString(const Char *listItemEditor);
#endif

    typedef bool (*DeserializeValueSignature)(const Courier::DataItemValue &value, const Candera::Char *buf);

    /**
        <summary>Deserialize a list property. Generates a GenericObject list from serialized list items.</summary>
        <param name="objectList">[in,out]The resulting GenericObjectList object.</param>
        <param name="buf">The buffer with serialized list data.</param>
        <param name="itemTypeId">DataItemType of the list items.</param>
        <param name="DeserializeValue">Function pointer to deserialize a single list item.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    extern bool DeserializeListProperty(SharedPointer<GenericObjectList> &objectList,
                                        const Char *buf,
                                        DataItemTypeId itemTypeId,
                                        DeserializeValueSignature DeserializeValue);
//@}
}}}

namespace Candera { namespace MetaInfo {
    template<> struct DataType<Courier::DataContextItem::SharedPointer>
    {
#if defined(CANDERA_META_DESCRIPTION)
        static const Char* GetEditor()
        {
            return "";
        }
#endif

        static bool ConvertToString(const Courier::DataContextItem::SharedPointer& /*val*/, Char* /*buf*/, UInt /*bufLen*/)
        {
            return true;
        }

        static bool ConvertFromString(Courier::DataContextItem::SharedPointer& /*val*/, const Char* /*buf*/)
        {
            return true;
        }
    };

    template<> struct DataType<Courier::GenericListPropertyType> {
#if defined(CANDERA_META_DESCRIPTION)
        static const Char* GetEditor() {
            return "";
        }
#endif

        static bool ConvertToString(const Courier::GenericListPropertyType& /*val*/, Char* /*buf*/, UInt /*bufLen*/)
        {
            return true;
        }

        static bool ConvertFromString(Courier::GenericListPropertyType& /*val*/, const Char* /*buf*/)
        {
            return true;
        }
    };

    template<typename T> struct DataType<Courier::ListPropertyType<T> >
    {
#if defined(CANDERA_META_DESCRIPTION)
        static const Char* GetEditor()
        {
            using Courier::Internal::DataBinding::ComposeListPropertyEditorString;
            static const std::string cEditorString(ComposeListPropertyEditorString(Candera::MetaInfo::DataType<T>::GetEditor()));
            return cEditorString.c_str();
        }
#endif

        static bool ConvertToString(const Courier::ListPropertyType<T> &, Char *buffer, UInt bufferSize)
        {
            // serialization of list properties not supported / required
            FEATSTD_DEBUG_ASSERT((buffer != 0) && (bufferSize > 0));
            if ((buffer != 0) && (bufferSize > 0)) {
                *buffer = 0;
            }
            return true;
        }

        /**
            <summary>Deserialize list item from given character buffer.</summary>
            <param name="value">The value initialized with a mutable list item object of type T.</param>
            <param name="buf">The buffer with serialized list item.</param>
            <returns>true if deserializaton succeeded, false otherwise.</returns>
         */
        static bool DeserializeValue(const Courier::DataItemValue &value, const Candera::Char *buf)
        {
            T *object = value.GetMutableValue<T>();
            return (object != 0) && DataType<T>::ConvertFromString(*object, buf);
        }

        static bool ConvertFromString(Courier::ListPropertyType<T> &val, const Candera::Char *buf)
        {
            using Courier::SharedAsyncListInterface;
            using Courier::DataItemType;
            using Courier::Internal::DataBinding::DeserializeListProperty;
            using Courier::Internal::DataBinding::GenericObjectList;

            Courier::SharedPointer<GenericObjectList> objectList;
            bool ok = DeserializeListProperty(objectList, buf, DataItemType<T>::Ident(), &DeserializeValue);
            if (ok) {
                val.AttachTo(Courier::SharedPointer<SharedAsyncListInterface>(objectList.GetPointerToSharedInstance()));
            }
            return ok;
        }
    };
}}

#endif // Courier_DataBinding_ListPropertyType_h
