//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_ListEventHandler_h)
#define Courier_DataBinding_ListEventHandler_h

#include <Courier/DataBinding/ListInterfaceBase.h>
#include <Courier/DataBinding/DataItemValue.h>
#include <Courier/DataBinding/DataBindingTypes.h>

namespace Courier {
    /**
        <summary>List event contains information about events happened on a list.</summary>
     */
    class ListEvent {
        public:
            ListEvent(ListEventType::Enum eventType);
            ListEvent(ListEventType::Enum eventType, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex);
            ListEvent(ListEventType::Enum eventType, FeatStd::SizeType newIndex,
                      FeatStd::SizeType oldIndex, const DataItemValue &value);

            /**
                <summary>Gets the event type.</summary>
                <returns>The event type.</returns>
             */
            ListEventType::Enum EventType() const {
                return mEventType;
            }

            /**
                <summary>Sets the event type of the event.</summary>
                <param name="value">The value.</param>
             */
            void SetEventType(ListEventType::Enum value) {
                mEventType = value;
            }

            /**
                <summary>
                    Denotes the index of the item affected by the event
                    - ListEventType::AddedItem: the index of the newly added item
                    - ListEventType::RemovedItem: the index of the item that got removed
                    - ListEventType::MovedItem: the new index of the moved item
                    - ListEventType::ModifiedItem: the index of the changed item
                    - ListEventType::RequestedItem: the index of the requested item
                    - ListEventType::RefreshList: no meaning.
                    - ListEventType::ItemCountChanged: the update number of items in the list.
                </summary>
                <returns>index of the item affected by the event.</returns>
             */
            FeatStd::SizeType NewIndex() const {
                return mNewIndex;
            }

            /**
                <summary>Set the new index field of the event.</summary>
                <param name="value">The value.</param>
             */
            void SetNewIndex(FeatStd::SizeType value) {
                mNewIndex = value;
            }

            /**
                <summary>
                    contains the index of the item previous to the change in the list If the change/event type
                    affected the index of an item, OldIndex will return the index of the item prior to the change
                    Valid only for:
                    - ListEventType::MovedItem.
                    - ListEventType::ItemCountChanged: the previous number of items in the list
                </summary>
                <returns>index of the item prior to the list change if applicable, NewIndex otherwise.</returns>
             */
            FeatStd::SizeType OldIndex() const {
                return mOldIndex;
            }

            /**
                <summary>Set the old index field of the event.</summary>
                <param name="value">The value.</param>
             */
            void SetOldIndex(FeatStd::SizeType value) {
                mOldIndex = value;
            }

            /**
                <summary>Query if this object has a data item.</summary>
                <returns>true if data item, false if not.</returns>
             */
            bool HasDataItem() const {
                return mItemValue.IsValid();
            }

            /**
                <summary>Returns the DataItemValue to the item in case of ListEventType::RequestedItem.</summary>
                <returns>
                    DataItemValue to the requested list item, invalid DataItemValue the request was invalid (item got
                    removed etc.).
                </returns>
             */
            const DataItemValue& Item() const {
                return mItemValue;
            }

            /**
                <summary>Typed version of Item. Gets the item from the event object.</summary>
                <returns>pointer to data item, null if types don't match or invalid DataItemValue.</returns>
             */
            template<typename T> const T* GetItem() const {
                return mItemValue.GetValue<T>();
            }

            /**
                <summary>Set the item field of the event.</summary>
                <param name="value">The value.</param>
             */
            void SetItem(const DataItemValue &value) {
                mItemValue = value;
            }

        private:
            ListEventType::Enum mEventType;
            FeatStd::SizeType mNewIndex;
            FeatStd::SizeType mOldIndex;
            DataItemValue mItemValue;
    };

    typedef void (*ListEventSignature)(const ListEvent &listEvent, void *userData);

    /**
        <summary>Stores list event callback function and user data</summary>
     */
    class ListEventHandler {
        public:
            ListEventHandler();

            /**
                <summary>Query if this object has valid callback.</summary>
                <returns>true if valid, false if not.</returns>
             */
            bool IsValid() const {
                return mCallbackObject != 0;
            }

            /**
                <summary>Set the callback function to be invoked when a list event happens.</summary>
                <param name="object">[in,out] The callback object or 0 (null).</param>
                <param name="CallbackFn">
                    [in,out] pointer to the member function of object.
                </param>
             */
            template<typename T> inline void SetEventCallback(T *object, void (T::*CallbackFn)(const ListEvent &listEvent)) {
                // pointer to member functions of classes with multiple inheritance is not supported
                FEATSTD_COMPILETIME_ASSERT(sizeof(CallbackFn) == sizeof(Callback));
                InternalSetEventCallback(object, reinterpret_cast<Callback>(CallbackFn));
            }

            /**
                <summary>Resets the event callback.</summary>
             */
            void ResetEventCallback();

            /**
                <summary>Executes the event callback.</summary>
                <param name="listEvent">The list event.</param>
             */
            void operator()(const ListEvent &listEvent);

        private:
            typedef void (ListEventHandler::*Callback)(const ListEvent &listEvent);

            Callback mCallback;                         ///< The event callback
            void *mCallbackObject;                      ///< the callback object

            void InternalSetEventCallback(void *object, Callback CallbackFn);
    };

//@}
}

#endif // Courier_DataBinding_ListEventHandler_h
