//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_Infrastructure_h)
#define Courier_DataBinding_Infrastructure_h

#include <Courier/Util/Limits.h>
#include <Courier/Util/Identifier.h>
#include <Courier/DataBinding/DataItemType.h>
#include <FeatStd/Util/Alignment.h>
#include <FeatStd/Util/PodUtil.h>

namespace Courier {
    /// @addtogroup COURIER_DATABINDING
    /// @{
#if !defined(COURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX)
    /**
    <summary>
    Courier configuration option. defines the maximum hierarchy nodes that can exist in a binding source.
    </summary>
    */
    static const FeatStd::UInt32 cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX = 0xfff;
#define COURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX Courier::cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX
#else
    static const FeatStd::UInt32 cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX = COURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX;
#endif

#if !defined(COURIER_ITEM_DESCRIPTOR_MAX)
    /**
    <summary>
    Courier configuration option. Defines the maximum number of DataItemDescriptor that can exist in the
    application.
    </summary>
    */
    static const FeatStd::UInt32 cCOURIER_ITEM_DESCRIPTOR_MAX = 0xfff;
#define COURIER_ITEM_DESCRIPTOR_MAX Courier::cCOURIER_ITEM_DESCRIPTOR_MAX
#else
    static const FeatStd::UInt32 cCOURIER_ITEM_DESCRIPTOR_MAX = COURIER_ITEM_DESCRIPTOR_MAX;
#endif

#if !defined(COURIER_BINDING_SOURCE_MAX)
    /**
    <summary>
    Courier configuration option. Defines the maximum number of binding sources that can exist in a
    application.
    </summary>
    */
    static const FeatStd::UInt32 cCOURIER_BINDING_SOURCE_MAX = 0xfff;
#define COURIER_BINDING_SOURCE_MAX Courier::cCOURIER_BINDING_SOURCE_MAX
#else
    static const FeatStd::UInt32 cCOURIER_BINDING_SOURCE_MAX = COURIER_BINDING_SOURCE_MAX;
#endif

    /**
    <summary>Courier defines and intilizes databinding infrastructure.</summary>
    <param name="infrastructure">The infrastructure.</param>
    */
#define COURIER_DATABINDING_INFRASTRUCTURE(infrastructure) \
    const Courier::Internal::DataBinding::DataBindingInfrastructure * const Courier::Internal::DataBinding::gInfrastructure = &infrastructure

    /**
    <summary>
    Defines a default infrastructure if no source code generated data is in the application. Linker will
    report unresolved externals if no infrastructure is defined.
    </summary>
    */
#define COURIER_DATABINDING_DEFAULT_INFRASTRUCTURE() \
    const Courier::Internal::DataBinding::DataBindingInfrastructure * const Courier::Internal::DataBinding::gInfrastructure = &Courier::Internal::DataBinding::cDefaultInfrastructure

    /**
    <summary>
    Defines an alias representing zero-based index of the binding source. Depending on the number of
    BindingSources in the system, the data type will be UInt8, UIn16, or UIn32. Index 0 is reserved for
    invalid BindingSource.
    </summary>
    */
    typedef Courier::Internal::TaylorUnsignedIntType<cCOURIER_BINDING_SOURCE_MAX>::Type BindingSourceIndex;

    /**
    <summary>
    Defines an alias representing zero-based index of the hierarchy node. Depending on the maximum number of
    nodes in a BindingSource, this will be defined to UInt8, UInt16, or UInt32. Index 0 is the root node of
    the binding source.
    </summary>
    */
    typedef Courier::Internal::TaylorUnsignedIntType<cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX>::Type HierarchyNodeIndex;

    /**
    <summary>
    Defines an alias representing the data item key. DataItemKey is an index to the global HierarchyNode
    array (holding references to all HierarchyNodes in the system. Depending on the number of nodes in the
    system, this will be defined to UInt8, UInt16, or UInt32. Index 0 is reserved for the invalid
    HierarchyNode.
    </summary>
    */
    typedef Courier::Internal::TaylorUnsignedIntType<cCOURIER_BINDING_SOURCE_MAX * cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX>::Type DataItemKey;

    /**
    <summary>
    Defines an alias representing zero-based index of the item descriptor. Depending on the number if
    ItemDescriptor objects, this will be defined to UInt8, UInt16, or UInt32.
    </summary>
    */
    typedef Courier::Internal::TaylorUnsignedIntType<cCOURIER_ITEM_DESCRIPTOR_MAX>::Type ItemDescriptorIndex;

    // ------------------------------------------------------------------------
    static const HierarchyNodeIndex cBindingSourceHierarchyNodeMax = cCOURIER_BINDING_SOURCE_HIERARCHY_NODE_MAX;
    static const ItemDescriptorIndex cItemDescriptorMax = cCOURIER_ITEM_DESCRIPTOR_MAX;
    static const BindingSourceIndex cBindingSourceMax = cCOURIER_BINDING_SOURCE_MAX;

    static const BindingSourceIndex cInvalidBindingSourceIndex = 0;
    static const HierarchyNodeIndex cInvalidHierarchyNodeIndex = 0;
    static const DataItemKey cInvalidDataItemKey = 0;

    // ------------------------------------------------------------------------
    /** determines pointer to the given member of HostType
    @param HostType any structure or class
    @param member variable of HostType */
#define COURIER_ITEM_MEMBER(HostType, member) FEATSTD_UNCHECKED_OFFSET_OF(HostType, member)

    namespace Internal {
        namespace DataBinding {
            struct ZeroOffset {
                FeatStd::UInt32 mMember;
            };

            /** member pointer intializer for root objects */
#define COURIER_NULL_ITEM_MEMBER() COURIER_ITEM_MEMBER(::Courier::Internal::DataBinding::ZeroOffset, mMember)

            /** describes a member in a compound data type */
            struct DataItemDescriptor {
                DataItemTypeId mTypeId;                                     ///< data item type id
                FeatStd::SizeType mMemberOffset;                              ///< data member offset

                void* ToMemberPtr(void *object) const {
                    FEATSTD_LINT_NEXT_EXPRESSION(925, "this cast is required")
                        return static_cast<FeatStd::UInt8*>(object) + mMemberOffset;
                }

                const void* ToConstMemberPtr(const void *object) const {
                    return FeatStd::Internal::PointerToPointer<const UInt8*>(object) + mMemberOffset;
                }

                DataItemTypeId TypeId() const {
                    return mTypeId;
                }
            };

            // ------------------------------------------------------------------------
            namespace DataItemHierarchyNodeFlags {
                enum Enum {
                    None = 0,
                    Invalid = 1,            //(1 << 0),
                    LeaveNode = 2,          //(1 << 1),
                    ListNode = 4,           //(1 << 2),
                    ReadOnly = 8,           //(1 << 3),
                    ListAllowEdit = 16,     //(1 << 4),
                    ListAllowAdd = 32,      //(1 << 5),
                    ListAllowRemove = 64    //(1 << 6)
                };
            }

            namespace DataItemHierarchyNodeType {
                enum Enum {
                    Elementary,
                    Compound,
                    List
                };
            }

            /** describes a node in the hierarchy of data items */
            struct DataItemHierarchyNode {
                IdentifierStorage mIdent;                   ///< the identifier of the item
                FeatStd::UInt8 mFlags;                               ///< node specific flags
                BindingSourceIndex mBindingSourceIndex;     ///< index into DataBindingInfrastructure::mBindingSourceNodes
                HierarchyNodeIndex mParentIndex;            ///< index of the parent node
                HierarchyNodeIndex mLocalIndex;             ///< index of the node in node array
                ItemDescriptorIndex mDescriptorIndex;       ///< index of the DataItemDescriptor object
                DataItemKey mItemKey;                       ///< the data item key of this node

                bool IsLeaveNode() const {
                    return (mFlags & UInt8(DataItemHierarchyNodeFlags::LeaveNode)) != 0;
                }

                bool IsListNode() const {
                    return (mFlags & UInt8(DataItemHierarchyNodeFlags::ListNode)) != 0;
                }

                bool IsRootNode() const {
                    return mLocalIndex == 0;
                }

                bool IsReadOnly() const {
                    return (mFlags & UInt8(DataItemHierarchyNodeFlags::ReadOnly)) != 0;
                }

                bool TestFlag(DataItemHierarchyNodeFlags::Enum flag) const {
                    return (mFlags & UInt8(flag)) != 0;
                }

                DataItemHierarchyNodeType::Enum NodeType() const {
                    if(IsListNode()) {
                        return DataItemHierarchyNodeType::List;
                    } else {
                        if(IsLeaveNode()) {
                            return DataItemHierarchyNodeType::Elementary;
                        } else {
                            return DataItemHierarchyNodeType::Compound;
                        } 
                    }   
                }

                inline const DataItemDescriptor& Descriptor() const;

                DataItemTypeId TypeId() const {
                    return Descriptor().TypeId();
                }
            };

            // ------------------------------------------------------------------------
            struct BindingSourceRootStorage {
                UInt32 mNodeCount;
                const DataItemHierarchyNode *mNodes;
            };

            /** defines link to generated parts of DataBinding infrastructure */
            struct DataBindingInfrastructure {
                typedef bool (*InitSignature)();

                const DataItemDescriptor& Descriptor(UInt32 index) const {
                    FEATSTD_DEBUG_ASSERT(index < mDescriptorCount);
                    return (index < mDescriptorCount) ? mItemDesc[index] : mItemDesc[0];
                }

                const DataItemHierarchyNode& Node(DataItemKey itemKey) const {
                    FEATSTD_DEBUG_ASSERT(itemKey < mGlobalNodeListCount);
                    return (itemKey < mGlobalNodeListCount) ? *mGlobalNodeList[itemKey] : *mGlobalNodeList[0];
                }

                bool Init() const {
                    return (mInitFn != 0) && mInitFn();
                }

                const DataItemHierarchyNode& Node(BindingSourceIndex bsIndex, HierarchyNodeIndex nodeIndex) const;

                FeatStd::UInt32 mGlobalNodeListCount;                        ///< Number of data item identifier storages (ie. DataItemHierarchyNodes in all BindingSources)
                const DataItemHierarchyNode * const * mGlobalNodeList;      ///< pointer array to all nodes sorted by HashId

                FeatStd::UInt32 mBindingSourceCount;                         ///< the number of binding sources in the application
                const BindingSourceRootStorage *mBindingSourceNodes;  ///< pointer to an array of pointers to hierarchy nodes

                FeatStd::UInt32 mDescriptorCount;                            ///< number of DataItemDescriptors
                const DataItemDescriptor *mItemDesc;                ///< pointer to an array of pointers of DataItemDescriptor objects

                InitSignature mInitFn;                              ///< pointer to intialization function of generated databinding source
            };

            extern const DataBindingInfrastructure * const gInfrastructure;

            // ------------------------------------------------------------------------
            inline const DataItemDescriptor& DataItemHierarchyNode::Descriptor() const
            {
                return gInfrastructure->Descriptor(mDescriptorIndex);
            }
        }   // DataBinding
    } //Internal
    //@}
}

#include <Courier/DataBinding/DataItemAccessor.h>

namespace Courier { namespace Internal { namespace DataBinding {
    /** default item map for applications without DataBinding */
    extern const DataBindingInfrastructure cDefaultInfrastructure;
}}}

#endif // Courier_DataBinding_Infrastructure_h
