//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_GenericObjectList_h)
#define Courier_DataBinding_GenericObjectList_h

FEATSTD_LINT_FILE(967, GenericObjectList.h, "false positive - file has valid include guards.")

#include <Courier/DataBinding/SharedAsyncListInterface.h>

namespace Courier { namespace Internal { namespace DataBinding {
    /**
        <summary>
            List of generic objects. The list is initialized for a immutable capacity and object type. The list items
            are read-only, adding and replacing items is not possible through AsyncListInterface.
        </summary>
        <seealso cref="SharedAsyncListInterface"/>
     */
    class GenericObjectList : public SharedAsyncListInterface {
        public:
            GenericObjectList();
            virtual ~GenericObjectList();

            /**
                <summary>Initialises the container with the given capacity for the given type.</summary>
                <param name="capacity">The capacity of the container.</param>
                <param name="itemTypeId">Identifier for the item type.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool Init(FeatStd::SizeType capacity, DataItemTypeId itemTypeId);

            virtual DataItemTypeId ListItemTypeId() const override;
            virtual FeatStd::SizeType Count() const override;
            virtual FeatStd::SizeType Capacity() const override;
            virtual bool EndOfList(FeatStd::SizeType index) const override;

            virtual bool Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool updateOnly) override;
            virtual bool RequestChange(FeatStd::SizeType idx, const DataItemValue &item) override;
            virtual bool RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item) override;
            virtual bool RequestAdd(FeatStd::SizeType idx, const DataItemValue &item) override;
            virtual bool RequestRemove(FeatStd::SizeType idx) override;
            virtual bool RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src) override;
            virtual bool RequestClear() override;
            virtual bool RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems) override;
            virtual UInt32 Flags() const override;

            /**
                <summary>Appends the given object to the container.</summary>
                <param name="value">The value.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool PushBack(const DataItemValue &value);

            /**
                <summary>Typed version of PushBack.</summary>
                <param name="value">The value.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            template<typename T> inline bool PushBack(const T &value) {
                return PushBack(DataItemValue(&value));
            }

            /**
                <summary>
                    Deserializes the item with the given Deserialize function pointer and adds the deserialized
                    object to the collection.
                </summary>
                <param name="buffer">The buffer holding the serialized data.</param>
                <param name="DeserializeValue">function pointer to the deserialization function.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool PushBack(const Char *buffer, DeserializeValueSignature DeserializeValue);

            /**
                <summary>Array indexer operator.</summary>
                <param name="index">Zero-based index.</param>
                <returns>
                    DataItemValue referencing the requested item or an invalid DataItemValue if the index was out of
                    range.
                </returns>
             */
            DataItemValue operator[](FeatStd::SizeType index) const;

            /**
                <summary>returns a pointer to the item with the given index.</summary>
                <param name="index">Zero-based index.</param>
                <returns>null if it types dont match or index out of range.</returns>
             */
            template<typename T> inline const T* Item(FeatStd::SizeType index) const {
                return operator[](index).GetValue<T>();
            }

        protected:
            virtual void OnReferrerUpdate(bool addReferrer) override;

        private:
            UInt32 mRefCount;                   ///< the reference count of the object
            FeatStd::SizeType mItemCount;                  ///< number of items in the container
            FeatStd::SizeType mItemSize;                   ///< the size of a single container item
            FeatStd::SizeType mCapacity;                   ///< the capacity of the container
            DataItemTypeId mItemTypeId;         ///< the type id of container items
            UInt8 *mBuffer;                     ///< pointer to the container items buffer

            /**
                <summary>Returns pointer of the item at the given index.</summary>
                <param name="index">Zero-based index.</param>
                <returns>returns invalid pointers if index out of range.</returns>
             */
            void* ItemPtr(FeatStd::SizeType index) {
                return mBuffer + (mItemSize * index);
            }

            /**
                <summary>Returns pointer of the item at the given index.</summary>
                <param name="index">Zero-based index.</param>
                <returns>returns invalid pointers if index out of range.</returns>
             */
            const void* ItemPtr(FeatStd::SizeType index) const {
                return mBuffer + (mItemSize * index);
            }

            FEATSTD_MAKE_CLASS_UNCOPYABLE(GenericObjectList);
    };

//@}
}}}

#endif // Courier_DataBinding_GenericObjectList_h
