//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Platform/Memory.h>

#include "DataItemType.h"
#include "GenericObjectList.h"

namespace Courier { namespace Internal { namespace DataBinding {
    using namespace Internal;
    // ------------------------------------------------------------------------
    GenericObjectList::GenericObjectList() : mRefCount(0), mItemCount(0), mItemSize(0), mCapacity(0), 
                                             mItemTypeId(InvalidDataItemType::Ident()), mBuffer(0)
    {
    }

    // ------------------------------------------------------------------------
    GenericObjectList::~GenericObjectList()
    {
        if (mBuffer != 0) {
            while (mItemCount > 0) {
                --mItemCount;
                mItemTypeId->Destruct(ItemPtr(mItemCount));
            }
            FEATSTD_DELETE_ARRAY(mBuffer);
            mBuffer = 0;
        }
        mItemTypeId = 0;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::Init(FeatStd::SizeType capacity, DataItemTypeId itemTypeId)
    {
        FEATSTD_DEBUG_ASSERT(capacity > 0);

        if ((mBuffer != 0) || (capacity == 0)) {
            return false;
        }

        FeatStd::SizeType bufferSize = static_cast<FeatStd::SizeType>(((capacity * itemTypeId->Size()) + 3) / 4);
        mBuffer = reinterpret_cast<UInt8*>(FEATSTD_NEW_ARRAY(UInt32, bufferSize));
        if (mBuffer != 0) {
            mCapacity = capacity;
            mItemTypeId = itemTypeId;
            mItemSize = itemTypeId->Size();
        }
        return mBuffer != 0;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::PushBack(const DataItemValue &value)
    {
        bool ok = (mBuffer != 0) &&
                  (mItemCount < mCapacity) &&
                  (value.TypeId() == mItemTypeId) &&
                  value.IsValid();

        if (ok) {
            mItemTypeId->Construct(ItemPtr(mItemCount), value);
            ++mItemCount;
        }

        return ok;
    }

    // ------------------------------------------------------------------------
    DataItemValue GenericObjectList::operator[](FeatStd::SizeType index) const
    {
        FEATSTD_DEBUG_ASSERT(index < mItemCount);
        FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
        return DataItemValue(mItemTypeId, (index < mItemCount) ? ItemPtr(index) : 0);
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::PushBack(const Char *buffer, DeserializeValueSignature DeserializeValue)
    {
        bool ok = (mBuffer != 0) && (mItemCount < mCapacity);

        if (ok) {
            // get item pointer
            void *itemPtr = ItemPtr(mItemCount);

            // construct the item
            mItemTypeId->Construct(itemPtr, DataItemValue());

            // deserialize the item value from buffer
            DataItemValue value(mItemTypeId, itemPtr, true);
            ok = DeserializeValue(value, buffer);
            if (ok) {
                // deserialization succeeded -> increase item count
                ++mItemCount;
            }
            else {
                // deserialization failed -> destruct constructed item again
                mItemTypeId->Destruct(itemPtr);
            }
        }

        return ok;
    }

    // ------------------------------------------------------------------------
    DataItemTypeId GenericObjectList::ListItemTypeId() const
    {
        return mItemTypeId;
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericObjectList::Count() const
    {
        return mItemCount;
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericObjectList::Capacity() const
    {
        return mCapacity;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::EndOfList(FeatStd::SizeType index) const
    {
        return index >= mItemCount;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool /*updateOnly*/)
    {
        if (mBuffer != 0) {
            for (FeatStd::SizeType i = index; (i < mItemCount) && (nItems > 0); ++i) {
                FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
                ListEvent listEvent(ListEventType::RequestedItem,
                                    i,
                                    cInvalidListIndex,
                                    DataItemValue(mItemTypeId, ItemPtr(i)));
                OnListEvent(listEvent);
                --nItems;
            }
        }
        return mBuffer != 0;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestChange(FeatStd::SizeType, const DataItemValue &)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestChangeDataItem(FeatStd::SizeType, DataItemKey, const DataItemValue&)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestAdd(FeatStd::SizeType, const DataItemValue &)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestRemove(FeatStd::SizeType)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestMove(FeatStd::SizeType, FeatStd::SizeType)
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestClear()
    {
        return false;
    }

    // ------------------------------------------------------------------------
    bool GenericObjectList::RequestPrefetch(FeatStd::SizeType, FeatStd::SizeType)
    {
        return true;
    }

    // ------------------------------------------------------------------------
    UInt32 GenericObjectList::Flags() const
    {
        return UInt32(ListFlags::None);
    }

    // ------------------------------------------------------------------------
    void GenericObjectList::OnReferrerUpdate(bool addReferrer)
    {
        if (addReferrer) {
            ++mRefCount;
        }
        else if (mRefCount == 1) {
            FEATSTD_DELETE(this);
        }
        else {
            --mRefCount;
        }
    }
}}}
