//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_DataItemValue_h)
#define Courier_DataBinding_DataItemValue_h

#include <Courier/Base.h>

#include <Courier/DataBinding/DataItemType.h>

#include <FeatStd/MemoryManagement/PlacementNew.h>


namespace Courier {

    /// @addtogroup COURIER_DATABINDING
    /// @{
    /** holds a reference to an arbitrary data item
    DataItemValue are temporary objects that reference arbitrary objects. The ownership
    of the object is not controlled by DataItemValue. Objects referenced by DataItemValue
    must not be destroyed before the referencing DataItemValue. Objects referenced by the
    DataItemValue can be const or mutable. DataItemValue also holds the type of the
    referenced object and performs type checks before reading or writing to the object. */
    class DataItemValue {
    public:
        /** default ctor - initialize invalid DataItemValue object */
        DataItemValue();

        /** Create a an object holding a reference to the given item
        @param item pointer to the object referenced by this DataItemValue object
        @param isMutable defines if the DataItemValue shall be mutable or not */
        template<typename T> inline DataItemValue(T *item, bool isMutable)
            : mIsMutable(isMutable), mDataPtr(const_cast<T *>(item)), mTypeId(DataItemType<T>::Ident()) {}

        /** creates an immutable DataItemValue object referencing the given item
        @param item the object referred to by this DataItemValue */
        template<typename T> inline explicit DataItemValue(const T *item)
            : mIsMutable(false), mDataPtr(const_cast<T *>(item)), mTypeId(DataItemType<T>::Ident()) {}

        /** creates a DataItemValue object with the given typeId and object
        @param typeId the type id of item
        @param item pointer to the object
        @param isMutable determines if the object is mutable or not */
        DataItemValue(DataItemTypeId typeId, void *item, bool isMutable)
            FEATSTD_LINT_NEXT_EXPRESSION(925, "this cast is required")
            : mIsMutable(isMutable), mDataPtr(const_cast<void*>(item)), mTypeId(typeId) {}

        /** creates an immutable DataItemValue object with the given typeId and object
        @param typeId the type id of item
        @param item pointer to the object */
        DataItemValue(DataItemTypeId typeId, const void *item);

        /** Initialize the DataItemValue with the given type and object
        @param typeId the DataItemTypeId of the referred object
        @param item pointer to the referred object. It is in the callers responsibility
        that the object matches the DataItemTypeId given in parameter typeId */
        FEATSTD_LINT_NEXT_EXPRESSION(818, "item could be made const, but the semantics of the function is about mutable data, thus remains non const")
            void InitMutable(DataItemTypeId typeId, void *item) {
                InternalInit(item, true, typeId);
        }

        /** Initialize the DataItemValue with the given object
        @param item pointer to the referred object. */
        template<typename T> inline void InitMutable(T *item) {
            InternalInit(item, true, DataItemType<T>::Ident());
        }

        /** Initialize the DataItemValue with the given type and object
        @param typeId the DataItemTypeId of the referred object
        @param item pointer to the referred object. It is in the callers responsibility
        that the object matches the DataItemTypeId given in parameter typeId */
        void InitImmutable(DataItemTypeId typeId, const void *item) {
            InternalInit(item, false, typeId);
        }

        /** Initialize the DataItemValue with the given object
        @param item pointer to the referred object. */
        template<typename T> inline void InitImmutable(T *item) {
            InternalInit(item, false, DataItemType<T>::Ident());
        }

        /** returns if DataItemValue holds a valid reference or not */
        bool IsValid() const {
            return mDataPtr != 0;
        }

        /** returns true if the item reference by the object is mutable or not */
        bool IsMutable() const {
            return mIsMutable;
        }

        /** resets the DataItemValue object to invalid*/
        void Reset();

        /** return the DataItemTypeId of the referred object */
        DataItemTypeId TypeId() const {
            return mTypeId;
        }

        /**
        <summary>Gets the mutable pointer to the referenced object.</summary>
        <returns>null if invalid or immutable value object.</returns>
        */
        void* MutablePointer() const {
            return (IsMutable()) ? mDataPtr : 0;
        }

        /**
        <summary>Gets the immutable pointer to the referenced object.</summary>
        <returns>null if invalid value object.</returns>
        */
        const void* ImmutablePointer() const {
            return mDataPtr;
        }

        /** returns a pointer to the reference object
        @return pointer to the referenced object or 0 (null) if the type requested
        does not match the value type. */
        template<typename T> const T* GetValue() const {
            return Check<T>() ? reinterpret_cast<const T*>(mDataPtr) : 0;
        }

        /** return a non-const pointer to the referred object
        @return pointer to the referred object, 0 (null) if requested type does not
        match stored type or the value object is immutable. */
        template<typename T> T* GetMutableValue() const {
            return (IsMutable() && Check<T>()) ? reinterpret_cast<T*>(mDataPtr) : 0;
        }

        /** set the referred object to the given value
        @param value the value to be set
        @return true if value could be set, false if the DataItemValue is immutable,
        type of value does not match stored type, or DataItemValue object
        is invalid. */
        template<typename T> bool SetValue(const T &value) const {
            bool ok = IsMutable() && Check<T>();
            if (ok) {
                *reinterpret_cast<T*>(mDataPtr) = value;
            }
            return ok;
        }

        /**
        <summary>Copies the referenced vitem to the given item.</summary>
        <param name="item">The item to receive the value.</param>
        <returns>true if object is valid and types match, false otherwise.</returns>
        */
        template<typename T> bool CopyTo(T &item) const {
            bool ok = Check<T>();
            if (ok) {
                item = *reinterpret_cast<T*>(mDataPtr);
            }
            return ok;
        }

        /**
        @brief Sets a value to the reference
        this object must be a valid, mutable DataItemValue. Also TypeId must
        match the given TypeID.
        @param  value The value.
        @return true if the value could be assigned, false otherwise
        */
        bool SetValue(const DataItemValue & value) const;

    private:
        bool mIsMutable;                ///< true if the referred object is not const
        void *mDataPtr;                 ///< pointer to the referred object
        DataItemTypeId mTypeId;         ///< DataItemTypeId of the referred object

        FEATSTD_LINT_INITIALIZER_FUNCTION(Courier::DataItemValue::InternalInit)

            /** intializes the DataItemValue object
            @param item referred object
            @param isMutable false if the referred object should be treated as const
            @param typeId the type of the referred object */
            void InternalInit(const void *item, bool isMutable, DataItemTypeId typeId);

        /** checks the given vs. the stored type */
        template<typename T> inline bool Check() const {
            return IsValid() && (DataItemType<T>::Ident() == mTypeId);
        }
    };

    namespace Internal {
        namespace DataBinding {
            /**
            <summary>
            Provides basic operations Construct, Destruct, and Assign for the specified type.
            As construction and destruction are not supported for abstract types and void, a
            specialization of the class provides empty implementations for abstract and void
            type.
            </summary>

            */
            template<typename T, bool isAbstract = bool(Internal::IsAbstract<T>::Value)> struct TypeOperations {
                /**
                <summary>
                Assigns src to dest.
                This internal functions does not check if the given pointers are valid (non-null).
                Pointers are passed to generate valid parameter definitions for type 'void'.
                </summary>
                <param name="dst">[in,out] The receiver of the value. Must be non-null.</param>
                <param name="src">The source of the value. Must be non-null.</param>
                */
                static inline void Assign(T *dst, const T *src) {
                    *dst = *src;
                }

                /**
                <summary>Constructs at the given location an object of type T with initial value initVal.</summary>
                <param name="p">
                [in,out] pointer to the buffer the object should be constructed with. Must not be 0 (null)
                </param>
                <param name="initVal">The initialise value, may be 0 (null).</param>
                */
                static inline void Construct(void *p, const T *initVal) {
                    if (initVal != 0) {
                        FeatStd::MemoryManagement::Construct<T>(static_cast<T*>(p), *initVal);
                    }
                    else {
                        FeatStd::MemoryManagement::Construct<T>(static_cast<T*>(p));
                    }
                }

                /**
                <summary>Destructs the object of type referenced by parameter p.</summary>
                <param name="p">[in,out] If non-null, the T object at p will be destructed.</param>
                <returns>The size of type T in number of bytes.</returns>
                */
                static inline UInt32 Destruct(T *p) {
                    if (p != 0) {
                        FeatStd::MemoryManagement::Destruct(p);
                    }
                    return UInt32(sizeof(T));
                }
            };

            /** specialization of TypeOperations for arrays of size N */
            template<typename T, UInt32 N>
            struct TypeOperations<T[N], false> {
                typedef T ArrayType[N];

                static inline void Assign(ArrayType *dst, const ArrayType *src) {
                    for (UInt32 i = 0; i < N; ++i) {
                        (*dst)[i] = (*src)[i];
                    }
                }

                static inline void Construct(void *p, const ArrayType *initVal) {
                    if (initVal != 0) {
                        for (UInt32 i = 0; i < N; ++i) {
                            FeatStd::MemoryManagement::Construct<T>((*static_cast<ArrayType*>(p)) + i, (*initVal)[i]);
                        }
                    }
                    else {
                        for (UInt32 i = 0; i < N; ++i) {
                            FeatStd::MemoryManagement::Construct<T>((*static_cast<ArrayType*>(p)) + i);
                        }
                    }
                }

                static inline UInt32 Destruct(ArrayType *p) {
                    if (p != 0) {
                        for (UInt32 i = 0; i < N; ++i) {
                            FeatStd::MemoryManagement::Destruct((*p) + i);
                        }
                    }
                    return UInt32(sizeof(T[N]));
                }
            };

            /**
            <summary>Dummy implementation of the type operations for abstract types and void./summary>
            */
            template<typename T> struct TypeOperations<T, true> {
                static inline void Assign(T *, const T *) {
                    FEATSTD_DEBUG_FAIL();
                };

                static inline void Construct(void *, const T *) {
                    FEATSTD_DEBUG_FAIL();
                }

                static inline UInt32 Destruct(T *) {
                    FEATSTD_DEBUG_FAIL();
                    return 0;
                }
            };

            // ------------------------------------------------------------------------
            template<typename T> void DataTypeInfo::AssignImpl(const DataItemValue &dst, const DataItemValue &src)
            {
                TypeOperations<T>::Assign(dst.GetMutableValue<T>(), src.GetValue<T>());
            }

            // ------------------------------------------------------------------------
            template<typename T> void DataTypeInfo::ConstructImpl(void *p, const DataItemValue &initValue)
            {
                FEATSTD_DEBUG_ASSERT((!initValue.IsValid()) || ((initValue.TypeId() == DataItemType<T>::Ident()) && (initValue.GetValue<T>() != 0)));
                TypeOperations<T>::Construct(p, initValue.GetValue<T>());
            }

            // ------------------------------------------------------------------------
            template<typename T> UInt32 DataTypeInfo::DestructImpl(void *p)
            {
                return TypeOperations<T>::Destruct(static_cast<T*>(p));
            }
        }
    }

    FEATSTD_LINT_MESSAGE_STRING(1963, "*14-8-2*Courier::DataItemValue::DataItemValue*",
        "DataItemType is no valid value for a DataItemValue. "
        "Thus template and non template ctor have different "
        "semantics and remaining parameters guarantee that "
        "functions cannot be confused.")


        //@}
}   // namespace

#include <Courier/DataBinding/DataContextItem.h>

#endif // Courier_DataBinding_DataItemValue_h
