//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_DataItemType_h)
#define Courier_DataBinding_DataItemType_h

#include <Courier/Util/Traits.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/SyncListInterface.h>

namespace Courier {
    /// @addtogroup COURIER_DATABINDING
    /// @{
    /** use this macro to resolve static references to BindingTypeIds
    for static initializations, this macro can be used to resolve the reference
    to the given type identifier.
    @param type the type to retrieve the DataItemTypeId for
    @code
    static DataItemTypeId cBoolTypeId = COURIER_DATA_ITEM_TYPE_ID(bool);
    @endcode*/
#define COURIER_DATA_ITEM_TYPE_ID(type) \
    (&::Courier::Internal::DataBinding::DataTypeInfoInstanceProvider< ::Courier::Internal::PureType< ::Courier::Internal::DataItemTypeMapper< type >::Type >::Type >::mDataTypeInfo)

#define COURIER_DATA_ITEM_TYPENAME_ID(type) \
    (&::Courier::Internal::DataBinding::DataTypeInfoInstanceProvider< typename ::Courier::Internal::PureType< typename ::Courier::Internal::DataItemTypeMapper< type >::Type >::Type >::mDataTypeInfo)

#define COURIER_INVALID_DATA_ITEM_TYPE_ID() \
    (&::Courier::Internal::DataBinding::DataTypeInfoInstanceProvider<Courier::Internal::DataBinding::InvalidType>::mDataTypeInfo)

    namespace Internal {
        template <FeatStd::UInt bytes>
        struct DataItemTypeMapperHelper
        {
        };

        template <>
        struct DataItemTypeMapperHelper<2>
        {
            typedef Int16 Int;
            typedef UInt16 UInt;
            typedef UInt16 SizeType;
        };

        template <>
        struct DataItemTypeMapperHelper<4>
        {
            typedef Int32 Int;
            typedef UInt32 UInt;
            typedef UInt32 SizeType;
        };

        template <>
        struct DataItemTypeMapperHelper<8>
        {
            typedef Int64 Int;
            typedef UInt64 UInt;
            typedef UInt64 SizeType;
        };

        template <typename T>
        struct DataItemTypeMapper
        {
            typedef T Type;
        };

        template <>
        struct DataItemTypeMapper<FeatStd::Int>
        {
            typedef DataItemTypeMapperHelper<sizeof(FeatStd::Int)>::Int Type;
        };

        template <typename T1, typename T2> struct IsSameType {
            enum {
                Value = 0
            };
        };

        template <typename T> struct IsSameType<T, T> {
            enum {
                Value = 1
            };
        };

        template <FeatStd::UInt value> struct SafeUIntType {
            typedef FeatStd::UInt Type;
        };

        struct DummyType { };

        template <> struct SafeUIntType<1> {
            typedef DummyType Type;
        };

        template <>
        struct DataItemTypeMapper< SafeUIntType< IsSameType<FeatStd::UInt, FeatStd::SizeType >::Value >::Type >
        {
            typedef DataItemTypeMapperHelper<sizeof(FeatStd::UInt)>::UInt Type;
        };

        template <>
        struct DataItemTypeMapper<FeatStd::SizeType>
        {
            typedef DataItemTypeMapperHelper<sizeof(FeatStd::SizeType)>::SizeType Type;
        };

        namespace DataBinding {
            /**
            <summary>Helper for InvalidDataType declaration.</summary>
            */
            struct InvalidType {
            };

            namespace ListInterfaceType {
                /**
                <summary>Identifies the type of a list interface (sync / async)./</summary>
                */
                enum Enum {
                    Asynchronous,                           ///< the type exposes a AsynchronousListInterface
                    Synchronous,                            ///< the type exposes a SynchronousListInterface
                    NoList                                  ///< type does not expose a list interface
                };
            }

            /**
            <summary>Types derived from AsyncListInterface or SyncListInterface </summary>
            */
            struct ListDataTypeInfo {
                ListInterfaceType::Enum mInterfaceType;     ///< Synchronous or Asynchronous
                DataItemTypeId mListElementTypeId;          ///< the type id of the list items

                inline ListInterfaceType::Enum InterfaceType() const {
                    return mInterfaceType;
                }

                inline DataItemTypeId ListItemTypeId() const {
                    return mListElementTypeId;
                }
            };

            /**
            <summary>DataTypeInfo holds information for all data types used in DataBinding.</summary>
            */
            struct DataTypeInfo {
                /**
                <summary>Signature of the assign function</summary>
                */
                typedef void(*AssignSignature)(const DataItemValue &dst, const DataItemValue &src);
                typedef void (*ConstructSignature)(void *p, const DataItemValue &initValue);
                typedef UInt32 (*DestructSignature)(void *p);

                AssignSignature mAssignFn;                      ///< The assign function
                ConstructSignature mConstructFn;
                DestructSignature mDestructFn;

                /** extended information on list data types or null if no list type
                if the data type is derived from AsyncListInterface or SyncListInterface, this
                member will refer to extended type information. In all other cases this member
                is initialized to 0 (null) */
                const ListDataTypeInfo *mListDataTypeInfo;

                /**
                <summary>Helper to invoke assign function</summary>
                <param name="dst">Destination value.</param>
                <param name="src">Source value.</param>
                */
                inline void Assign(const DataItemValue &dst, const DataItemValue &src) const {
                    mAssignFn(dst, src);
                }

                inline void Construct(void *p, const DataItemValue &initValue) const {
                    FEATSTD_DEBUG_ASSERT(p != 0);
                    mConstructFn(p, initValue);
                }

                inline UInt32 Size() const {
                    return mDestructFn(0);
                }

                inline void Destruct(void *p) const {
                    FEATSTD_DEBUG_ASSERT(p != 0);
                    (void) mDestructFn(p);
                }

                template<typename T> static void AssignImpl(const DataItemValue &dst, const DataItemValue &src);
                template<typename T> static void ConstructImpl(void *p, const DataItemValue &initValue);
                template<typename T> static UInt32 DestructImpl(void *p);

                /**
                <summary>Query if this DataTypeInfo belongs to a list type./</summary>
                <returns>true if list type, false if not.</returns>
                */
                inline bool IsListType() const {
                    return mListDataTypeInfo != 0;
                }

                inline ListInterfaceType::Enum ListType() const {
                    return (mListDataTypeInfo != 0) ? mListDataTypeInfo->InterfaceType() : ListInterfaceType::NoList;
                }

                inline DataItemTypeId ListItemTypeId() const;

                inline const ListDataTypeInfo* GetListDataTypeInfo() const {
                    return mListDataTypeInfo;
                }
            };

            /**
            <summary>List interface type provider. Default for non list interface exposing types.</summary>
            */
            template<ListInterfaceType::Enum lifType> struct ListInterfaceTypeProvider {
                /**
                <summary>Dummy type for non list types</summary>
                */
                typedef InvalidType Type;
            };

            /**
            <summary>Specialization - the interface type for Synchronous lists is SyncListInterface</summary>
            */
            template<> struct ListInterfaceTypeProvider<ListInterfaceType::Synchronous> {
                typedef SyncListInterface Type;
            };

            /**
            <summary>Specialization - the interface type for Asynchronous lists is AsyncListInterface</summary>
            */
            template<> struct ListInterfaceTypeProvider<ListInterfaceType::Asynchronous> {
                typedef AsyncListInterface Type;
            };

            /**
            <summary>Helper class to detect classes derived from SyncListInterface or AsyncListInterface</summary>
            */
            template<typename T> struct ListInterfaceIdentifier {
                /// true if the given type exposes a SyncListInterface
                static const bool IsSyncListInterface = Internal::IsDerivedFrom<T, SyncListInterface>::Value;

                /// true if the given type exposes a AsyncListInterface
                static const bool IsAsyncListInterface = Internal::IsDerivedFrom<T, AsyncListInterface>::Value;

                /// true if the given type exposes a SyncListInterface or AsyncListInterface
                static const bool IsListInterface = IsSyncListInterface || IsAsyncListInterface;

                /// evaluates depending on the given type to one of the ListInterfaceType enumeration values
                FEATSTD_LINT_NEXT_EXPRESSION(944, "it is intended to evaluate to a compile time const value")
                    static const ListInterfaceType::Enum Value = static_cast<ListInterfaceType::Enum>(IsSyncListInterface ? ListInterfaceType::Synchronous :
                    (IsAsyncListInterface ? ListInterfaceType::Asynchronous : ListInterfaceType::NoList));

                /// defines the corresponding interface type for the identified ListInterfaceType
                typedef typename ListInterfaceTypeProvider<Value>::Type Type;
            };

            /**
            <summary>
            Helper class to extract the ListItemType from list types This default implementation is used for non
            list types. The ListItemType will be defaulted to void.
            </summary>
            */
            template<typename T, bool isListType> struct ListElementTypeExtractor {
                typedef InvalidType ListItemType;
            };

            /**
            <summary>This specialization expects that all list types expose a ListItemType.</summary>
            */
            template<typename T> struct ListElementTypeExtractor<T, true> {
                typedef typename T::ListItemType ListItemType;
            };

            /**
            <summary>This class provides an ListDataTypeInfo instance for the given list type T.</summary>
            */
            template<typename T> struct ListDataTypeInfoInstanceProvider {
                typedef typename ListElementTypeExtractor<T, ListInterfaceIdentifier<T>::IsListInterface>::ListItemType ListItemType;
                static const ListDataTypeInfo mListDataTypeInfo;
            };

            /**
            <summary>This class provides a DataTypeInfo instance for the given type T</summary>
            */
            template<typename T> struct DataTypeInfoInstanceProvider {
                static const DataTypeInfo mDataTypeInfo;
            };

            template<typename T> const ListDataTypeInfo ListDataTypeInfoInstanceProvider<T>::mListDataTypeInfo = {
                ListInterfaceIdentifier<T>::Value,
                COURIER_DATA_ITEM_TYPENAME_ID(ListItemType)
            };

            template<typename T> const DataTypeInfo DataTypeInfoInstanceProvider<T>::mDataTypeInfo = {
                &DataTypeInfo::AssignImpl<T>,
                &DataTypeInfo::ConstructImpl<T>,
                &DataTypeInfo::DestructImpl<T>,
                ListInterfaceIdentifier<T>::IsListInterface ? &ListDataTypeInfoInstanceProvider<T>::mListDataTypeInfo : 0
            };

            // ------------------------------------------------------------------------
            DataItemTypeId DataTypeInfo::ListItemTypeId() const
            {
                return (mListDataTypeInfo != 0) ? mListDataTypeInfo->ListItemTypeId() : COURIER_INVALID_DATA_ITEM_TYPE_ID();
            }
        }
    }
    /** provides type identity for the given type
    @param T any type */
    template<typename T> struct DataItemType {
        /** remove const and volatile qualifiers from type */
        typedef typename Internal::PureType<T>::Type UnqualifiedType;


        /** return the type identity */
        static inline DataItemTypeId Ident() {
            return COURIER_DATA_ITEM_TYPENAME_ID(UnqualifiedType);
        }
    };

    /** define DataItemType<InvalidType> as InvalidDataItemType */
    typedef DataItemType<Internal::DataBinding::InvalidType> InvalidDataItemType;
    //@}
}   // namespace

#include <Courier/DataBinding/DataItemValue.h>

#endif // Courier_DataBinding_DataItemType_h
