//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_DataItemListFragment_h)
#define Courier_DataBinding_DataItemListFragment_h

#include <Courier/Util/Limits.h>
#include <Courier/DataBinding/DataItemList.h>

namespace Courier {

    /** Represents a fragment of a list.
        DataItemListFragment stores a subset of the owning list. The subset is defined
        by the start index and the fragment capacity. The fragment is filled with 
        items of the owning list.
        @tparam T the list item type
        @tparam fragmentSize the size of the fragment - i.e. the number of list items 
                             that can be stored in the fragment */
    template<typename T, FeatStd::SizeType fragmentSize> 
    class DataItemListFragment : public DataItemList<T, fragmentSize> {
        typedef DataItemList<T, fragmentSize> Base;
    private:
            COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);
    public:
            static const FeatStd::SizeType cFragmentSize = fragmentSize;

            DataItemListFragment() :
                Base(),
                mListCapacity(ListInterfaceBase::UnknownQuantity()),
                mListItemCount(ListInterfaceBase::UnknownQuantity()),
                mStartIndex(0)
            {
            }

            DataItemListFragment(const DataItemListFragment& other) :
                Base(other),
                mListCapacity(other.mListCapacity),
                mListItemCount(other.mListItemCount),
                mStartIndex(other.mStartIndex)
            {
            }

            DataItemListFragment& operator=(const DataItemListFragment& other)
            {
                Base::operator=(other);
                mListCapacity = other.mListCapacity;
                mListItemCount = other.mListItemCount;
                mStartIndex = other.mStartIndex;
                return *this;
            }

            virtual ~DataItemListFragment()
            {
            }

            /**
            <summary>Initializes the fragment at the given start index. Previous contained items will be cleared.</summary>
            <param name="index">Zero-based start index of the fragment within the list.</param>
            */
            void InitFragment(FeatStd::SizeType startIndex)
            {
                mStartIndex = startIndex;
                this->Clear();
            }

            /**
                <summary>Gets the fragment start index.</summary>
                <returns>fragment start index.</returns>
             */
            virtual FeatStd::SizeType FragmentStartIndex() const override
            {
                return mStartIndex;
            }

            /**
                <summary>Gets the fragment capacity.</summary>
                <returns>the capacity of the list fragment.</returns>
             */
            FeatStd::SizeType FragmentCapacity() const
            {
                return Base::Capacity();
            }

            /**
                <summary>Gets the fragment item count.</summary>
                <returns>Number of list items in the fragment.</returns>
             */
            virtual FeatStd::SizeType FragmentItemCount() const override
            {
                return Base::Count();
            }

            /**
                <summary>Gets the capacity of the list the fragment read items from.</summary>
                <returns>capacity of the list owning the fragment.</returns>
             */
            virtual FeatStd::SizeType Capacity() const override
            {
                return mListCapacity;
            }

            /**
                <summary>Gets the count of items in the list the fragment read items from.</summary>
                <returns>count of items in the list the fragment read items from.</returns>
             */
            virtual FeatStd::SizeType Count() const override
            {
                return mListItemCount;
            }

            /**
                <summary>
                    Returns the item with the given list index list index is based on the owning list index.
                </summary>
                <param name="index">Zero-based list index.</param>
                <returns>The item with the specified index or an invalid DataItemValue.</returns>
             */
            virtual DataItemValue Item(FeatStd::SizeType index) const override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    return DataItemValue();
                }
                return Base::Item(LocalIndex(index));
            }

            virtual bool Change(FeatStd::SizeType index, const DataItemValue &value) override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                return Base::Change(LocalIndex(index), value);
            }

            virtual bool Add(FeatStd::SizeType index, const DataItemValue &value) override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                return Base::Add(LocalIndex(index), value);
            }

            virtual bool Remove(FeatStd::SizeType index) override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                return Base::Remove(LocalIndex(index));
            }

            virtual bool Move(FeatStd::SizeType dst, FeatStd::SizeType src) override
            {
                if (!Contains(dst)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", dst, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                if (!Contains(src)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", src, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                return Base::Move(LocalIndex(dst), LocalIndex(src));
            }

            virtual bool IsModified(FeatStd::SizeType index) const override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    return false;
                }
                return Base::IsModified(LocalIndex(index));
            }

            virtual void SetModified(FeatStd::Optional<FeatStd::SizeType> index) override
            {
                if (index.IsSet()) {
                    if (!Contains(*index)) {
                        COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", *index, FragmentStartIndex(), this->FragmentEndIndex());
                    }
                    else {
                        Base::SetModified(FeatStd::Optional<FeatStd::SizeType>(LocalIndex(*index)));
                    }
                }
                else {
                    Base::SetModified(index);
                }
            }

            virtual void ResetModified(FeatStd::Optional<FeatStd::SizeType> index) override
            {
                if (index.IsSet()) {
                    if (!Contains(*index)) {
                        COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", *index, FragmentStartIndex(), this->FragmentEndIndex());
                    }
                    else {
                        Base::ResetModified(FeatStd::Optional<FeatStd::SizeType>(LocalIndex(*index)));
                    }
                }
                else {
                    Base::ResetModified(index);
                }
            }

            virtual const T& operator[](FeatStd::SizeType index) const override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    static T s_dummy;
                    return s_dummy;
                }
                return Base::operator[](LocalIndex(index));
            }

            virtual T& operator[](FeatStd::SizeType index) override
            {
                if (!Contains(index)) {
                    COURIER_LOG_FATAL("index %d is out of bound in list fragment (fragment start index = %d, fragment end index = %d)", index, FragmentStartIndex(), this->FragmentEndIndex());
                    static T s_dummy;
                    return s_dummy;
                }
                return Base::operator[](LocalIndex(index));
            }

            /**
                <summary>
                    Initializes the fragment from the given Collection class. The list fragment will be initialized
                    with. The function will copy as many list items as possible to the fragment. 
                </summary>
                <param name="collection">
                    The collection to copy the data from. The type collection must at expose a Count, Capacity, and
                    operator[] as defined by SyncListInterfaceBase .
                </param>
                <param name="startIndex">The start index.</param>
                <returns>true if copying succeeded, false otherwise.</returns>
             */
            template<typename Collection> 
            bool ReadFragment(const Collection &collection, FeatStd::SizeType startIndex)
            {
                bool ok = true;

                // store the list count and capacity
                mListItemCount = collection.Count();
                mListCapacity = collection.Capacity();

                // in case we received an invalid index, re-use the current set start index
                // and only refresh data without changing the fragment bounds
                if (startIndex >= mListItemCount) {
                    startIndex = mStartIndex;
                }

                // clear current items in fragment
                // if the collection has not enough items, adjust start index
                Base::Clear();

                if ((startIndex + FragmentCapacity()) >= mListItemCount) {
                    // if the fragment end index is beyond the list end index,
                    // adjust the start index.
                    startIndex = (mListItemCount < FragmentCapacity()) ? 0 : (mListItemCount - FragmentCapacity());
                }

                // store the start index of the fragment copy data until fragment is full or 
                // at end of collection
                mStartIndex = startIndex;
                while ((startIndex < collection.Count()) && (FragmentItemCount() < FragmentCapacity())) {
                    ok = Base::PushBack(collection[startIndex]);
                    ++startIndex;
                }
                return ok;
            }

            /**
                <summary>Updates the list item count only.</summary>
                <param name="listItemCount">Number of list items.</param>
             */
            void UpdateListItemCount(FeatStd::SizeType listItemCount)
            {
                mListItemCount = listItemCount;
            }

        private:
            FeatStd::SizeType mListCapacity;
            FeatStd::SizeType mListItemCount;
            FeatStd::SizeType mStartIndex;

            FeatStd::SizeType EndIndex() const
            {
                return mStartIndex + Base::Count();
            }

            bool Contains(FeatStd::SizeType index) const
            {
                return (index >= mStartIndex) && (index < EndIndex());
            }

            FeatStd::SizeType LocalIndex(FeatStd::SizeType globalIndex) const
            {
                return globalIndex - mStartIndex;
            }
    };

}

#endif // Courier_DataBinding_DataItemListFragment_h
