//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_DataItemList_h)
#define Courier_DataBinding_DataItemList_h

#include <Courier/Util/FixedSizeList.h>
#include <Courier/DataBinding/TypedListInterfaces.h>
#include <Courier/DataBinding/DataItemValue.h>
#include <Courier/Util/DynamicSizeList.h>
#include <Courier/Util/FixedSizeBitVector.h>
#include <Courier/Util/DynamicSizeBitVector.h>

namespace Courier {
    namespace Internal {
        template<typename T, FeatStd::SizeType maxSize>
        struct ClonableListSelector {
            typedef Internal::FixedSizeBitVector<maxSize> BitVector;
            typedef FixedSizeList<T, maxSize> ListType;
            static void ResizeBitVector(FeatStd::SizeType /*size*/, BitVector& /*bitVector*/)
            {
            }
        };

        template<typename T>
        struct ClonableListSelector<T, 0> {
            typedef DynamicSizeBitVector BitVector;
            typedef DynamicSizeList<T> ListType;
            static void ResizeBitVector(FeatStd::SizeType size, BitVector& bitVector)
            {
                bitVector.Resize(size);
            }
        };

        template<typename T, FeatStd::SizeType maxSize, bool IsDataContextItem>
        struct ClonableList
        {
            ClonableList()
            {
            }

            ClonableList(const ClonableList& other) :
                mList(other.mList)
            {
            }

            ClonableList& operator=(const ClonableList& other)
            {
                mList = other.mList;
                return *this;
            }

            typename ClonableListSelector<T, maxSize>::ListType mList;
        };

        template<typename T>
        struct IsDataContextItem
        {
            enum
            {
                Value = false
            };
        };

        template<>
        struct IsDataContextItem<DataContextItem>
        {
            enum
            {
                Value = true
            };
        };

        template<typename T>
        struct IsDataContextItemPtr
        {
            enum
            {
                Value = false
            };
        };

        template<typename T>
        struct IsDataContextItemPtr< FeatStd::MemoryManagement::SharedPointer<T> >
        {
            enum
            {
                Value = ::Courier::Internal::IsDataContextItem<T>::Value
            };
        };

        template<typename T, FeatStd::SizeType maxSize>
        struct ClonableList<T, maxSize, true>
        {
            ClonableList()
            {
            }

            ClonableList(const ClonableList& other) :
                mList(other.mList)
            {
                for (FeatStd::SizeType i = 0; i < mList.Count(); ++i) {
                    mList[i] = Internal::DataContextItemAccess::Clone(mList[i]);
                }
            }

            ClonableList& operator=(const ClonableList& other)
            {
                mList = other.mList;
                for (FeatStd::SizeType i = 0; i < mList.Count(); ++i) {
                    mList[i] = Internal::DataContextItemAccess::Clone(mList[i]);
                }
                return *this;
            }

            typename ClonableListSelector<T, maxSize>::ListType mList;
        };
    }

    /**
        <summary>Fixed size list of data items.</summary>
        <seealso cref="SyncListInterface"/>
     */
    template<typename T, FeatStd::SizeType maxSize> class DataItemList : public TypedSyncListInterface<T> {
    private:
        COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);
    public:
            static const FeatStd::SizeType cCapacity = maxSize;

            DataItemList()
            {
            }

            DataItemList(const DataItemList& other) :
                mModified(other.mModified),
                mList(other.mList)
            {
            }

            DataItemList& operator=(const DataItemList& other)
            {
                mModified = other.mModified;
                mList = other.mList;
                return *this;
            }

            virtual ~DataItemList()
            {
            }

            virtual DataItemTypeId ListItemTypeId() const override
            {
                return COURIER_DATA_ITEM_TYPENAME_ID(T);
            }

            virtual FeatStd::SizeType Count() const override
            {
                return mList.mList.Count();
            }

            virtual  FeatStd::SizeType Capacity() const override
            {
                return cCapacity;
            }

            virtual bool EndOfList(FeatStd::SizeType index) const override
            {
                return index >= mList.mList.Count();
            }

            virtual bool IsModified(FeatStd::SizeType index) const override
            {
                if (index >= mList.mList.Count()) {
                    COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", index, mList.mList.Count());
                    return false;
                }
                return mModified.Test(index);
            }

            virtual void SetModified(FeatStd::Optional<FeatStd::SizeType> index) override
            {
                if (index.IsSet()) {
                    if (*index >= mList.mList.Count()) {
                        COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", *index, mList.mList.Count());
                    }
                    else {
                        mModified.Set(*index);
                    }
                }
                else {
                    mModified.Set();
                }
            }

            virtual void ResetModified(FeatStd::Optional<FeatStd::SizeType> index) override
            {
                if (index.IsSet()) {
                    if (*index >= mList.mList.Count()) {
                        COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", *index, mList.mList.Count());
                    }
                    else {
                        mModified.Reset(*index);
                    }
                }
                else {
                    mModified.Reset();
                }
            }

            virtual DataItemValue Item(FeatStd::SizeType index) const override
            {
                if (index >= mList.mList.Count()) {
                    COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", index, mList.mList.Count());
                    return DataItemValue();
                }
                return DataItemValue(&mList.mList[index]);
            }

            virtual bool Change(FeatStd::SizeType index, const DataItemValue &value) override
            {
                if (index >= mList.mList.Count()) {
                    COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", index, mList.mList.Count());
                    return false;
                }
                if (!value.CopyTo(mList.mList[index])) {
                    COURIER_LOG_FATAL("copy failed!");
                    return false;
                }
                DataItemList::SetModified(FeatStd::Optional<FeatStd::SizeType>(index));
                return true;
            }

            virtual bool Add(FeatStd::SizeType index, const DataItemValue &value) override
            {
                const T *valuePtr = value.GetValue<T>();
                bool ok = (valuePtr != 0) && mList.mList.InsertAt(index, *valuePtr);
                if (!ok) {
                    COURIER_LOG_FATAL("Add failed at %d (count = %d)", index, mList.mList.Count());
                    return false;
                }
                Internal::ClonableListSelector<T, maxSize>::ResizeBitVector(mList.mList.Count(), mModified);
                for (FeatStd::SizeType i = index; i < mList.mList.Count(); ++i) {
                    DataItemList::SetModified(FeatStd::Optional<FeatStd::SizeType>(i));
                }
                return true;
            }

            virtual bool Remove(FeatStd::SizeType index) override
            {
                bool ok = mList.mList.RemoveAt(index);
                if (!ok) {
                    COURIER_LOG_FATAL("Remove failed at %d (count = %d)", index, mList.mList.Count());
                    return false;
                }
                for (FeatStd::SizeType i = index; i < mList.mList.Count(); ++i) {
                    DataItemList::SetModified(FeatStd::Optional<FeatStd::SizeType>(i));
                }
                return true;
            }

            virtual bool Move(FeatStd::SizeType dst, FeatStd::SizeType src) override
            {
                bool ok = mList.mList.Move(dst, src);
                if (!ok) {
                    COURIER_LOG_FATAL("Move failed from %d to %d (count = %d)", src, dst, mList.mList.Count());
                    return false;
                }
                FeatStd::SizeType begin = (dst > src) ? src : dst;
                FeatStd::SizeType end = ((dst > src) ? dst : src) + 1;
                for (FeatStd::SizeType i = begin; i < end; ++i) {
                    DataItemList::SetModified(FeatStd::Optional<FeatStd::SizeType>(i));
                }
                return true;
            }

            virtual bool Clear() override
            {
                mList.mList.Clear();
                ResetModified(FeatStd::Optional<FeatStd::SizeType>());
                return true;
            }

            bool PushBack(const T &item)
            {
                bool ok = mList.mList.PushBack(item);
                if (!ok) {
                    COURIER_LOG_FATAL("PushBack failed!");
                    return false;
                }
                Internal::ClonableListSelector<T, maxSize>::ResizeBitVector(mList.mList.Count(), mModified);
                DataItemList::SetModified(FeatStd::Optional<FeatStd::SizeType>(mList.mList.Count() - 1));
                return true;
            }

            virtual const T& operator[](FeatStd::SizeType index) const
            {
                if (index >= mList.mList.Count()) {
                    COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", index, mList.mList.Count());
                    static T s_dummy;
                    return s_dummy;
                }
                return mList.mList[index];
            }

            virtual T& operator[](FeatStd::SizeType index)
            {
                if (index >= mList.mList.Count()) {
                    COURIER_LOG_FATAL("index %d is out of bound in list (count = %d)", index, mList.mList.Count());
                    static T s_dummy;
                    return s_dummy;
                }
                return mList.mList[index];
            }

        private:
            typename Internal::ClonableListSelector<T, maxSize>::BitVector mModified;
            Internal::ClonableList<T, maxSize, Internal::IsDataContextItemPtr<T>::Value > mList;
    };

//@}
}

#endif // Courier_DataBinding_DataItemList_h
