//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Util/BinarySearch.h>
#include "DataItemAccessor.h"

namespace Courier {
    namespace Internal {
        namespace DataBinding {
            // ------------------------------------------------------------------------
            DataItemValue DataItemAccessor::GetImmutableValue(const DataItemHierarchyNode *hnItem,
                const void *object,
                DataItemTypeId typeId)
            {
                bool ok;
                const DataItemHierarchyNode *hn = hnItem;

                for(;;) {                                                   // iterate bottom up to root or type matches the given object type
                    ok = Descriptor(hn).TypeId() == typeId;
                    if (ok || hn->IsRootNode()) {
                        break;
                    }
                    object = Descriptor(hn).ToConstMemberPtr(object);       // adjust to pointer to member (offset of item within parent structure)
                    hn = Parent(hn);
                }

                FEATSTD_LINT_NEXT_EXPRESSION(1963, "template and non-template ctor have different semantics and argument list")
                    return ok ? DataItemValue(Descriptor(hnItem).TypeId(), object) : DataItemValue();
            }

            // ------------------------------------------------------------------------
            void* DataItemAccessor::TraverseUp(const DataItemHierarchyNode *hn,
                DataItemTypeId typeId,
                void *object)
            {
                bool ok;
                for(;;) {
                    ok = Descriptor(hn).TypeId() == typeId;
                    if (ok || hn->IsRootNode()) {
                        break;
                    }
                    object = Descriptor(hn).ToMemberPtr(object);
                    hn = Parent(hn);
                }
                return ok ? object : 0;
            }

            // ------------------------------------------------------------------------
            DataItemValue DataItemAccessor::GetMutableValue(const DataItemHierarchyNode *hn,
                void *object,
                DataItemTypeId typeId)
            {
                object = TraverseUp(hn, typeId, object);
                return (object != 0) ? DataItemValue(Descriptor(hn).TypeId(), object, true) : DataItemValue();
            }

            // ------------------------------------------------------------------------
            bool DataItemAccessor::SetValue(const DataItemHierarchyNode *hn,
                void *object, DataItemTypeId typeId,
                const DataItemValue &value)
            {
                object = TraverseUp(hn, typeId, object);

                if (object != 0) {
                    DataItemTypeId itemTypeId = Descriptor(hn).TypeId();
                    DataItemValue dst(itemTypeId, object, true);
                    itemTypeId->Assign(dst, value);
                }
                return object != 0;
            }

            // ------------------------------------------------------------------------
            static Int ItemMapCmp(UInt32 hash, const DataItemHierarchyNode *hn)
            {
                return static_cast<Int>((hash == hn->mIdent.HashId()) ? 0 : ((hash < hn->mIdent.HashId()) ? -1 : 1));
            }

            // ------------------------------------------------------------------------
            const DataItemHierarchyNode* DataItemAccessor::Locate(UInt32 hash)
            {
                const DataItemHierarchyNode * const *hn;
                hn = BinarySearch(hash, gInfrastructure->mGlobalNodeList, gInfrastructure->mGlobalNodeListCount, &ItemMapCmp);
                return (hn != 0) ? *hn : InvalidNode();
            }
        }
    }

    using namespace Internal::DataBinding;

#if defined(b4cb5b61_43a0_4508_87d0_c6e339de27c1)
    class SyncListInterface;

    /** class DataItemAccessor stores a path reference to a data item
    DataItemAccessor uses an DataBindingInfrastructure provided by code generation
    to get information about the hierarchy of the data items. */
    class DataItemAccessor {
    public:
        /** constructs a DataItemAccessor with a reference to an invalid data item */
        DataItemAccessor() {
            mNode = Internal::DataBinding::DataItemAccessor::InvalidNode();
        }

        /** constructs a DataItemAccessor with the given itemKey
        @param itemKey the item id as generated by source code generation
        @code
        DataItemAccessor itemAcc(::ItemId::Radio::CurrentStation::NameItem);
        @endcode */
        explicit DataItemAccessor(DataItemKey itemKey) {
            mNode = Internal::DataBinding::DataItemAccessor::Node(itemKey);
        }

        /** constructs a DataItemAccessor from a hash identifier
        complexity log2(N) with N beeing the number of data item hierarchy nodes
        @param id a hash identifier */
        explicit DataItemAccessor(const Identifier &id);

        /** set the DataItemAccessor to invalid */
        void Reset() {
            mNode = Internal::DataBinding::DataItemAccessor::InvalidNode();
        }

        /** returns true if the object references a valid data item */
        bool IsValid() const {
            return mNode != Internal::DataBinding::DataItemAccessor::InvalidNode();
        }

        /** lets the object reference the data item with the given item id
        @param itemKey the item id
        @return true if a data item with the given id exists, false otherwise
        @code
        DataItemAccessor itemRef1("/Radio/CurrentStation");
        DataItemAccessor itemRef2(Generated::ItemId::CurrentStationItem);
        @endcode */
        DataItemAccessor& operator=(DataItemKey itemKey);

        /** checks if the DataItemAccessor is a sub path of the given DataItemAccessor
        i.e. /Radio/CurrentStation is a sub path of /Radio. The check will also
        succeed for /Radio checked against /Radio.
        @param itemKey the DataItemAccessor to check against
        @return true if the the the object is a sub path, false otherwise */
        bool IsSubPathOf(DataItemKey itemKey) const;

        /** returns if the referenced data item is a root item
        @code
        DataItemAccessor itemAcc(Generated::ItemId::RadioItem);
        FEATSTD_DEBUG_ASSERT(itemAcc.IsRoot());
        @endcode*/
        bool IsRoot() const {
            return mNode->IsRootNode();
        }

        /**
        <summary>returns if the referred data item is a leave node in the data item hierarchy</summary>
        <returns>true if the referred node is a leave node, false otherwise</returns>
        */
        bool IsLeaveNode() const {
            return mNode->IsLeaveNode();
        }

        /**
        <summary>returns wether the referred data item is read only</summary>
        <returns>true if the data item is read only, false otherwise.</returns>
        */
        bool IsReadOnly() const {
            return mNode->IsReadOnly();
        }

        /**
        <summary>Gets the referred node type.</summary>
        <returns>the type of the data item hierarchy node</returns>
        */
        DataItemHierarchyNodeType::Enum NodeType() const {
            return mNode->NodeType();
        }

        /** returns the DataItemTypeId of the referenced data item */
        DataItemTypeId TypeId() const {
            return Internal::DataBinding::DataItemAccessor::Descriptor(mNode).TypeId();
        }

        /** returns from the given object an immutable value object for the referenced data item
        @param object the object to retrieve the data item from
        @return a valid DataItemValue object if the object contains the data item, an invalid otherwise */
        template<typename T> inline DataItemValue GetImmutableValue(const T &object) const {
            return Internal::DataBinding::DataItemAccessor::GetImmutableValue(mNode, object);
        }

        /** returns from the given object a mutable value object for the referenced data item
        @param object the object to retrieve the data item from
        @return a valid DataItemValue object if the object contains the data item, an invalid otherwise */
        template<typename T> inline DataItemValue GetMutableValue(T &object) const {
            return Internal::DataBinding::DataItemAccessor::GetMutableValue(mNode, object);
        }

        /** Set the value in the given object
        @param object the object to receive the value
        @param value a valid DataItemValue object
        @return true if the value could be set, false otherwise */
        template<typename T> inline bool SetValue(T &object, const DataItemValue &value) const {
            return Internal::DataBinding::DataItemAccessor::SetValue(mNode, object, value);
        }

        /**
        <summary>Equality operator.</summary>
        <param name="itemAcc">The item reference.</param>
        <returns>true if the parameters are considered equivalent.</returns>
        */
        bool operator==(const DataItemAccessor &itemAcc) const {
            return mNode == itemAcc.mNode;
        }

        bool operator!=(const DataItemAccessor &itemAcc) const {
            return mNode != itemAcc.mNode;
        }

    private:
        const DataItemHierarchyNode *mNode;                     ///< the reference to the current DataItemHierarachyNode

        /** construct from a concrete node reference */
        DataItemAccessor(const DataItemHierarchyNode *hn) {
            mNode = hn;
        }

        FEATSTD_LINT_INITIALIZER_FUNCTION(::Courier::DataItemAccessor::From)
    };

    // ------------------------------------------------------------------------
    DataItemAccessor& DataItemAccessor::operator=(DataItemKey itemKey)
    {
        mNode = DataItemAccessor::Node(itemKey);
        return *this;
    }

    // ------------------------------------------------------------------------
    //bool DataItemAccessor::IsSubPathOf(const DataItemAccessor &itemAcc) const
    //{
    //    // if itemAcc hierarchy level is larger than ours, we cannot be a subpath of itemAcc
    //    if (itemAcc.mNode->mHierarchyLevel > this->mNode->mHierarchyLevel) {
    //        return false;
    //    }

    //    // traverse the hierary up until we reached the same level as itemAcc
    //    const DataItemHierarchyNode *hn = this->mNode;
    //    while(itemAcc.mNode->mHierarchyLevel < hn->mHierarchyLevel) {
    //        hn = DataItemAccessor::Parent(hn);
    //    }

    //    // if we refer to the same entry, we are a sub-path of itemAcc
    //    return hn == itemAcc.mNode;
    //}
#endif
}
