//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_DataBindingMsg_h)
#define Courier_DataBinding_DataBindingMsg_h

#include <Courier/Diagnostics/Log.h>
#include <Courier/Foundation/Component.h>
#include <Courier/Messaging/MessageReferrer.h>
#include <Courier/Util/FixedSizeBitVector.h>
#include <Courier/DataBinding/DataBindingMsgs.h>
#include <Courier/DataBinding/BindingSourceRevisionStore.h>
#include <Courier/DataBinding/ChangeSet.h>

namespace Courier {
/// @addtogroup COURIER_DATABINDING
/// @{
    /**
        <summary>
            Data item container is a helper function for model implementation. the DataItemContainer holds all data
            of a binding source and provides functions to update this data to the other system components (controller
            / view)
        </summary>
     */
    template<typename T> class DataItemContainer {
        public:
            typedef Internal::FixedSizeBitVector<T::cHierarchyNodeCount> BitVector;
            typedef typename T::Data ValueType;

            DataItemContainer() : mValue(){
            }

            DataItemContainer(const ValueType &value) : mValue(value) {
            }

            /**
                <summary>Sends an update of the current data to the controller / view.</summary>
                <param name="bulk">(optional) instruct the update receiver to refresh all data.</param>
                <returns>true if sending the data succeeds, false if it fails.</returns>
             */
            bool SendUpdate(bool bulk = false);

            /**
                <summary>Query if data item with 'itemKey' has been modified.</summary>
                <param name="itemKey">The item key.</param>
                <returns>true if item modified, false if not.</returns>
             */
            bool IsItemModified(DataItemKey itemKey) const {
                return CheckItemKey(itemKey) && mModifiedBits.Test(ItemKeyToLocalIndex(itemKey));
            }

            /**
                <summary>Mark item with the given key as modified.</summary>
                <param name="itemKey">The item key.</param>
             */
            bool MarkItemModified(DataItemKey itemKey) {
                bool ok = CheckItemKey(itemKey);
                if (ok) {
                    mModifiedBits.Set(ItemKeyToLocalIndex(itemKey));
                }
                return ok;
            }

            /**
                <summary>Mark all items as modified.</summary>
             */
            void MarkAllItemsModified() {
                mModifiedBits.Set();
            }
			
			/**
			    HMIBase modified: functionality added
		     */
            void ResetAllItemsModified() 
            {
               mModifiedBits.Reset();
            }

            /**
                <summary>Returns a const reference to the binding source data.</summary>
                <returns>instance of the binding source data type.</returns>
             */
            inline const ValueType& operator*() const {
                return mValue;
            }

            /**
                <summary>
                    Returns a mutable reference to the binding source data. If data is changed, the application is
                    responsible to set the modified bits accordingly (see MarkItemModified)
                </summary>
                <returns>instance of the binding source data type.</returns>
             */
            inline ValueType& operator*() {
                return mValue;
            }

            /**
                <summary>Gets the item key first modified item.</summary>
                <returns>The first modified item key or cInvalidDataItemKey if no modified items exist.</returns>
             */
            DataItemKey GetFirstModifiedItem() {
                return NextModifiedItem(cInvalidDataItemKey);
            }

            /**
                <summary>
                    returns the next item which modified bit is set. Pass cInvalidDataItemKey with in initial call to
                    receive the first modified item.
                </summary>
                <param name="itemKey">
                    The item key returned from the previous call or cInvalidDataItemKey at the initial call.
                </param>
                <returns>
                    data item key of next modified item or cInvalidDataItemKey if iteration passed the last modified item.
                </returns>
             */
            DataItemKey NextModifiedItem(DataItemKey itemKey) const {
                using Internal::DataBinding::DataItemAccessor;

                FeatStd::SizeType idx;
                if (itemKey == cInvalidDataItemKey) {   // initial invocation
                    idx = mModifiedBits.GetFirstSetBit();
                }
                else {                                  // subssequent invocation - request next bit
                    FEATSTD_DEBUG_ASSERT(DataItemAccessor::GetBindingSourceIndex(itemKey) == T::cBindingSourceIndex);
                    idx = mModifiedBits.GetNextSetBit(DataItemAccessor::GetLocalNodeIndex(itemKey));
                }
                if (idx != mModifiedBits.EndIndex()) {  // map the binding source local index to the corresponding item key
                    itemKey =  DataItemAccessor::ItemKey(T::cBindingSourceIndex, HierarchyNodeIndex(idx));
                }
                else {
                    itemKey = cInvalidDataItemKey;      // return invalid item key if no more modified items exist
                }
                return itemKey;
            }

            /**
                <summary>Query if this object has modified items.</summary>
                <returns>true if modified items, false if not.</returns>
             */
            bool HasModifiedItems() const {
                return NextModifiedItem(cInvalidDataItemKey) != cInvalidDataItemKey;
            }

            /**
                <summary>
                    Gets a mutable data item value. Requesting a mutable value automatically will set the modified
                    flag of for the corresponding data item.
                </summary>
                <param name="itemKey">The key of the data item.</param>
                <returns>The mutable value. Check DataItemValue::IsValid() if succeeded.</returns>
             */
            DataItemValue GetMutableValue(DataItemKey itemKey) {
                DataItemValue div = Internal::DataBinding::DataItemAccessor::GetMutableValue(itemKey, mValue);
                if (div.IsValid()) {
                    MarkItemModified(itemKey);
                }
                return div;
            }

            /**
                <summary>Gets an immutable value of for the given data item key.</summary>
                <param name="itemKey">The data item key.</param>
                <returns>The immutable value, check DataItemValue::IsValid if request succeeded.</returns>
             */
            DataItemValue GetImmutableValue(DataItemKey itemKey) {
                return Internal::DataBinding::DataItemAccessor::GetImmutableValue(itemKey, mValue);
            }

            /**
                <summary>Sets the given value for the specified data item key.</summary>
                <param name="itemKey">The item key.</param>
                <param name="value">The value.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool SetValue(DataItemKey itemKey, const DataItemValue &value) {
                DataItemValue div = Internal::DataBinding::DataItemAccessor::GetMutableValue(itemKey, mValue);
                bool ok = div.IsValid() && div.SetValue(value);
                if (ok) {
                    MarkItemModified(itemKey);
                }
                return ok;
            }

            UInt16 DataRevision() const {
                return BindingSourceRevisionStore::GetRevision(T::cDataItemKey);
            }

        private:
            /** the index of the associated binding source */
            static const BindingSourceIndex cBindingSourceIndex = T::cBindingSourceIndex;
            BitVector mModifiedBits;                    ///< bit vector - each data item has a bit expressing if the data item has been modified
            ValueType mValue;                           ///< message payload - the data items

            /**
                <summary>convert an item key to the binding source local index.</summary>
                <param name="itemKey">The item key.</param>
                <returns>local index.</returns>
             */
            static inline HierarchyNodeIndex ItemKeyToLocalIndex(DataItemKey itemKey) {
                return Internal::DataBinding::DataItemAccessor::GetLocalNodeIndex(itemKey);
            }

            /**
                <summary>Check item key - item keys passed must be sub keys of the associated binding source.</summary>
                <param name="itemKey">The item key.</param>
                <returns>true if the item key is part of the binding source, false if not.</returns>
             */
            inline bool CheckItemKey(DataItemKey itemKey) const {
                return Internal::DataBinding::DataItemAccessor::GetBindingSourceIndex(itemKey) == cBindingSourceIndex;
            }

            /**
                <summary>Test the modified flag of the given item key.</summary>
                <param name="itemKey">The item key.</param>
                <returns>true if the item has been modified, false if not.</returns>
             */
            bool ItemFlag(DataItemKey itemKey) const {
                return CheckItemKey(itemKey) && mModifiedBits.Test(ItemKeyToLocalIndex(itemKey));
            }

            COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

            friend struct UnitTest::DataItemContainerTestAccessor< DataItemContainer<T> >;
    };

    // ========================================================================

    /** Abstract base class for all data item messags. Data item messages are used
        to communication data asynchronously to other components. */
    class AbstractDataItemMsg : public Internal::DataItemMsgBase {
        public:
            AbstractDataItemMsg(DataItemKey itemKey) : Internal::DataItemMsgBase(itemKey, 0) {
            }

            /** read a data item from the message payload
                @param itemKey the reference of the data item
                @return valid DataItemValue with a reference to the requested data, invalid
                        DataItemValue if the data item is not in the message */
            virtual DataItemValue GetItemValue(DataItemKey itemKey) const = 0;

            /**
                <summary>
                    Determines if the specified data item requires update - i.e. has been modified. The function will
                    yield true if the modified flag of the item itself or one of the parent items is set.
                </summary>
                <param name="itemKey">The item key.</param>
                <returns>
                    true if the item has been modified, false if the item and none of the parent items has the
                    modified bit set.
                </returns>
             */
            virtual bool ItemRequiresUpdate(DataItemKey itemKey) const = 0;
    };

    /** Concrete data item message implementation.
        Data item messages may carry different payloads depending on the DataItemAccessor associated
        with the message. For each BindingSource an according data item message will be generated
        by code generation.
        @param T the payload type (ie. BindingSource data structure) */
    template<typename T> class DataItemMsg : public AbstractDataItemMsg {
        public:
            typedef typename T::Data ValueType;

            /**
                <summary>
                    Constructs a DataItemMsg from a ValueType. The value type is typically the generated
                    BindingSource data structure.
                </summary>
                <param name="value">The value type.</param>
             */
            DataItemMsg(const ValueType &value) : AbstractDataItemMsg(T::cDataItemKey), mContainer(value) {
                UpdateDataRevision();
            }

            /**
                <summary>Constructs a DataItemMsg from a DataItemContainer.</summary>
                <param name="payload">The container with the data payload.</param>
             */
            DataItemMsg(const DataItemContainer<T> &payload) : AbstractDataItemMsg(T::cDataItemKey), mContainer(payload) {
                UpdateDataRevision();
            }

            /**
                <summary>Gets the data payload of the message.</summary>
                <returns>Data payload of the message.</returns>
             */
            inline const ValueType& Value() const {
                return mContainer.Value();
            }

            /**
                <summary>Gets the data payload of the message.</summary>
                <returns>Data payload of the message.</returns>
             */
            inline ValueType& Value() {
                return *mContainer;
            }

            /**
                <summary>Sets the data payload of the message.</summary>
                <param name="value">The data payload value.</param>
             */
            inline void SetValue(const ValueType &value) {
                *mContainer = value;
            }

            /**
                <summary>Mark all data items modified.</summary>
             */
            inline void MarkAllItemsModified() {
                mContainer.MarkAllItemsModified();
            }

            /**
                <summary>Mark the specified item modified.</summary>
                <param name="itemKey">The item key of the modified data item.</param>
             */
            inline void MarkItemModified(DataItemKey itemKey) {
                mContainer.MarkItemModified(itemKey);
            }

            /** retrieves a data item from the message payload
                itemKey must be a sub path of the ItemRef associated with the message
                @param itemKey the data item key
                @return an immutable DataItemValue referring to the specified data item, invalid DataItem if the
                        item is not contained in the message */
            virtual DataItemValue GetItemValue(DataItemKey itemKey) const {
                return Internal::DataBinding::DataItemAccessor::GetImmutableValue(itemKey, *mContainer);
            }

            virtual bool ItemRequiresUpdate(DataItemKey itemKey) const {
                using Internal::DataBinding::DataItemAccessor;
                bool update;
                for (;;) {
                    // break if modified bit is set
                    update = mContainer.IsItemModified(itemKey);
                    if (update) {
                        break;
                    }
                    // iterate up the type hierarchy until the root item
                    itemKey = DataItemAccessor::Parent(itemKey);
                    if (itemKey == cInvalidDataItemKey) {
                        // root item of the type hierachy reached -> no update required
                        break;
                    }
                }
                return update;
            }

    protected:
#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
        virtual ::FeatStd::UInt32 AppendTo(::FeatStd::StringBuffer& stringBuffer) const {
            ::FeatStd::UInt32 tcharCount = 0;
            tcharCount += stringBuffer.Append("::Courier::DataItemMsg { ");
            tcharCount += Base::AppendTo(stringBuffer);
            tcharCount += stringBuffer.Append(", Container = ");
            tcharCount += stringBuffer.AppendObject(mContainer);
            tcharCount += stringBuffer.Append(" } ");
            return tcharCount;
        }
#endif
        private:
            DataItemContainer<T> mContainer;

            void UpdateDataRevision() {
                UInt16 revision;
                if (mContainer.HasModifiedItems()) {
                    // if the container has modified items, increment the data revision number
                    revision = BindingSourceRevisionStore::IncRevision(GetItemKey());
                }
                else {
                    // otherwise get current revision number
                    revision = BindingSourceRevisionStore::GetRevision(GetItemKey());
                }
                // set the revision number in the message object
                SetDataRevision(revision);
            }
    };

    // ------------------------------------------------------------------------
    template<typename T> bool DataItemContainer<T>::SendUpdate(bool bulk)
    {
        if (bulk) {
            mModifiedBits.Set();
        }
        DataItemMsg<T> *msg = COURIER_MESSAGE_NEW(DataItemMsg<T>)(*this);
        bool ok = msg != 0;
        if (ok) {
            COURIER_LOG_DEBUG("BS=%u (rev %u)", msg->GetItemKey(), msg->GetDataRevision());
            ok = msg->Post();
        }
        if (ok) {
            DataItemKey itemKey = GetFirstModifiedItem();
            while (itemKey != cInvalidDataItemKey) {
                DataItemValue item = Internal::DataBinding::DataItemAccessor::GetMutableValue(itemKey, mValue);
                if (item.IsValid()) {
                    DataItemTypeId typeId = item.TypeId();
                    if (0 != typeId) {
                        const Courier::Internal::DataBinding::ListDataTypeInfo* listDataTypeInfo = typeId->GetListDataTypeInfo();
                        if (0 != listDataTypeInfo) {
                            Courier::Internal::DataBinding::ListInterfaceType::Enum interfaceType = listDataTypeInfo->InterfaceType();
                            if (Courier::Internal::DataBinding::ListInterfaceType::Synchronous == interfaceType) {
                                SyncListInterface* slif = FeatStd::Internal::PointerToPointer<SyncListInterface*>(item.MutablePointer());
                                if (0 != slif) {
                                    slif->ResetModified(FeatStd::Optional<FeatStd::SizeType>());
                                }
                            }
                        }
                    }
                }
                itemKey = NextModifiedItem(itemKey);
            }
            mModifiedBits.Reset();
        }
        return ok;
    }

    // ========================================================================

    /**
        <summary>
            Update model message is responsible to carry requests to modify or update data back from view /
            controller to the model component. Each UpdateModelMsg may contain multiple Courier::Request objects for
            multiple binding sources. Internally UpdateModelMsg uses Courier::Internal::DataBinding::ChangeSet to store
            the request information.
            The size of the buffer Courier::Internal::DataBinding::ChangeSet is using to store request data can be
            controlled with COURIER_CHANGE_SET_BUFFER_SIZE.
        </summary>
        <seealso cref="Internal::InternalUpdateModelMsg"/>
     */
    class UpdateModelMsg : public Internal::InternalUpdateModelMsg {
        public:
            /**
                <summary>Gets the number of request carried by the message.</summary>
                <returns>Number of requests.</returns>
             */
            UInt32 RequestCount() const {
                return mChangeSet.Count();
            }

            /**
                <summary>Gets a request object from the message.</summary>
                <param name="reqIndex">Zero-based index of the request.</param>
                <returns>The request object - invalid object if index is out of bounds.</returns>
             */
            Request GetRequest(FeatStd::SizeType reqIndex) const {
                return mChangeSet.GetChangeRequest(reqIndex);
            }

            /**
                <summary>Getter for the change set.</summary>
                <returns>The change set.</returns>
             */
            Internal::DataBinding::ChangeSet& GetChangeSet() {
                return mChangeSet;
            }

        private:
            /** the Courier::ChangeSet stores the request data */
            Internal::DataBinding::ChangeSet mChangeSet;


            friend class Internal::AsyncModelBindingSource;
    };

    /** import release message from internal namespace to make it public accessable */
    typedef Internal::ReleaseDataItemMsgBase ReleaseDataItemMsg;

//@}
}   // namespace

#endif // Courier_DataBinding_DataBindingMsg_h
