//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_AsyncWidgetPropertyBindings_h)
#define Courier_DataBinding_AsyncWidgetPropertyBindings_h

#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <Courier/DataBinding/AsyncBindingBase.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/Infrastructure.h>
#include <Courier/DataBinding/PropertyBindingData.h>
#include <Courier/DataBinding/TypeConverter.h>
#include <Courier/DataBinding/SharedAsyncListInterface.h>
#include <Courier/Diagnostics/ContextCheck.h>
#include <Courier/Util/FixedSizeList.h>


// in order to use short names for the binding object class names,
// all binding objects are defined in Async name space.
namespace Courier { namespace Internal {

        /// @addtogroup COURIER_DATABINDING
        /// @{

        namespace DataBinding { namespace Async {
            /**
            <summary>
            Base class for asynchronous binding source objects.
            </summary>
            */
            class AsyncWidgetPropertyBinding : public Internal::AsyncBindingBase {
            public:
                FEATSTD_LINT_CURRENT_SCOPE(1748, "the non-virtual base class 'Binding' has no members")
                virtual ~AsyncWidgetPropertyBinding();

                virtual const AsyncWidgetPropertyBinding* ToAsyncWidgetPropertyBinding() const override {
                    return this;
                }

                virtual const Char* GetTargetName() const override {
                    return GetData().GetPropertyName();
                }

                virtual bool SetTargetValueFrom(const AsyncBinding &binding) override;
                virtual bool SetTargetValue(const DataItemValue &value) override;
                virtual bool GetTargetValue(const DataItemValue &value) const override;
                virtual bool Init(Internal::AsyncModelBindingSource *bindingSource,
                    DataItemKey itemKey,
                    const Internal::PropertyBindingData::PropertyMetaInfo *metaInfo,
                    FrameworkWidget *widget,
                    bool enableTypeConversion,
                    const DataItemValue &defaultValue) = 0;

            protected:
                AsyncWidgetPropertyBinding();

                FEATSTD_LINT_NEXT_EXPRESSION(1511, "the hidden non-virtual member function Setup is unlikely to be mistaken with the this member function Setup functions (different number of arguments)")
                    bool Setup(Internal::AsyncModelBindingSource *bindingSource,
                    DataItemKey itemKey,
                    const Internal::PropertyBindingData::PropertyMetaInfo *metaInfo,
                    FrameworkWidget *widget);

                FEATSTD_LINT_NEXT_EXPRESSION(1511, "the hidden non-virtual member function InternalInit is unlikely to be mistaken with the this member function InternalInit functions (different number of arguments)")
                    bool InternalInit(const Internal::PropertyBindingData::PropertyMetaInfo *propertyMetaInfo, FrameworkWidget *widget);

                Internal::PropertyBindingData& GetData() const {
                    return const_cast<Internal::PropertyBindingData&>(mData);
                }

            private:
                Internal::PropertyBindingData mData;

                FEATSTD_MAKE_CLASS_UNCOPYABLE(AsyncWidgetPropertyBinding);
                friend struct ListAccessor;
                friend class Internal::UpdateFlagGuard;
            };

            // ========================================================================


            // ========================================================================

            /** defines a binding between a elementary data item in the model component and a widget property */
            class ElementaryBinding : public AsyncWidgetPropertyBinding {
            public:
                FEATSTD_LINT_CURRENT_SCOPE(1748, "the non-virtual base class 'Binding' has no members")
                    virtual bool Init(Internal::AsyncModelBindingSource *bindingSource,
                    DataItemKey itemKey,
                    const Internal::PropertyBindingData::PropertyMetaInfo *metaInfo,
                    FrameworkWidget *widget,
                    bool enableTypeConversion,
                    const DataItemValue &defaultValue);

                virtual void SetTargetModifiedTag();
            };

            // ========================================================================

            /** defines a binding between a list data item in the model component and a widget property */
            class ListBinding : public AsyncWidgetPropertyBinding, public SharedAsyncListInterface {
            public:
                FEATSTD_LINT_CURRENT_SCOPE(1748, "the non-virtual base class 'Binding' has no members")
                    ListBinding();

                virtual bool Init(Internal::AsyncModelBindingSource *bindingSource,
                    DataItemKey itemKey,
                    const Internal::PropertyBindingData::PropertyMetaInfo *metaInfo,
                    FrameworkWidget *widget,
                    bool enableTypeConversion,
                    const DataItemValue &defaultValue);

                /**
                <summary>On ehte event of an update from the model, ObModelValueUpdate is reading the
                value from the DataItemMsg and handles it accordingly.</summary>
                <returns>true if the update succeeded, false if it failed.</returns>
                */
                virtual bool OnDataItemMsg(const AbstractDataItemMsg *msg) override;
                virtual void ProcessListEvent(const ListEvent &listEvent) override;
                virtual bool OnTargetInitialized() override;

                virtual DataItemTypeId ListItemTypeId() const override;
                virtual FeatStd::SizeType Count() const override;
                virtual FeatStd::SizeType Capacity() const override;
                virtual bool EndOfList(FeatStd::SizeType index) const override;
                virtual FeatStd::SizeType FragmentStartIndex() const override;
                virtual FeatStd::SizeType FragmentItemCount() const override;
                virtual bool Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool updateOnly) override;
                virtual bool RequestChange(FeatStd::SizeType idx, const DataItemValue &item) override;
                virtual bool RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item) override;
                virtual bool RequestAdd(FeatStd::SizeType idx, const DataItemValue &item) override;
                virtual bool RequestRemove(FeatStd::SizeType idx) override;
                virtual bool RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src) override;
                virtual bool RequestClear() override;
                virtual bool RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems) override;
                virtual UInt32 Flags() const override;

            protected:
                virtual void OnReferrerUpdate(bool addReferrer) override;

            private:
                BitField mRefCount                      : 8;    ///< reference counter of the binding object
                BitField mAllowEdit                     : 1;    ///< cached flag if list item editing is allowed
                BitField mAllowAdd                      : 1;    ///< cached flag if adding of list items is allowed
                BitField mAllowRemove                   : 1;    ///< cached flag if removing of list items is allowed
                BitField mDelayItemRequestProcessing    : 1;    ///< true if RequestItem request processing shall be delayed or immeditly executed
                FeatStd::SizeType mRequestItemStartIndex;               ///< current start index for item requests
                FeatStd::SizeType mRequestItemEndIndex;                 ///< current end index for item requests
                FeatStd::SizeType mRequestedItemStartIndex;             ///< start of requested items sent to the model
                FeatStd::SizeType mRequestedItemEndIndex;               ///< end of requested items sent to the model
                SharedPointer<SharedAsyncListInterface> mDefaultValue;  ///< default value for the binding

                static bool mInItemRequestProcessing;       ///< guard to detect new item Request during request processing
                static bool mIsEventProcessing;             ///< guard to detect recursive invocations of SendListEvent
                static FixedSizeList<ListEvent, cListEventQueueSize>& EventQueue(); ///< fifo queue to queue list events in case of recursive invocation

                /**
                <summary>Gets the list synchronous list interface from the current DataItemMsg.</summary>
                <returns>null if it fails / no current message is available, else a list of.</returns>
                */
                const SyncListInterface* List() const;

                DataItemValue ListItem(FeatStd::SizeType index) const;
                void ProcessPendingListItemRequests(bool updateOnly);

                void ClearPendingItemRequest();

                bool HasPendingItemRequest() const {
                    COURIER_CHECK_SCOPE_AFFINITY();
                    return mRequestItemStartIndex != ListInterfaceBase::InvalidListIndex();
                }

                /**
                <summary>Sends a list event to the widget property (via capability object).</summary>
                <param name="listEvent">The list event.</param>
                <returns>true if it succeeds, false if it fails.</returns>
                */
                bool SendListEvent(const ListEvent &listEvent);

                bool PostListRequest(ListRequestType::Enum reqType,
                    FeatStd::SizeType newIndex = ListInterfaceBase::InvalidListIndex(),
                    FeatStd::SizeType oldIndex = ListInterfaceBase::InvalidListIndex(),
                    const DataItemValue &itemValue = DataItemValue()) const;
            };

            typedef Internal::ObjectListFunc<Internal::AsyncBinding, Internal::AsyncBindingSourceListAccessor> BindingObjectList;
            typedef Internal::ObjectListIterator<Internal::AsyncBinding, Internal::AsyncBindingSourceListAccessor> BindingObjectListIt;

        }} ///@}
    }}  // namespace

#endif // !defined(Courier_DataBinding_AsyncWidgetPropertyBindings_h)
