//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncWidgetPropertyBindings.h"

#include <Courier/Platform/Memory.h>
#include <Courier/Diagnostics/Log.h>
#include "BindableProperty.h"
#include "AsyncModelBindingSource.h"

/** \ingroup COURIER_DATABINDING */

namespace Courier { namespace Internal {

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

    using Candera::MetaInfo::WidgetPropertyMetaInfo;

    namespace DataBinding { namespace Async {

    // ------------------------------------------------------------------------
    AsyncWidgetPropertyBinding::AsyncWidgetPropertyBinding()
    {
    }

    // ------------------------------------------------------------------------
    AsyncWidgetPropertyBinding::~AsyncWidgetPropertyBinding()
    {
    }

    // ------------------------------------------------------------------------
    bool AsyncWidgetPropertyBinding::Setup(AsyncModelBindingSource *bindingSource, DataItemKey itemKey,
        const PropertyBindingData::PropertyMetaInfo *metaInfo, FrameworkWidget *widget)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (IsInitialized()) {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        bool ok = InternalInit(metaInfo, widget);
        if (ok) {
            ok = AsyncBindingBase::Setup(bindingSource, itemKey);
        }
        SetInitialized(ok);
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncWidgetPropertyBinding::InternalInit(const Internal::PropertyBindingData::PropertyMetaInfo *propertyMetaInfo, FrameworkWidget *widget)
    {
        bool ok = (propertyMetaInfo != 0) && (GetData().GetMeta() == 0);
        if (ok) {
            GetData().SetMeta(propertyMetaInfo);
            ok = BindingBase::InternalInit(widget);
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncWidgetPropertyBinding::SetTargetValueFrom(const AsyncBinding &binding)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const AsyncWidgetPropertyBinding* asyncWidgetPropertyBinding = binding.ToAsyncWidgetPropertyBinding();
        if (0 == asyncWidgetPropertyBinding) {
            return false;
        }

        const BindablePropertyCapability* sourceCap = asyncWidgetPropertyBinding->GetData().GetCapabilityObject();
        if (0 == sourceCap) {
            return false;
        }
        const BindablePropertyCapability* targetCap = mData.GetCapabilityObject();
        if (0 == targetCap) {
            return false;
        }

        FEATSTD_LINT_NEXT_EXPRESSION(1774, "type is known")
        return targetCap->SetFrom(static_cast<FrameworkWidget*>(GetTargetHost()), static_cast<FrameworkWidget*>(asyncWidgetPropertyBinding->GetTargetHost()), *sourceCap);
    }

    // ------------------------------------------------------------------------
    FEATSTD_LINT_CURRENT_SCOPE(1788, "Variable 'updateGuard' is referenced only by its constructor or destructor")
    bool AsyncWidgetPropertyBinding::SetTargetValue(const DataItemValue &value)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const BindablePropertyCapability *cap = mData.GetCapabilityObject();
        bool ok = cap != 0;

        UpdateFlagGuard updateGuard(this, &ok);
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type is known")
        ok = ok && cap->Set(static_cast<FrameworkWidget*>(GetTargetHost()), value, GetConvertFn());
        SetTargetInitialized(true);
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncWidgetPropertyBinding::GetTargetValue(const DataItemValue &value) const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const BindablePropertyCapability *cap = mData.GetCapabilityObject();
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type is known")
        return (cap != 0) && cap->Get(static_cast<FrameworkWidget*>(GetTargetHost()), value, GetConvertFn());
    }

    // ------------------------------------------------------------------------
    bool ElementaryBinding::Init(Internal::AsyncModelBindingSource *bindingSource,
                                  DataItemKey itemKey,
                                  const Candera::MetaInfo::WidgetPropertyMetaInfo *metaInfo,
                                  FrameworkWidget *widget,
                                  bool enableTypeConversion,
                                  const DataItemValue &defaultValue)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (!Setup(bindingSource, itemKey, metaInfo, widget)) {
            return false;
        }
        // get Capability object of the property
        const BindablePropertyCapability *cap = GetData().GetCapabilityObject();

        // can only happen if binding is on a non-bindable property
        bool ok = cap != 0;
        if (ok) {
            const DataItemDescriptor &itemDesc = DataItemAccessor::Descriptor(GetItemKey());
            DataItemTypeId dataItemTypeId = itemDesc.TypeId();

            // if the capability object does not accept the data item type
            if ((cap->TypeId() != dataItemTypeId) && enableTypeConversion) {
                ok = SetupTypeConverter(dataItemTypeId, cap->TypeId()); // try to setup a type converter
                if (!ok) {
                    COURIER_LOG_WARN("failed to create binding from ItemKey %u to %s - no valid type converter",
                                      UInt32(itemKey), metaInfo->GetName());
                }
            }

            // the widget property value will be set in OnWidgetInitialized to
            // prevent that subsequent property initializations of the asset loader
            // overwrite the model value
            if(ok && defaultValue.IsValid()) {
                (void) SetTargetValue(defaultValue);
            }
        }

        return ok;
    }


    // ------------------------------------------------------------------------
    void ElementaryBinding::SetTargetModifiedTag()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if ((!IsUpdating()) && (!IsReadOnly())) {
            AsyncModelBindingSource *bs = GetBindingSource();
            (void) UpdateModifiedFlag((bs != 0) && bs->AddChangeRecord(*this));
        }
    }

    // ========================================================================

    bool ListBinding::mIsEventProcessing = false;
    bool ListBinding::mInItemRequestProcessing = false;

    static UInt32 gTypeConversionBuffer[(AsyncBindingBase::cTypeConversionBufferSize + 3) / 4];

    // ------------------------------------------------------------------------
    ListBinding::ListBinding() : mRefCount(0),
                                 mAllowEdit(false), mAllowAdd(false), mAllowRemove(false), mDelayItemRequestProcessing(false),
                                 mRequestItemStartIndex(ListInterfaceBase::InvalidListIndex()),
                                 mRequestItemEndIndex(ListInterfaceBase::InvalidListIndex()),
                                 mRequestedItemStartIndex(ListInterfaceBase::InvalidListIndex()),
                                 mRequestedItemEndIndex(ListInterfaceBase::InvalidListIndex())
    {
        COURIER_CHECK_SCOPE_AFFINITY();
    }

    // ------------------------------------------------------------------------
    bool ListBinding::Init(Internal::AsyncModelBindingSource *bindingSource,
                           DataItemKey itemKey,
                           const WidgetPropertyMetaInfo *metaInfo,
                           FrameworkWidget *widget,
                           bool enableTypeConversion,
                           const DataItemValue &)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (!Setup(bindingSource, itemKey, metaInfo, widget)) {
            return false;
        }

        // get Capability object of the property
        const BindablePropertyCapability *cap = GetData().GetCapabilityObject();
        if (cap == 0) {
            // can only happen if binding is on a non-bindable property
            return false;
        }

        const DataItemDescriptor &itemDesc = DataItemAccessor::Descriptor(GetItemKey());
        DataItemTypeId dataItemTypeId = itemDesc.TypeId();
        DataItemTypeId propertyTypeId = cap->TypeId();

        // DataItemList exposes a synchronous interface / props async
        bool ok = (dataItemTypeId->ListType() == ListInterfaceType::Synchronous) &&
                  (propertyTypeId->ListType() == ListInterfaceType::Asynchronous);
        if (!ok) {
            // if one or both are no lists, we need to bail out
            return false;
        }

        // get list item types from the list types
        dataItemTypeId = dataItemTypeId->ListItemTypeId();
        propertyTypeId = propertyTypeId->ListItemTypeId();

        // if list item types wont match, find a conversion function
        if ((dataItemTypeId != propertyTypeId) && enableTypeConversion) {
            // check if both types size fit into gTypeConversionBuffer. This buffer is used
            // for temporary objects resulting from type conversion
            ok = (dataItemTypeId->Size() < sizeof(gTypeConversionBuffer)) &&
                       (propertyTypeId->Size() < sizeof(gTypeConversionBuffer));
            if (!ok) {
                COURIER_LOG_WARN("binding type size exceed type conversion buffer size. consider adjusting COURIER_TYPE_CONVERSION_BUFFER_SIZE");
            }
            // try to setup a type converter
            ok = ok && SetupTypeConverter(dataItemTypeId, propertyTypeId);
            if (!ok) {
                COURIER_LOG_WARN("failed to create binding from ItemKey %u to %s - no valid type converter",
                                  UInt32(itemKey), metaInfo->GetName());
            }
        }

        if (!IsReadOnly()) {
            // setup type of modifications allowed to the list
            const DataItemHierarchyNode *hn = Internal::DataBinding::DataItemAccessor::Node(itemKey);
            FEATSTD_DEBUG_ASSERT(hn != 0);
            mAllowEdit = hn->TestFlag(DataItemHierarchyNodeFlags::ListAllowEdit);
            mAllowAdd = hn->TestFlag(DataItemHierarchyNodeFlags::ListAllowAdd);
            mAllowRemove = hn->TestFlag(DataItemHierarchyNodeFlags::ListAllowRemove);
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ListBinding::OnTargetInitialized()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        // the binding will be assigned to the property after all widget properties
        // have been initialized from the asset loader. Otherwise list bindings
        // potentially are overwritten by default values of the list property

        bool ok = true;
        if (!TargetInitialized()) {
            // get capability object
            const BindablePropertyCapability *cap = GetData().GetCapabilityObject();
            ok = cap != 0;
            if (ok) {
                // the capability object set expects a shared pointer to an async list
                SharedPointer<SharedAsyncListInterface> ptr(this);
                DataItemValue value(&ptr);
                // set the property value with the capability object
                FEATSTD_LINT_NEXT_EXPRESSION(1774,"type is known")
                ok = cap->Set(static_cast<FrameworkWidget*>(GetTargetHost()), value, 0);

                // send refresh list so the widget refreshes the list content
                ListEvent listEvent(ListEventType::RefreshList);
                // if the update fails, trust that subsequent updates wont fail
                (void) SendListEvent(listEvent);
            }
            SetTargetInitialized(ok);
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ListBinding::OnDataItemMsg(const AbstractDataItemMsg *msg)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        mDelayItemRequestProcessing = true;

        ProcessPendingListItemRequests(true);

        bool ok = true;
        if (msg->ItemRequiresUpdate(GetItemKey())) {
            DataItemValue value(msg->GetItemValue(GetItemKey()));
            ok = value.IsValid();
            if (ok) {
                const SyncListInterface *slif = FeatStd::Internal::PointerToPointer<const SyncListInterface*>(value.ImmutablePointer());
                ListEvent listEvent(ListEventType::NewFragment,
                    slif->FragmentStartIndex(),
                    slif->FragmentEndIndex());
                ok = SendListEvent(listEvent);
            }
            mDelayItemRequestProcessing = false;
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    FixedSizeList<ListEvent, AsyncBindingBase::cListEventQueueSize>& ListBinding::EventQueue()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        static FixedSizeList<ListEvent, AsyncBindingBase::cListEventQueueSize> eventQueue;
        return eventQueue;
    }

    // ------------------------------------------------------------------------
    bool ListBinding::SendListEvent(const ListEvent &listEvent)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        bool ok;
        if (mIsEventProcessing) {                       // we got invoked in an SendListEvent
            ok = EventQueue().PushBack(listEvent);      // queue the event and return
        }
        else {
            mIsEventProcessing = true;                  // guard to detect recursive invocation
            FEATSTD_DEBUG_ASSERT(GetData().GetCapabilityObject() != 0);

            const BindablePropertyCapability *cap = GetData().GetCapabilityObject();
            if (cap != 0) {
                OnListEvent(listEvent);                 // let the capability object send the event
                while (EventQueue().Count() > 0) {      // process all events caused by the processing of the
                    ListEvent tmp(EventQueue()[0]);     // proceeding event
                    (void) EventQueue().RemoveAt(0);
                    OnListEvent(tmp);
                }
            }
            ok = cap != 0;
            mIsEventProcessing = false;
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    const SyncListInterface* ListBinding::List() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        DataItemValue value(GetModelValue());
        return value.IsValid() ? FeatStd::Internal::PointerToPointer<const SyncListInterface*>(value.ImmutablePointer()) : 0;
    }

    // ------------------------------------------------------------------------
    DataItemValue ListBinding::ListItem(FeatStd::SizeType index) const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *sli = List();
        return (sli == 0) ? DataItemValue() : sli->Item(index);
    }

    // ------------------------------------------------------------------------
    DataItemTypeId ListBinding::ListItemTypeId() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? COURIER_INVALID_DATA_ITEM_TYPE_ID() : slif->ListItemTypeId();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType ListBinding::Count() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? cUnknownQuantity : slif->Count();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType ListBinding::Capacity() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? cUnknownQuantity : slif->Capacity();
    }

    // ------------------------------------------------------------------------
    bool ListBinding::EndOfList(FeatStd::SizeType index) const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? true : slif->EndOfList(index);
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType ListBinding::FragmentStartIndex() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? true : slif->FragmentStartIndex();
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType ListBinding::FragmentItemCount() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        const SyncListInterface *slif = List();
        return (slif == 0) ? true : slif->FragmentItemCount();
    }

    // ------------------------------------------------------------------------
    bool ListBinding::Request(FeatStd::SizeType index, FeatStd::SizeType nItems, bool updateOnly)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        // block requests while in ProcessPendingListItemRequests
        if (mInItemRequestProcessing) {
            return false;
        }

        const SyncListInterface *slif = List();

        mRequestItemStartIndex = index;
        mRequestItemEndIndex = index + nItems;
        if (slif != 0) {
            // if the list interface is available, check request bounds
            if (mRequestItemEndIndex > slif->Count()) {
                // upper limit
                mRequestItemEndIndex = slif->Count();
            }
            // lower limit, if the request is completely off limits, ignore the request
            if (mRequestItemStartIndex >= mRequestItemEndIndex) {
                ClearPendingItemRequest();
            }
        }

        COURIER_LOG_DEBUG("Request: %u-%u", mRequestItemStartIndex, mRequestItemEndIndex);

        bool pendingRequest = mRequestItemStartIndex != ListInterfaceBase::InvalidListIndex();

        // in normal message processing, RequestItem requests are processed at
        // the end of message processing. if RequestItem has been invoked in a non
        // processing context (eg. a ViewController invokes a widget member function that
        // results in a ItemRequest), the processing must be done immediatly.
        if ((! static_cast<bool>(mDelayItemRequestProcessing)) && pendingRequest) {
            ProcessPendingListItemRequests(updateOnly);
        }
        return pendingRequest;
    }

    // ------------------------------------------------------------------------
    void ListBinding::ClearPendingItemRequest()
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        mRequestItemStartIndex = ListInterfaceBase::InvalidListIndex();
        mRequestItemEndIndex = ListInterfaceBase::InvalidListIndex();
        mRequestedItemStartIndex = ListInterfaceBase::InvalidListIndex();
        mRequestedItemEndIndex = ListInterfaceBase::InvalidListIndex();
    }

    // ------------------------------------------------------------------------
    void ListBinding::ProcessPendingListItemRequests(bool updateOnly)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (!HasPendingItemRequest()) {
            // no pending requests
            return;
        }
        FeatStd::SizeType requestItemStartIndex = mRequestItemStartIndex;
        FeatStd::SizeType requestItemEndIndex = mRequestItemEndIndex;
        // block all item new requests while in processing
        mInItemRequestProcessing = true;
        // get list interface
        const SyncListInterface *slif = List();
        if (slif != 0) {
            // send all items present in the cached DataItemMsg
            //todo: optimize to read already available data if mRequestItemStartIndex is not in fragment

            DataItemTypeId propertyTypeId = InvalidDataItemType::Ident();

            for (;;) {                                  // check if data is available, if not break
                if ((mRequestItemStartIndex < slif->FragmentStartIndex()) || (mRequestItemStartIndex >= slif->FragmentEndIndex())) {
                    break;
                }

                DataItemValue listItemValue(slif->Item(mRequestItemStartIndex));
                bool ok = !HasConvertFn();
                if (!ok) {
                    // requires type conversion
                    if (propertyTypeId == InvalidDataItemType::Ident()) {
                        // get the list item type of the property
                        propertyTypeId = GetData().GetCapabilityObject()->TypeId()->ListItemTypeId();
                        // construct in gTypeConversionBuffer an object of type propertyTypeId
                        propertyTypeId->Construct(gTypeConversionBuffer, DataItemValue());
                    }
                    DataItemValue dst(propertyTypeId, gTypeConversionBuffer, true);
                    // convert the list item to the property list item
                    ok = InvokeConvertFn(dst, listItemValue);
                    if (!ok) {
                        COURIER_LOG_WARN("conversion failed");
                    }
                    // pass the list item value to the event
                    listItemValue = dst;
                }

                if (ok) {
                    ListEvent listEvent(slif->IsModified(mRequestItemStartIndex) ? ListEventType::ModifiedItem : ListEventType::RequestedItem,
                                        mRequestItemStartIndex,
                                        ListInterfaceBase::InvalidListIndex(),
                                        listItemValue);
                    FEATSTD_DEBUG_ASSERT(listEvent.Item().IsValid());
                    if (!SendListEvent(listEvent)) {
                        break;                              // break and hope for a later retry
                    }
                }

                ++mRequestItemStartIndex;
                if (mRequestItemStartIndex >= mRequestItemEndIndex) {
                    ClearPendingItemRequest();
                    break;
                }
            }
            if (propertyTypeId != InvalidDataItemType::Ident()) {
                // if type conversion happend, clean up gTypeConversionBuffer - destruct the temporary objects
                propertyTypeId->Destruct(gTypeConversionBuffer);
            }
        }

        // requested items not available -> request from model
        // request the update if pending requests exist and the previously requested item range differs
        // from the current
        bool sendUpdateRequest = (!updateOnly) && HasPendingItemRequest() &&
                                 ((mRequestedItemStartIndex != requestItemStartIndex) || (mRequestedItemEndIndex != requestItemEndIndex));
        if (sendUpdateRequest) {
            // post list request to the model
            mRequestedItemStartIndex = requestItemStartIndex;
            mRequestedItemEndIndex = requestItemEndIndex;
            bool ok = PostListRequest(ListRequestType::NewFragment, requestItemStartIndex, requestItemEndIndex);
            if (!ok) {
                COURIER_LOG_ERROR("could not send ListRequest");
            }
        }
        mRequestItemEndIndex = ListInterfaceBase::InvalidListIndex();
        mRequestItemStartIndex = ListInterfaceBase::InvalidListIndex();
        mInItemRequestProcessing = false;
    }

    // ------------------------------------------------------------------------
    bool ListBinding::PostListRequest(ListRequestType::Enum reqType, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex, const DataItemValue &itemValue) const
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        AsyncModelBindingSource *bs = GetBindingSource();
        FEATSTD_DEBUG_ASSERT(bs != 0);

        bool ok = bs != 0;
        if (!ok) {
            return false;
        }

        DataItemTypeId dataItemTypeId = InvalidDataItemType::Ident();
        if (itemValue.IsValid() && HasConvertFn()) {
            // get the list item type id of the data item
            dataItemTypeId = DataItemAccessor::Descriptor(GetItemKey()).TypeId()->ListItemTypeId();

            // construct in gTypeConversionBuffer an object of type dataItemTypeId
            dataItemTypeId->Construct(gTypeConversionBuffer, DataItemValue());

            // apply conversion
            DataItemValue dst(dataItemTypeId, gTypeConversionBuffer, true);
            ok = InvokeConvertFn(dst, itemValue);
            if (!ok) {
                COURIER_LOG_WARN("type conversion failed");
            }

            // hand request over to BindingSource
            ok = ok && bs->AddListRequestRecord(GetItemKey(), reqType, newIndex, oldIndex, dst);

            // clean up the conversion object
            dataItemTypeId->Destruct(gTypeConversionBuffer);
        }
        else {
            ok = bs->AddListRequestRecord(GetItemKey(), reqType, newIndex, oldIndex, itemValue);
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestChange(FeatStd::SizeType idx, const DataItemValue &item)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return static_cast<bool>(mAllowEdit) && PostListRequest(ListRequestType::ChangeItem, idx, ListInterfaceBase::InvalidListIndex(), item);
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        FEATSTD_COMPILETIME_ASSERT(sizeof(FeatStd::SizeType) >= sizeof(DataItemKey));
        return static_cast<bool>(mAllowEdit) && PostListRequest(ListRequestType::ChangeDataItem, idx, FeatStd::SizeType(itemKey), item);
    }
    // ------------------------------------------------------------------------
    bool ListBinding::RequestAdd(FeatStd::SizeType idx, const DataItemValue &item)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return static_cast<bool>(mAllowAdd) && PostListRequest(ListRequestType::AddItem, idx, ListInterfaceBase::InvalidListIndex(), item);
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestRemove(FeatStd::SizeType idx)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return static_cast<bool>(mAllowRemove) && PostListRequest(ListRequestType::RemoveItem, idx);
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return static_cast<bool>(mAllowAdd) && static_cast<bool>(mAllowRemove) && PostListRequest(ListRequestType::MoveItem, dst,  src);
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestClear()
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return static_cast<bool>(mAllowRemove) && PostListRequest(ListRequestType::ClearList);
    }

    // ------------------------------------------------------------------------
    bool ListBinding::RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return PostListRequest(ListRequestType::PrefetchItems, startIndex, nItems);
    }

    // ------------------------------------------------------------------------
    UInt32 ListBinding::Flags() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        UInt32 flags = UInt32(ListFlags::None);
        if (static_cast<bool>(mAllowAdd)) {
            flags |= UInt32(ListFlags::AllowAdd);
        }
        if (static_cast<bool>(mAllowEdit)) {
            flags |= UInt32(ListFlags::AllowEdit);
        }
        if (static_cast<bool>(mAllowRemove)) {
            flags |= UInt32(ListFlags::AllowRemove);
        }
        return flags;
    }

    // ------------------------------------------------------------------------
    void ListBinding::ProcessListEvent(const ListEvent &listEvent)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        mDelayItemRequestProcessing = true;
        if (listEvent.EventType() == ListEventType::NewFragment) {
            ProcessPendingListItemRequests(true);
        }
        (void) SendListEvent(listEvent);
        ProcessPendingListItemRequests(true);
        mDelayItemRequestProcessing = false;
    }

    // ------------------------------------------------------------------------
    void ListBinding::OnReferrerUpdate(bool addReferrer)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        if (addReferrer) {
            ++mRefCount;
        }
        else if (mRefCount > 1) {
            --mRefCount;
        }
        else {
            Destroy();                                  // in case the property value gets overwritten
        }
    }
}}}}   // namespace
