//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncTextParameterBindingFactory.h"

#include <Courier/DataBinding/AsyncModelBindingSource.h>
#include <Courier/DataBinding/AsyncTextParameterBinding.h>

namespace Courier { 

#if defined(CANDERA_GLOBALIZATION_ENABLED)
COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);
#endif


namespace Internal {

    using namespace DataBinding;
   /**
    <summary>Creates a binding from the specified data item to widget property.</summary>
    <param name="itemKey">The data item key that is the source of the binding.</param>
    <param name="parameterIndex">[in,out] Index of the parameter to bind to.</param>
    <param name="textInstance">[in,out] the concrete widget instance.</param>
    <param name="typeConvert">(optional) specifies if the binding shall use use type conversion.</param>
    <param name="defaultValue">(optional) the default value.</param>
    <returns>true if the binding could be established, false otherwise.</returns>
    */
#if defined(CANDERA_GLOBALIZATION_ENABLED)
    bool AsyncTextParameterBindingFactory::CreateBinding( DataItemKey itemKey,
                        UInt8 parameterIndex,
                        Candera::Globalization::Internal::ParameterizedLocalizableText *textInstance,
                        bool typeConvert,
                        const DataItemValue &defaultValue) const
    {
        const DataItemHierarchyNode *node = DataItemAccessor::Node(itemKey);
        ModelBindingSource *bindingSource = ModelBindingSourceMap::Locate(ComponentId(ComponentType::View), DataItemAccessor::Root(node)->mItemKey, Candera::AbstractNodePointer());

        FEATSTD_DEBUG_ASSERT((parameterIndex < AsyncTextParameterBinding::cMaxParameterCount) && (textInstance != 0));

        if ((parameterIndex >= AsyncTextParameterBinding::cMaxParameterCount) || (textInstance == 0) || (bindingSource == 0)) {
                return false;
        }

        AsyncTextParameterBinding *binding = FEATSTD_NEW(AsyncTextParameterBinding);
        FEATSTD_LINT_NEXT_EXPRESSION(774, "(binding == 0) always is false")
        if (binding == 0) {
            return false;
        }

        if (!binding->Init(static_cast<AsyncModelBindingSource*>(bindingSource), itemKey, parameterIndex, textInstance, typeConvert, defaultValue)) {
            FEATSTD_DELETE(binding);
            binding = 0;
        } 
        else {
            COURIER_LOG_DEBUG("Item(%u)->%s", UInt32(itemKey), binding->GetTargetName());

            // copy value from binding source to binding.
            DataItemValue currentValue = bindingSource->GetDataItemValue(itemKey);
            if (currentValue.IsValid()) {
                (void)binding->SetTargetValue(currentValue);
            }
        }

        FEATSTD_LINT_CURRENT_SCOPE(429, "binding object will be released on reference counting or widget / binding source dtor")
        return binding != 0;
    }
#else
    bool AsyncTextParameterBindingFactory::CreateBinding(DataItemKey itemKey,
                                                           UInt8 parameterIndex,
                                                           Candera::Globalization::Internal::ParameterizedLocalizableText *textInstance,
                                                           bool typeConvert,
                                                           const DataItemValue &defaultValue) const
    {
        COURIER_UNUSED(itemKey);
        COURIER_UNUSED(parameterIndex);
        COURIER_UNUSED(textInstance);
        COURIER_UNUSED(typeConvert);
        COURIER_UNUSED(defaultValue);
        return false;
    }

#endif

}}

