//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include "AsyncTextParameterBinding.h"

#if defined(CANDERA_GLOBALIZATION_ENABLED)

#include <Courier/Diagnostics/Log.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::DataBinding);

using namespace Candera::Globalization::Internal;

namespace Internal {

    using namespace DataBinding;

    const UInt8 cInvalidParameterIndexLocal = AsyncTextParameterBinding::cInvalidParameterIndex; // for 1938 PC Lint warning
    // ------------------------------------------------------------------------
    AsyncTextParameterBinding::AsyncTextParameterBinding() :
        mCapability(0),
        mParameterIndex(cInvalidParameterIndexLocal)
    {
        mName[0] = '\0';
    }

    // ------------------------------------------------------------------------
    AsyncTextParameterBinding::~AsyncTextParameterBinding()
    {
        mParameterIndex = cInvalidParameterIndexLocal;
        CANDERA_DELETE(mCapability);
        mCapability = 0;
    }

    // ------------------------------------------------------------------------
    FEATSTD_LINT_CURRENT_SCOPE(1788, "Variable 'updateGuard' is referenced only by its constructor or destructor")
    bool AsyncTextParameterBinding::SetTargetValue(const DataItemValue &value)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        bool ok = true;
        UpdateFlagGuard updateGuard(this, &ok);
        ok = ok && (mCapability!=0) && mCapability->Set(static_cast<ParameterizedLocalizableText*>(GetTargetHost()), value, GetConvertFn());
        SetTargetInitialized(true);
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncTextParameterBinding::GetTargetValue(const DataItemValue &value) const
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return (mCapability!=0) && mCapability->Get(static_cast<ParameterizedLocalizableText*>(GetTargetHost()), value, GetConvertFn());
    }

    // ------------------------------------------------------------------------
    const Char* AsyncTextParameterBinding::GetTargetName() const
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        return mName;
    }

    // ------------------------------------------------------------------------
    DataItemTypeId TypeIdFromParameterType(ValueType parameterType)
    {
        DataItemTypeId retId;
        switch (parameterType) {
            case BooleanValueType:
                retId = DataItemType<bool>::Ident();
                break;
            case FloatValueType:
                retId = DataItemType<Float>::Ident();
                break;
            case Int32ValueType:
                retId = DataItemType<Int32>::Ident();
                break;
            case StringValueType:
                retId = DataItemType<FeatStd::String>::Ident();
                break;
            case InvalidValueType:
            default:
                retId = InvalidDataItemType::Ident();
        }
        return retId;
    }

    // ------------------------------------------------------------------------
    bool AsyncTextParameterBinding::Init(AsyncModelBindingSource *bindingSource,
        DataItemKey itemKey,
        UInt8 parameterIndex,
        ParameterizedLocalizableText* text,
        bool enableTypeConversion,
        const DataItemValue &defaultValue)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        bool ok = CustomSetup(bindingSource, itemKey, parameterIndex, text);
        if (!ok) {
            return false;
        }

        Parameters parameters(text->GetParametersHandle());
        ValueType parameterType = parameters.GetType(parameterIndex);
        ok = SetupCapability(parameterIndex, parameterType);

        DataItemTypeId parameterTypeId = TypeIdFromParameterType(parameterType);

        const DataItemDescriptor &itemDesc = DataItemAccessor::Descriptor(GetItemKey());
        const DataItemTypeId dataItemTypeId = itemDesc.TypeId();

        if (enableTypeConversion && (parameterTypeId != dataItemTypeId)) {
            ok = SetupTypeConverter(dataItemTypeId, parameterTypeId);
            if (!ok) {
                COURIER_LOG_WARN("failed to create binding from ItemKey %u to Parameter {%u} - no valid type converter",
                    UInt32(itemKey), parameterIndex);
            }
        }

        if(ok && defaultValue.IsValid()) {
            (void) SetTargetValue(defaultValue);
        }

        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncTextParameterBinding::CustomSetup(AsyncModelBindingSource *bindingSource,
        DataItemKey itemKey,
        UInt8 parameterIndex,
        ParameterizedLocalizableText *text)
    {
        COURIER_CHECK_SCOPE_AFFINITY();

        if (IsInitialized()) {
            FEATSTD_DEBUG_FAIL();
            return false;
        }

        bool ok = CustomInternalInit(parameterIndex, text);

        if (ok) {
            ok = AsyncBindingBase::Setup(bindingSource, itemKey);
        }

        SetInitialized(ok);
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncTextParameterBinding::CustomInternalInit(UInt8 parameterIndex, ParameterizedLocalizableText *text)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        Parameters parameters(text->GetParametersHandle());

        bool ok = (mParameterIndex == cInvalidParameterIndex) && (parameterIndex < parameters.GetCount());

        if (ok) {
            mParameterIndex = parameterIndex;
            InitName();
            ok = BindingBase::InternalInit(text);
        }
        return ok;
    }

    // ------------------------------------------------------------------------
    bool AsyncTextParameterBinding::SetupCapability(UInt8 parameterIndex, ValueType parameterType)
    {
        COURIER_CHECK_SCOPE_AFFINITY();
        switch (parameterType) {
            case BooleanValueType:
                mCapability = CANDERA_NEW(BindableBoolParameterCapability)(parameterIndex);
                break;
            case Int32ValueType:
                mCapability = CANDERA_NEW(BindableInt32ParameterCapability)(parameterIndex);
                break;
            case FloatValueType:
                mCapability = CANDERA_NEW(BindableFloatParameterCapability)(parameterIndex);
                break;
            case StringValueType:
                mCapability = CANDERA_NEW(BindableStringParameterCapability)(parameterIndex);
                break;
            case InvalidValueType:
            default:
                COURIER_LOG_ERROR("Unable to setup capability object for parameter %d (unknown parameter type).", parameterIndex);
                return false;
        }
        return true;
    }

    // ------------------------------------------------------------------------
    void AsyncTextParameterBinding::InitName() {
        COURIER_CHECK_SCOPE_AFFINITY();
        COURIER_DEBUG_ASSERT(mParameterIndex < cMaxParameterCount);

        static const Char* cNameBase = "Parameter ";

        FeatStd::Internal::String::Copy(mName, cNameBase);
        mName[cNameLength-2] = static_cast<Char>(mParameterIndex) + '0';
        mName[cNameLength-1] = '\0';
    }
}}

#endif
