//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_AsyncModelBindingSource_h)
#define Courier_DataBinding_AsyncModelBindingSource_h

#include <Courier/Messaging/Message.h>
#include <Courier/Messaging/MessageReferrer.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/ModelBindingSource.h>
#include <Courier/DataBinding/ModelBindingSourceMap.h>
#include <Courier/DataBinding/DataBindingMsg.h>

namespace Courier { namespace Internal {
/// @addtogroup COURIER_DATABINDING
/// @{

    /**
        <summary>Provides an asynchronous (message based) access to data items in the model component.
        Data is provided by the model with BindingSource DataItemMsg messages. </summary>
        <seealso cref="ModelBindingSource"/>
     */
    class AsyncGenericModelBindingSource : public ModelBindingSource {
       public:
            AsyncGenericModelBindingSource(DataItemKey itemKey);
            virtual ~AsyncGenericModelBindingSource();

            template<typename ItemType>
            static ItemType* GetDataItemValue(Courier::ComponentId componentId, DataItemKey itemKey, const Candera::AbstractNodePointer& node = Candera::AbstractNodePointer()) {
                DataItemValue value(GetGenericDataItemValue(componentId, itemKey, node));
                if (value.IsValid()) {
                    return value.GetValue<ItemType>();
                }
                return 0;
            }

            static UInt16 GetDataItemRevision(Courier::ComponentId componentId, DataItemKey itemKey, const Candera::AbstractNodePointer& node = Candera::AbstractNodePointer()) {
                ModelBindingSource *bindingSource = ModelBindingSourceMap::Locate(componentId, itemKey, node);
                if (bindingSource != 0) {
                    return bindingSource->DataRevision();
                }
                return 0;
            }

            /**
                <summary>Gets the data revision from CurrentDataItemMsg.</summary>
                <returns>Data revision or 0 if no message has been received so far.</returns>
             */
            virtual UInt16 DataRevision() const;

            virtual DataItemValue GetDataItemValue(DataItemKey itemKey);
            virtual void ReleaseDataItemMsgReference();

        protected:
            /**
                <summary>Gets the current data item message.</summary>
                <returns>null if no message has been received so far, else the message.</returns>
             */
            const AbstractDataItemMsg* CurrentDataItemMsg() const;

            virtual void OnDataItemMsg(const AbstractDataItemMsg *msg);
            virtual void OnListEventMsg(const ListEventMsg *msg);
            virtual bool CreateBinding(DataItemKey itemKey, 
                                       const ::Candera::MetaInfo::WidgetPropertyMetaInfo *wmi,
                                       FrameworkWidget *widgetInstance,
                                       bool typeConvert,
                                       const DataItemValue &defaultValue);

            static DataItemValue GetGenericDataItemValue(Courier::ComponentId componentId, DataItemKey itemKey, const Candera::AbstractNodePointer& node) {
                ModelBindingSource *bindingSource = ModelBindingSourceMap::Locate(componentId, itemKey, node);
                if (bindingSource != 0) {
                    return bindingSource->GetDataItemValue(itemKey);
                }
                return DataItemValue();
            }

        private:
            TypedMessageReferrer<const AbstractDataItemMsg> mCurrentDataItemMsg;
    };

    class AsyncModelBindingSourceDelegationHelper;

    /**
        <summary>Provides an asynchronous (message based) access to data items in the model component.
        Data is provided by the model with BindingSource DataItemMsg messages. </summary>
        <seealso cref="ModelBindingSource"/>
     */
    class AsyncModelBindingSource : public AsyncGenericModelBindingSource {
        public:
            AsyncModelBindingSource(DataItemKey itemKey);
            virtual ~AsyncModelBindingSource();

            virtual AsyncModelBindingSource* ToAsyncModelBindingSource()
            {
                return this;
            }

        protected:
            /**
                <summary>
                    Executes the post processing operation. Post processing for async binding source means to
                    send pending update model messages.
                </summary>
             */
            virtual void DoPostProcessing();

            /**
                <summary>
                    Gets a sync list interface for the specified data item from the current data message.
                </summary>
                <param name="itemKey">The data item key.</param>
                <returns>null if it fails, else the list interface.</returns>
             */
            const SyncListInterface* GetListInterface(DataItemKey itemKey) const;

            /**
                <summary>Gets a list item form a list with the specified data item key.</summary>
                <param name="itemKey">The data item key.</param>
                <param name="index">Zero-based index of the the list item.</param>
                <returns>
                    A valid DataItemValue of list item if succeeded, an invalid DataItemValue otherwise.
                </returns>
             */
            DataItemValue GetListItem(DataItemKey itemKey, FeatStd::SizeType index) const;

            virtual void OnDataItemMsg(const AbstractDataItemMsg *msg);
            virtual void OnListEventMsg(const ListEventMsg *msg);

            AsyncBinding *GetBindingHead() const
            {
                return mBindingHead;
            }

            /**
                <summary>Creates a binding between a data item and a widget property.</summary>
                <param name="itemKey">The item key.</param>
                <param name="wmi">[in,out] Pointer to the widget property meta information object.</param>
                <param name="widgetInstance">[in,out] the widget instance.</param>
                <param name="typeConvert">true if type conversion shall be used to establish the binding.</param>
                <param name="defaultValue">The default value.</param>
                <returns>true if the binding could be created, false if not.</returns>
             */
            virtual bool CreateBinding(DataItemKey itemKey, 
                                       const ::Candera::MetaInfo::WidgetPropertyMetaInfo *wmi,
                                       FrameworkWidget *widgetInstance,
                                       bool typeConvert,
                                       const DataItemValue &defaultValue);

    private:
        friend class AsyncModelBindingSourceDelegationHelper;
            AsyncBinding *mBindingHead;
            static UpdateModelMsg *mCachedUpdateModelMsg;

            /**
                <summary>
                    Gets an existing or creates a new update model message. The created message object remains in the
                    ownership of the BindingSource until PostUpdateModelMsg is invoked. Subsequent changes of widget
                    properties will be collected in the message object and bulk sent to the model in either binding
                    source post processing (DoPostProcessing) or by explictly call PostUpdateModelMsg.
                </summary>
                <returns>null if message object could not be created, else the update model message object.</returns>
             */
            static UpdateModelMsg* GetCachedUpdateModelMsg();

            /**
                <summary>
                    Posts a pending UpdateModelMsg. This function is invoked explicitely or in binding source
                    DoPostProcessing. If message post fails, the binding source will request post processing again in
                    the hope that next message cycle post will succeed.
                </summary>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            static bool PostCachedUpdateModelMsg();

            /**
                <summary>Enlists the given binding object to the list of binding objects.</summary>
                <param name="binding">the binding object.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool Enlist(Internal::AsyncBinding *binding);
            /**
                <summary>Unlists the given binding object from the binding object list.</summary>
                <param name="binding">the binding.</param>
                <returns>true if it succeeds, false if it fails.</returns>
             */
            bool Unlist(Internal::AsyncBinding *binding);

            /**
            <summary>Request to copy the bound widget property value back to the model.</summary>
            <param name="binding">The binding object.</param>
            <returns>true if it succeeds, false if it fails.</returns>
            */
            virtual bool AddChangeRecord(const AsyncBinding &binding) const;
            virtual bool AddListRequestRecord(DataItemKey itemKey, ListRequestType::Enum requestType, FeatStd::SizeType newIndex,
                FeatStd::SizeType oldIndex, const DataItemValue &value) const;


            friend class Internal::DataBinding::Async::ElementaryBinding;
            friend class Internal::DataBinding::Async::ListBinding;
            friend class Internal::ModelBindingSourceMap;
            friend class Internal::AsyncBindingBase;
    };

    class AsyncModelBindingSourceDelegationHelper {
    public:
        static inline bool AddChangeRecord(const AsyncModelBindingSource& asyncModelBindingSource, const AsyncBinding &binding)
        {
            return asyncModelBindingSource.AddChangeRecord(binding);
        }

        static inline bool AddListRequestRecord(const AsyncModelBindingSource& asyncModelBindingSource, DataItemKey itemKey, ListRequestType::Enum requestType, FeatStd::SizeType newIndex, FeatStd::SizeType oldIndex, const DataItemValue &value)
        {
            return asyncModelBindingSource.AddListRequestRecord(itemKey, requestType, newIndex, oldIndex, value);
        }


    };

} ///@}
} // namespace

#endif // Courier_DataBinding_AsyncModelBindingSource_h
