//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_DataBinding_AsyncListInterface_h)
#define Courier_DataBinding_AsyncListInterface_h

#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/Util/SharedPointer.h>
#include <Courier/DataBinding/ListInterfaceBase.h>
#include <Courier/DataBinding/Infrastructure.h>

namespace Courier {  
/// @addtogroup COURIER_DATABINDING
/// @{
    namespace ListFlags {
        enum Enum {
            None = 0,

            /** existing list entries can modified */
            AllowEdit = (1U << 0U),

            /** new list items can be added to the list */
            AllowAdd = (1U << 1U),

            /** list item can be removed from the list */
            AllowRemove = (1U << 2U),

            /** alias for a list that allows all modification types */
            FullControl = (AllowEdit | AllowAdd | AllowRemove)
        };
    }

    namespace ListRequestType {
        enum Enum {
            /** no valid request */
            None = 0,

            /** requests to add a list item to the list */
            AddItem,

            /** requests to modify a list item */
            ModifyItem,

            /** alias for ModifyItem */
            ChangeItem = ModifyItem,

            /** request to remove a list item */
            RemoveItem,

            /** request to move a list item */
            MoveItem,

            /** request to send a new list fragment */
            NewFragment,

            /** request to clear the list */
            ClearList,

            /** request to prefetch items to the model local cache */
            PrefetchItems,

            /** requests to modify a data item in a list item */
            ChangeDataItem
        };
    }

    /**
        <summary>Defines an asynchronous list interface.</summary>
        <seealso cref="ListInterfaceBase"/>
     */
    class AsyncListInterface : public ListInterfaceBase {
        public:
            /**
                <summary>
                    Requests list items starting at the given index. The list will answer this request with series of
                    ListEventType::RequestedItem events. The underlaying list implementation might not be able to
                    handle more than one active request at a time. Thus repeated requests will supersede any
                    previously made item request.
                </summary>
                <param name="index">Zero-based start index.</param>
                <param name="nItems">Numbr of items to request.</param>
                <returns>true if it succeeds index is within list bounds.</returns>
             */
            virtual bool Request(FeatStd::SizeType index, FeatStd::SizeType nItems = 1, bool updateOnly = false) = 0;

            /**
                <summary>
                    Request change the list item with the given index. The request will trigger a
                    ListEventType::ModifiedItem event.
                </summary>
                <param name="idx">The index.</param>
                <param name="item">The item.</param>
                <returns>true if index is within list bounds and the list is not readonly.</returns>
             */
            virtual bool RequestChange(FeatStd::SizeType idx, const DataItemValue &item) = 0;

            /**
                <summary>
                    Request change the data item within the list item at the given index. The request will trigger a
                    ListEventType::ModifiedDataItem event.
                </summary>
                <param name="idx">The index.</param>
                <param name="item">The item.</param>
                <returns>true if index is within list bounds and the list is not readonly.</returns>
             */
            virtual bool RequestChangeDataItem(FeatStd::SizeType idx, DataItemKey itemKey, const DataItemValue &item) = 0;

            /**
                <summary>Request to add the given item at given index to the list.</summary>
                <param name="idx">The insertion index.</param>
                <param name="item">The item.</param>
                <returns>true if index is within list bounds and the list is not readonly.</returns>
             */
            virtual bool RequestAdd(FeatStd::SizeType idx, const DataItemValue &item) = 0;

            /**
                <summary>Request to remove item at given index from the list.</summary>
                <param name="idx">The index of the item to be removed.</param>
                <returns>true if the index is within list bounds and the list is not readonly.</returns>
             */
            virtual bool RequestRemove(FeatStd::SizeType idx) = 0;

            /**
                <summary>Request move item from the src to dst index.</summary>
                <param name="dst">Destination index.</param>
                <param name="src">Source index.</param>
                <returns>true if indices are withih list bounds and the list is not readonly</returns>
             */
            virtual bool RequestMove(FeatStd::SizeType dst, FeatStd::SizeType src) = 0;

            /**
                <summary>Requst to remove all items from the list.</summary>
                <returns>true if the list is not readonly.</returns>
             */
            virtual bool RequestClear() = 0;

            /**
                <summary>
                    Request to prefetch items in the model. This request indicates to the model that the requester
                    will issue Request calls for the given list fragment in the near future. If e.g. a list widget is
                    scrolling its content, the list widget can use this request to trigger prefetching in the model.
                    It is up to the model how to act on the request. The model should not answer the request with
                    sending a new fragment immediatly, but only get ready for the subsequent Request request for the
                    specified list fragment. Prefetch requests only will be sent if the list is sent in fragments
                    (i.e. a windowSize is defined in the XCDDL list data definition)
                </summary>
                <param name="startIndex">The start index of the prefetching fragment.</param>
                <param name="nItems">The number of items starting from stardIndex to prefetch.</param>
                <returns>true if the request could be sent, false if not.</returns>
             */
            virtual bool RequestPrefetch(FeatStd::SizeType startIndex, FeatStd::SizeType nItems) = 0;

            /**
                <summary>Executes the list event action.</summary>
                <param name="listEvent">The list event.</param>
             */
            virtual void OnListEvent(const ListEvent &listEvent) = 0;

            /**
                <summary>Gets the flags of the list.</summary>
                <returns>any combination of flags specified in ListFlags::Enum</returns>
             */
            virtual UInt32 Flags() const = 0;

    };

/// @}
}

#endif // Courier_DataBinding_AsyncListInterface_h
