//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_WIDGETMETAINFO_H)
#define CANDERA_WIDGETMETAINFO_H

#include <Candera/System/MetaInfo/MetaInfo.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/MetaInfo/PublicMacros.h>

#ifdef CANDERA_META_DESCRIPTION
#include <string>
#endif

namespace Candera {
    class WidgetBase;
#ifdef CANDERA_META_DESCRIPTION
    namespace Internal {
        class EventMetaInfoSetBase;
    }
#endif
    namespace MetaInfo {
/// @addtogroup WidgetBase
/// @{
        /**
         * @brief WidgetMetaInfo defines the meta information stored for each Widget type
         */
        class WidgetMetaInfo : public ObjectMetaInfo<WidgetBase> {
            typedef ObjectMetaInfo<WidgetBase> Base;
            public:
                enum Type {
                    Widget2D,   ///< 2D Widget.
                    Widget3D,   ///< 3D Widget.
                    WidgetMixed ///< 2D/3D Widget.
                };

                /**
                 *  Constructs object with given name.
                 *  @param name Name to assign to WidgetMetaInfo object.
                 *  @param baseWidgetMetaInfo the meta info of the base class (or 0 if no base class is available).
                 */
                WidgetMetaInfo(const Char *name, const WidgetMetaInfo* baseWidgetMetaInfo) :
                    Base(name),
                    m_baseWidgetMetaInfo(baseWidgetMetaInfo)
                {
                }

                /**
                 *  Retrieves the Type of this object.
                 *  @return The Type of this object.
                 */
                virtual Type GetType() const = 0;

#ifdef CANDERA_META_DESCRIPTION
                /**
                 *  Retrieves requirement string of the widget
                 *  Function must only be used in scene composer context.
                 *  The widget can define requirements for textures, meshes etc. The
                 *  requirement string contains a list of requirements separated with
                 *  semicolon. Each requirement consists of the requirement tag (e.g. texture)
                 *  and identifier separated by a colon.
                 *  Sample: texture:Needle;model:Tube;model:Needle
                 *  @return requirement string
                 */
                virtual std::string GetRequirements() const
                {
                    return (0 != m_baseWidgetMetaInfo) ? m_baseWidgetMetaInfo->GetRequirements() : std::string("");
                }

                virtual const Candera::Char* GetCategory() const
                {
                    return (0 != m_baseWidgetMetaInfo) ? m_baseWidgetMetaInfo->GetCategory() : Base::GetCategory();
                }

                virtual const Candera::Char* GetDescription() const
                {
                    return (0 != m_baseWidgetMetaInfo) ? m_baseWidgetMetaInfo->GetDescription() : Base::GetDescription();
                }

                virtual bool IsDeprecated() const
                {
                    return (0 != m_baseWidgetMetaInfo) ? m_baseWidgetMetaInfo->IsDeprecated() : Base::IsDeprecated();
                }

                virtual const Candera::Internal::EventMetaInfoSetBase* GetEvents() const
                {
                    return (0 != m_baseWidgetMetaInfo) ? m_baseWidgetMetaInfo->GetEvents() : 0;
                }

                class Events
                {
                public:
                    static const Candera::Internal::EventMetaInfoSetBase* GetInstance()
                    {
                        return 0;
                    }


                };
#endif

                /**
                 *  Creates a new instance of the corresponding Widget class.
                 *  The instance shall be destroyed by calling Dispose on the widget.
                 *  @return A pointer to the created Widget class
                 */
                virtual WidgetBase* Create() = 0;

                const WidgetMetaInfo* GetBaseWidgetMetaInfo() const
                {
                    return m_baseWidgetMetaInfo;
                }

            private:
                WidgetMetaInfo();
                WidgetMetaInfo(const WidgetMetaInfo&);
                WidgetMetaInfo& operator=(const WidgetMetaInfo&);
                const WidgetMetaInfo* m_baseWidgetMetaInfo;
        };

        typedef PropertyMetaInfo<WidgetBase> WidgetPropertyMetaInfo;

        //template<typename PropType, typename WidgetType> class TypedPropertyMetaInfo : public WidgetPropertyMetaInfo {
        //    public:
        //        virtual bool Get(const Widget *widget, Char *buf, UInt buf_size) {    // scene composer only
        //            return DataType<PropType>::ConvertToString(Get(reinterpret_cast<const WidgetType*>(widget)), buf, buf_size);
        //        }

        //        virtual bool Set(Widget *widget, const Char *str) {                    // scene composer only
        //            PropType val;
        //            bool ok = DataType<PropType>::ConvertFromString(val, str);
        //            if (ok) {
        //                Set(reinterpret_cast<WidgetType*>(widget), val);
        //            }
        //            return ok;
        //        }

        //        virtual bool Set(Widget *widget, const AnimationPropertyValue &apv) {
        //            PropType val;
        //            bool ok = DataType<PropType>::ConvertFromString(val, apv);
        //            if (ok) {
        //                Set(reinterpret_cast<WidgetType*>(widget), val);
        //            }
        //            return ok;
        //        }

        //        virtual bool Set(Widget *widget, ByteStream &bs) {
        //            PropType val;
        //            bool ok = DataType<PropType>::ConvertFromString(val, bs);
        //            if (ok) {
        //                Set(reinterpret_cast<WidgetType*>(widget), val);
        //            }
        //            return ok;
        //        }

        //        virtual const PropType& Get(const WidgetType &widget) = 0;
        //        virtual void Set(WidgetType &widget, const PropType &val) = 0;
        //};

        /**
         * @brief Defines meta information object for a WidgetSet
         */
        class WidgetSetMetaInfo : public MetaInfoBase {
            public:
                WidgetSetMetaInfo(const Char *name) : MetaInfoBase(name) { }

                /**
                 *  Lookup Widget type meta information object with the given name
                 *     @param name name of widget type
                 *     @return pointer to meta information object if found, null otherwise
                 */
                virtual WidgetMetaInfo* LookupItem(const Char *name) const = 0;

                /**
                 *  Get Widget type meta information object at given index
                 *     @param idx index of the item to retrieve
                 *     @return pointer to meta information object
                 */
                virtual WidgetMetaInfo* GetItem(Int32 idx) const = 0;

                /**
                 *  Return the number of widget meta information objects in the widget set
                 *     @return number of widgets in the widget set
                 */
                virtual Int32 GetItemCount() const = 0;

                /**
                 *  Retrieve WidgetSet version.
                 *     @return version.
                 */
                virtual Int32 GetVersion() const { return 0; }
        };
/// @}
    }     // namespace MetaInfo
}    // namespace Candera

#ifdef CANDERA_META_DESCRIPTION
#define _CdaWidgetDefInternal_GetEvents() \
    virtual const Candera::Internal::EventMetaInfoSetBase* GetEvents() const override { \
        const Candera::Internal::EventMetaInfoSetBase* events = Events::GetInstance(); \
        if (0 == events) { \
            events = Candera::Internal::Exposer<BaseMetaInfoProvider, DoCreate>::GetEvents<const Candera::Internal::EventMetaInfoSetBase*, Candera::MetaInfo::WidgetMetaInfo*>(); \
        } \
        return events; \
    }
#else
#define _CdaWidgetDefInternal_GetEvents()
#endif

/*  ----------------------------------------------------------------------------
    Public macros
    ---------------------------------------------------------------------------- */
/// @addtogroup WidgetBase
/// @{
/**
 *  internal macro - Opens a widget definition section
 *     @param SpecificClassType type of the widget to define meta information for
 *     @param BaseClassType widget base class that widgetType is derived from
 *     @param WidgetType the specific widget type enum
 *     @param CreateReturnValue the specific create call
 */
#define _CdaWidgetDefInternal(SpecificClassType, BaseClassType, WidgetType, CreateReturnValue) \
    /*lint -save -e578 -e1942  : 1942: MISRA C++ Rule 14-6-1 - Remove after rework */ \
    template<typename T, bool DoCreate> friend struct ::Candera::Internal::Exposer; \
    template <typename T> friend class ::Candera::Internal::AIR; \
    template<bool DoCreate> \
    class CMI : public ::Candera::MetaInfo::WidgetMetaInfo { \
    public: \
        typedef ::Candera::MetaInfo::WidgetMetaInfo BaseMetaInfoType; \
        typedef ::Candera::WidgetBase BaseHostType; \
        typedef BaseClassType BaseMetaInfoProvider; \
        typedef Candera::MetaInfo::PropertyMetaInfo<BaseHostType> AggregationItemBaseType; \
        typedef typename ::Candera::Internal::Exposer< BaseClassType, DoCreate>::CMI::P BasePropertyType; \
        typedef SpecificClassType CurrentType; \
        CMI(Candera::MetaInfo::WidgetMetaInfo* baseWidgetMetaInfo) : BaseMetaInfoType(FEATSTD_STRINGIZE(SpecificClassType), baseWidgetMetaInfo) { } \
        virtual typename BaseMetaInfoType::Type GetType() const { return WidgetType; } \
        virtual BaseHostType* Create() { return CreateReturnValue; } \
        static inline CurrentType* CdaCastOperator(BaseHostType* object, CurrentType*) { return static_cast<CurrentType*>(object); } \
        static inline const CurrentType* CdaCastOperator(const BaseHostType* object, const CurrentType*) { return static_cast<const CurrentType*>(object); } \
        _CdaWidgetDefInternal_GetEvents()
/**
*  Opens a widget definition section
*     @param SpecificClassType type of the widget to define meta information for
*     @param BaseClassType widget base class that widgetType is derived from
*/
#define CdaWidgetDef(SpecificClassType, BaseClassType) _CdaWidgetDefInternal(SpecificClassType, BaseClassType, Candera::MetaInfo::WidgetMetaInfo::Widget3D, FEATSTD_NEW(CurrentType)())
#define CdaWidget2DDef(SpecificClassType, BaseClassType) _CdaWidgetDefInternal(SpecificClassType, BaseClassType, Widget2D, FEATSTD_NEW(CurrentType)())
#define CdaWidgetAbstractDef(SpecificClassType, BaseClassType) _CdaWidgetDefInternal(SpecificClassType, BaseClassType, Candera::MetaInfo::WidgetMetaInfo::Widget3D, 0)

/**
 *  Closes a widget definition section
 */
#define CdaWidgetDefEnd() \
        typedef Candera::MetaInfo::Internal::StaticAggregation<AggregationItemBaseType, ItemAggregation > Aggregation; \
        virtual Candera::Int32 GetItemCount() const { return Aggregation::Count(); } \
        virtual typename Aggregation::ItemType* LookupItem(const Candera::Char *name) const { return Aggregation::Lookup(name); } \
        virtual typename Aggregation::ItemType* GetItem(Candera::Int32 idx) const { return Aggregation::Item(idx); } \
        virtual Candera::UInt32 GetHash() const { return Candera::MetaInfo::Internal::Fnv1aHash::MixHash(Candera::MetaInfo::Internal::Fnv1aHash::Hash(GetName()), Aggregation::GetHash()) & 0x7FFFFFFF; } \
    /*lint -restore*/ \
    }; \
    virtual Candera::MetaInfo::WidgetMetaInfo* GetMetaInfo() const { return GetMetaInfoImpl(); } \
    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Symbol hidden from public API with using, hiding is accepted.") \
    static Candera::MetaInfo::WidgetMetaInfo* GetMetaInfoImpl();

#undef CdaProperties
#define CdaProperties() \
    struct P : public BasePropertyType {

/**
 *  Closes a property definition section within a widget definition section (@see CdaWidgetDef)
 */
#undef CdaPropertiesEnd
#define CdaPropertiesEnd() \
    }; \
    struct ItemAggregation { \
        P p; \
        CdaInjectedWidgetPrivateProperties() \
    };

/**
 *  Opens a property definition section within a properties definition section (@see CdaProperties)
 *     @param Name name of the property (e.g. Position)
 *     @param Type type of the property (e.g. Point3D)
 *     @param getter getter function of the widget to get property value (e.g. GetPosition)
 *     @param setter setter function of the widget to set property value (e.g. SetPosition)
 */
#define _CdaWidgetPropertyProlog(Name, Type) \
    typedef class FEATSTD_CONCAT2(Cda_, Name) : public AggregationItemBaseType{ \
    public: \
    FEATSTD_CONCAT2(Cda_, Name)() : AggregationItemBaseType(FEATSTD_STRINGIZE(Name)) { } \
    /* next line is to avoid compiler warning "overloaded virtual function ... is only partially overridden in class..." */ \
    using AggregationItemBaseType::Set; \
    typedef Type TPropertyType; \
    typedef const Candera::Char * SetterValueType; \
    _CdaDataTypeEditorInfo(Type) \
    _CdaDimensions(Type) \
    _CdaHash(Name, Type)

#undef CdaProperty
#define CdaProperty(Name, Type, getter, setter) \
    _CdaWidgetPropertyProlog(Name, Type) \
        _CdaPropGetter(getter) \
        _CdaPropSetter(setter) \
        _CdaIndirectAPSFactory(Name, setter, getter)

#undef CdaFieldProperty
#define CdaFieldProperty(Name, Type, Prop) \
    _CdaWidgetPropertyProlog(Name, Type) \
        _CdaFieldProperty(Prop) \
        _CdaDirectAPSFactory(Name, Prop)

#undef CdaPropertyEnd
#define CdaPropertyEnd() \
        _CdaEditorInfoChooser() \
        FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
    } _CdaUniqueName(AggregationItem); \
    FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
    Candera::MetaInfo::Internal::AggregationInstanceRef<_CdaUniqueName(AggregationItem)> _CdaUniqueName(Item);

/**
 *  Defines an empty widget set for applications which do not include any widgets at all.
 */

#define CdaEmptyWidgetSet Candera::MetaInfo::WidgetSetMetaInfo* GetWidgetSet() { return 0; }

/**
 *  Opens a widget set definition.
 *     @param name of the widget set
 */
#define CdaWidgetSet(name) \
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 7-3-1: removing the symbol from global name space violates backward compatibility.") \
    typedef struct name : public Candera::MetaInfo::WidgetSetMetaInfo { \
        _CdaMetaInfoCtors(name, name, Candera::MetaInfo::WidgetSetMetaInfo)

/**
 *  Closes a widget set definition.
 */
#define CdaWidgetSetEnd() \
        FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
    } _CdaUniqueName(WidgetSet); \
    Candera::MetaInfo::WidgetSetMetaInfo* GetWidgetSet() { \
        static _CdaUniqueName(WidgetSet) widgetSet; \
        return &widgetSet; \
    }

/**
 *  Opens a widget enumeration section within a widget set section (@see CdaWidgetSet)
 */
#define CdaWidgets() \
    typedef Candera::WidgetBase BaseHostType; \
    _CdaStaticAggregationProlog(Widgets, Candera::MetaInfo::WidgetMetaInfo)

/**
 *  Closes a widget enumeration section within a widget set section (@see CdaWidgetSet)
 */
#define CdaWidgetsEnd() \
    _CdaStaticAggregationEpilog(Widgets, Widgets) \
    _CdaMetaInfoHashInterface(Widgets)

/**
 *  Adds the given widget to the widget set within a @see CdaWidgets section
 *     @param typeName of the widget type
 */
#define CdaWidget(typeName) \
    FEATSTD_LINT_NEXT_EXPRESSION(1516, "Member declaration hides inherited member [MISRA C++ Rule 2-10-2] - false positive, the name is not required and collisions of the generated name are not relevant!") \
    FEATSTD_LINT_NEXT_EXPRESSION(1013, "Symbol 'CMI' not a member of class 'ExposerDoCreate' - false positive, CMI is a member of ExposerDoCreate.")  \
    Candera::Internal::AIR<Candera::Internal::ExposerDoCreate< typeName >::CMI > _CdaUniqueName(Item);

    /**
     *  Provides the widget set meta information
     *     @return pointer to widget meta information
     */
    Candera::MetaInfo::WidgetSetMetaInfo* GetWidgetSet();

#ifdef CANDERA_META_DESCRIPTION
    /**
     *  Opens a requirements sections within a CdaWidgetDef section
     */
    #define CdaRequirements() \
        virtual std::string GetRequirements() const { \
        return BaseMetaInfoType::GetRequirements() + std::string(

    /**
     *  Closes a requirements sections within a CdaWidgetDef section
     */
    #define CdaRequirementsEnd() \
                ); \
        }

    /**
     *  Defines within a @see CdaRequirements section a required camera identifier
     *     @param req string identifier of object required by the widget
     */
    #define CdaCameraReq(req) ";Camera:" req

    /**
     *  Defines within a @see CdaRequirements section a required group identifier
     *     @param req string identifier of object required by the widget
     */
    #define CdaGroupReq(req) ";Group:" req

    /**
     *  Defines within a @see CdaRequirements section a required light identifier
     *     @param req string identifier of object required by the widget
     */
    #define CdaLightReq(req) ";Light:" req

    /**
     *  Defines within a @see CdaRequirements section a required mesh identifier
     *     @param req string identifier of object required by the widget
     */
    #define CdaMeshReq(req) ";Mesh:" req

    /**
     *  Defines within a @see CdaRequirements section a required node identifier
     *     @param req string identifier of object required by the widget
     */
    #define CdaNodeReq(req) ";Node:" req

#else

    #define CdaRequirements()
    #define CdaRequirementsEnd()
    #define CdaCameraReq(req)
    #define CdaGroupReq(req)
    #define CdaLightReq(req)
    #define CdaMeshReq(req)
    #define CdaNodeReq(req)

#endif
/// @}
#endif // CANDERA_METAINFO_H
