//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_WidgetBase_H)
    #define CANDERA_WidgetBase_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/Environment.h>

#include <CanderaWidget/WidgetBase/WidgetRtti.h>
#include <CanderaWidget/WidgetBase/WidgetDefaultMetaInfo.h>
#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>
#include <Candera/EngineBase/Common/StringIdentifier.h>

#if defined(CANDERA_INJECT_WIDGET_BASE_INCLUDE)
    /*  if defined, this include must pull in a class CanderaInjectedWidgetBase
        this class will then be used as base class for Candera::WidgetBase */
    #include CANDERA_INJECT_WIDGET_BASE_INCLUDE
#else
    /** if no external injected widget base class is supplied, define a dummy empty class
        frameworks can inject generic widget functionality by supplying a CanderaInjectedWidgetBase
        class. if no external base class is provided, use this empty dummy class as base class
        to WidgetBase. */
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "Only global scope is applicable to CanderaInjectedWidgetBase")
    class CanderaInjectedWidgetBase
    {
        public:
            CanderaInjectedWidgetBase() { }
            virtual ~CanderaInjectedWidgetBase() { }
    };
#endif

#if !defined(CdaInjectedWidgetPrivateProperties)
#define CdaInjectedWidgetPrivateProperties()
#endif

namespace Candera {
    class AssetProvider;
/// @addtogroup WidgetBase
/// @{
    /**
     * @brief Abstract base class for Candera Widgets.
     *
     */
    class WidgetBase : public CanderaInjectedWidgetBase
    {
        public:
            typedef MemoryManagement::ArrayDisposer<const Char*> Disposer;
            typedef Disposer::DisposerFunction DisposerFn;

            /**
             *  Constructor
             */
            WidgetBase();

            /**
             *  Destructor
             */
            virtual ~WidgetBase();

            /**
             *  Disposer
             *  Use this method to dispose widgets created through WidgetMetaInfo::Create.
             */
            void Dispose();

            /**
             *  Retrieves the name of the Widget object.
             *  @return the name of the Widget object
             */
            const Char* GetName() const { return m_name; }

            /**
             *  Sets the name of the Widget object.
             *  @param name The name of the Widget object that is set.
             */
            void SetName(const Char* name) { m_name = name; }

            /**
             *  Retrieves the id of the Widget object.
             *  @return the id of the Widget object
             */
            Id GetId() const { return m_id; }

            /**
             *  Sets the id of the Widget object.
             *  @param id The id of the Widget object that is set.
             */
            void SetId(Id id) { m_id = id; }

            /**
             * Set StringIdentifier of widget's owner (composite).
             * For composite widgets, the identifier must be the identifier of
             *  the CompositeGroup that owns the widget. For global widgets, the 
             *  identifier must be 0.
             * 
             * @param identifier Owner's identifier.
             */
            void SetOwnerId(StringIdentifier* identifier) { m_stringIdentifier.SetOwner(identifier); }

            /*
             * Set a string id that is unique in the owner's context.
             *
             * @param id String id.
             * @param disposer String disposer.
             */
            void SetStringId(const Char* id, StringIdentifier::DisposerFn disposer) { m_stringIdentifier.SetId(id, disposer); }

            /**
             * Get StringIdentifier of this widget.
             * @return StringIdentifier.
             */
            const StringIdentifier* GetStringId() const { return &m_stringIdentifier; }

            /**
             *  Initializes the widget so that all referred resource can be resolved
             *  @param assetProvider Asset provider to resolve all referred resources.
             */
            virtual void Init(AssetProvider* assetProvider) {
                FEATSTD_UNUSED(assetProvider);
            }

            /**
             *  Invoked before rendering. Change node properties here.
             */
            virtual void Update() { }

            /**
             *  Finalize the widget to release all referred resources.
             */
            virtual void Finalize() { }

            /**
             *  Assign a disposer to the widget to dispose the widget name.
             */
            void SetDisposer(DisposerFn disposer) { m_disposerFn = disposer; }

            /**
             *  Sets the AnimationTimeDispatcher for AnimationPlayers
             *  @param timeDispatcher AnimationTimeDispatcher to set.
             */
            void SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer timeDispatcher);

            /**
             *  Retrieves the widget's AnimationTimeDispatcher
             *  @return the AnimationTimeDispatcher
             */
            Animation::AnimationTimeDispatcher::SharedPointer GetAnimationTimeDispatcher() const { return m_animationTimeDispatcher; }

#ifdef CGI_DOXYGEN
            /**
            *  Get the MetaInfo object for this widget.
            */
            virtual Candera::MetaInfo::WidgetMetaInfo* GetMetaInfo() const {}
#endif
            CGI_WIDGET_RTTI_DECLARATION(WidgetBase, WidgetDefaultMetaInfo);

            CdaWidgetDef(WidgetBase, WidgetDefaultMetaInfo)
                CdaDescription("Candera Widget Base Class")

                CdaProperties()
                    CdaProperty(Name, const Char*, GetName, SetName)
                        CdaDescription("The name of the widget instance")
                    CdaPropertyEnd()
                CdaPropertiesEnd()
            CdaWidgetDefEnd()

        protected:
            virtual void OnAnimationTimeDispatcherChanged() { }

        private:
            const Char* m_name;
            Id m_id;
            StringIdentifier m_stringIdentifier;
            Animation::AnimationTimeDispatcher::SharedPointer m_animationTimeDispatcher;
            DisposerFn m_disposerFn;
    };

    namespace Internal {
        struct MetaInfoExposer {
            static inline Candera::MetaInfo::WidgetMetaInfo* GetMetaInfo(const WidgetBase* widget)
            {
                if (0 != widget) {
                    return widget->GetMetaInfo();
                }
                return 0;
            }

            template<typename T>
            static inline Candera::MetaInfo::WidgetMetaInfo* GetMetaInfo()
            {
                return T::GetMetaInfoImpl();
            }

            template<typename T>
            static inline const Candera::Char* GetName()
            {
                Candera::MetaInfo::WidgetMetaInfo* metaInfo = T::GetMetaInfoImpl();
                if (0 != metaInfo) {
                    return metaInfo->GetName();
                }
                return "";
            }
        };
    }
/// @}
}   // namespace Candera
#endif // CANDERA_WidgetBase_H
