//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_FRAGMENT_H)
#define CANDERA_TRANSITION_FRAGMENT_H

#include <FeatStd/Util/String.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Mathematics/Vector2.h>

#include <CanderaTransitions/Hint.h>
#include <CanderaTransitions/RequestFragment.h>
#include <CanderaTransitions/Rule.h>
#include <CanderaTransitions/Trigger.h>

namespace Candera {

namespace Transitions {

/// @addtogroup Transition
/// @{
class TransitionStrategy;

/**
*  @brief Abstract base class from which to derive TransitionFragment implementations. 
*  
*  A transition fragment implements transition behaviour. An example implementation is provided as DefaultTransitionFragment.
*  Transition fragments are generated by TransitionFragmentFactory implementations.
*/
class TransitionFragment
{
public:
    FEATSTD_TYPEDEF_SHARED_POINTER(TransitionFragment);
    FEATSTD_RTTI_DECLARATION();

    virtual ~TransitionFragment() { /*empty*/ };

    /**
     *  Called by the TransitionStrategy in regular intervals and implements frame-by-frame transition fragment behaviour. 
     *  @param deltaTime elapsed time since the last update, in milliseconds.
     */
    virtual void Update(TimeType deltaTime) = 0;

    /**
    *  Reverse this transition to its original state.
    */
    virtual void Reverse() { SetReversed(true); }

    /**
     *  Fast forward this transition to its target state immediately.
     */
    virtual void Finish() { SetFinished(true); }

    /**
     *  Apply a delay (from an Early Activate/Deactivate) to this TransitionFragment.
     *  @param lateDelay  The delay to apply to this TransitionFragment.
     */
    virtual void ApplyLateDelay(Float lateDelay) { FEATSTD_UNUSED(lateDelay); }

    /**
     *  The Trigger class holds the rule and request fragments that led to the instantiation of this specific transition fragment.
     *  @return A shared pointer to the Trigger instance.
     */
    const Trigger::SharedPointer& GetTrigger() const { return m_trigger; }

    /**
     *  Fetch the next fragment from the set.
     *  @return A shared pointer to the next transition fragment. At the end of the set the shared pointer will point to null.
     */
    const SharedPointer& GetNext() const { return m_nextFragment; }

    /**
    *  Whether or not the transition fragment has been reversed.
    *  @return true when the fragment has been reversed.
    */
    bool IsReversed() const { return m_isReversed; }

    /**
     *  Whether or not the transition fragment has finished. The DefaultTransitionStrategy uses this information to remove completed transition fragments from the current transition set.
     *  @return true when the fragment has finished execution.
     */
    bool IsFinished() const { return m_isFinished; }


    /**
     *  Get additional fragment that should be executed as one with this fragment.
     *  @return  The additional fragment that should be executed as one with this fragment.
     */
    const SharedPointer& GetAdditionalFragment() const { return m_additionalFragment; }

protected:
    TransitionFragment(const Trigger::SharedPointer& trigger, const SharedPointer& additionalFragment = SharedPointer());

    /**
    *  Sets the isReversed flag.
    *  @param isReversed The new value for the isReversed flag.
    */
    void SetReversed(bool isReversed) { m_isReversed = isReversed; }

    /**
     *  Sets the isFinished flag.
     *  @param isFinished The new value for the isFinished flag.
     */
    void SetFinished(bool isFinished) { m_isFinished = isFinished; }

    /** 
     *  Can hold a second fragment that should be executed as one with this fragment (used to group source and destination into one fragment without producing redundant code)
     */
     SharedPointer m_additionalFragment;

     /**
      *  The activation strategy from the Hint that was used to create this transition fragment.
      */
     Hint::FragmentStrategy m_activationStrategy;

     /**
      *  The deactivation strategy from the Hint that was used to create this transition fragment.
      */
     Hint::FragmentStrategy m_deactivationStrategy;

     /**
      *  The request fragment type that was used to create this transition fragment.
      */
     RequestFragment::Type m_requestFragmentType;

    /**
     *  Time after which the fragment has finished (e.g. maximum of fade/slide/scale duration, plus activation/deactivation delay)
     */
    Float m_finishTime; 

private:
    friend class TransitionStrategy;

    Trigger::SharedPointer m_trigger;
    SharedPointer m_nextFragment;

    bool m_isReversed;
    bool m_isFinished;

    CANDERA_SHARED_POINTER_DECLARATION();
};

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_FRAGMENT_H
