//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSITION_IDENTIFIER_H)
#define CANDERA_TRANSITION_IDENTIFIER_H

#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <FeatStd/Util/Optional.h>
#include <CanderaTransitions/CustomIdentifier.h>

namespace Candera {

    class Node;
    class Scene;
    class Node2D;
    class Scene2D;

namespace Transitions {

/// @addtogroup Transition
/// @{

class Identifier;
class TransitionFragmentFactory;

/**
* @brief 
*
* Currently known limitation: there is currently no suitable identification available of nodes that are accessible through composites (e.g. nodes in composites in combination with anchors) *
*/
class Identifier
{
public:

    /**
     *  
     */
    typedef enum
    {
        AnyScene2D, ///< Matches any 2D Scene
        AnyScene3D, ///< Matches any 3D Scene
        AnyNode2D,  ///< Matches any 2D Node
        AnyNode3D   ///< Matches any 3D Node
    } AnySelector;

    /**
     *  
     */
    typedef enum
    {
        Scene2DType, ///< Type for 2D Scenes
        Scene3DType  ///< Type for 3D Scenes
    } SceneType;

    /**
     *  
     */
    typedef enum
    {
        NoneBuiltInIdentifier = 0,  ///< Matches only itself
        Scene2DBuiltInIdentifier,   ///< Identifies a specific 2D Scene
        Scene3DBuiltInIdentifier,   ///< Identifies a specific 3D Scene
        Node2DBuiltInIdentifier,    ///< Identifies a specific 2D Node
        Node3DBuiltInIdentifier,    ///< Identifies a specific 3D Node
        AnyScene2DBuiltInIdentifier,///< Identifier for any 2D Scene
        AnyScene3DBuiltInIdentifier,///< Identifier for any 3D Scene
        AnyNode2DBuiltInIdentifier, ///< Identifier for any 2D Node
        AnyNode3DBuiltInIdentifier, ///< Identifier for any 3D Node
        ExternCustomIdentifier      ///< Custom Identifier implementation
    } Type;


    Identifier();
    explicit Identifier(AnySelector anySelector);
    Identifier(SceneType sceneType, Id sceneId);
    Identifier(SceneType sceneType, Id sceneId, Id nodeId);
    Identifier(const CustomIdentifier::SharedPointer& customIdentifier);
    Identifier(const Identifier& identifier);
    ~Identifier();
    bool operator==(const Identifier& rhs) const;

    /**
     *  Used to match this identifier with another.
     *
     *  The identifiers match if:
     *  - The right hand identifier is of type NoneBuiltInIdentifier.
     *  - One identifier is a wildcard (AnyNode3D/2D, AnyScene3D/2D) and the other identifies a respective object (i.e. AnyNode3D will match to a Node3D)
     *  - Both identifiers are Node identifiers and identify the same node in a scene.
     *  - Both identifiers are Scene identifiers and identify the same scene.
     *  - Both identifiers are of type ExternCustomIdentifier and the CustomIdentifier::Matches function returns true.
     *
     *  @param identifier the other identifier.
     *  @return true if the identifiers match, false otherwise.
     */
    bool Matches(const Identifier& identifier) const;

    /**
     * Fetch the Type of this Identifier.
     * @return the Type of the identifier.
     */
    Type GetType() const { return m_type; }

    /**
     *  Fetch the Candera::Id of the scene referenced by this identifier.
     *  The result is undefined when Identifier is of type ExternCustomIdentifier.
     *  @return the Candera::Id of the referenced scene.
     */
    Id GetSceneId() const;

    /**
     *  Fetch the Candera::Id of the node referenced by this identifier.
     *  The result is undefined when Identifier is of type ExternCustomIdentifier.
     *  @return the Candera::Id of the referenced scene.
     */
    Id GetNodeId() const;

private:

    void Reference();
    void Dereference();

    union Data {
        Data(Id sceneId, Id nodeId);
        Data(CustomIdentifier* customIdentifier);
        Data();

        struct DataId {
            Id m_sceneId;
            Id m_nodeId;

            bool operator==(const DataId& rhs) const {
                return ((m_sceneId == rhs.m_sceneId) && (m_nodeId == rhs.m_nodeId));
            }
        } m_id;
        CustomIdentifier* m_customIdentifier;

        bool operator==(const Data& rhs) const {
            return ((m_customIdentifier == rhs.m_customIdentifier) && (m_id == rhs.m_id));
        }

    };

    FeatStd::Initialized<Type> m_type;
    Data m_data;
};

/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_TRANSITION_IDENTIFIER_H
