//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEFAULT_TRANSITION_FACTORY_H)
#define CANDERA_DEFAULT_TRANSITION_FACTORY_H

#include <FeatStd/Util/String.h>

#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <Candera/EngineBase/Animation/AnimationPlayerListener.h>
#include <Candera/EngineBase/Animation/AnimationTimeDispatcher.h>

#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Rtti/Rtti.h>

#include <CanderaTransitions/RequestFragment.h>
#include <CanderaTransitions/Rule.h>
#include <CanderaTransitions/TransitionFragment.h>
#include <CanderaTransitions/TransitionFragmentFactory.h>
#include <CanderaTransitions/Trigger.h>

#include <Candera/System/Container/Map.h>
#include <Candera/System/Container/Vector.h>
namespace Candera {
    class Node;
    class Scene;
    class Node2D;
    class Node3D;

namespace Transitions {

/// @addtogroup Transition
/// @{
    class Trigger;
    class RequestFragment;

/**
 *  @brief Factory that handles creation of the default transition fragment instances implementing
 *  a simple fade transition which fades elements in and out by setting the alpha property.
 *
 *  The way the fade is executed is based on parameters passed in the Trigger. The fade timeline and target value are 
 *  taken from the Hint in matching Rule, the Identifier is taken from the RequestFragment.
 *  If no parameters are defined in the Hint, by default a basic switch will be performed.
 *
 *  In 3D Scenes the prerequisites for correct fading of a node are that the node and all its children must: 
 *  1. Have blending enabled via a RenderMode
 *  2. A Material assigned to them 
 *  3. A shader assigned which sets the alpha value from the diffuse material.
 *
 *  In 2D Scenes, the prerequisites for correct fading are that the node and all its children must have 
 *  an assigned Effect2D which takes the node's alpha value into account. (See TextBrushBlend or BitmapBrushColorBlend for example)
 */

class DefaultTransitionFactory : public TransitionFragmentFactory
{
public:

    /**
        *  Create an instance of the default transition fragment.
        *
        *  @param trigger A shared pointer to a Trigger instance.
        *  @return If creation was successful a shared pointer to the TransitionFragment instance, otherwise a shared pointer pointing to null.
        */
    virtual MemoryManagement::SharedPointer<TransitionFragment> Create(const MemoryManagement::SharedPointer<Trigger>& trigger);

    /**
    *  Sets the AnimationTimeDispatcher for AnimationPlayers
    *  @param timeDispatcher AnimationTimeDispatcher to set.
    */
    void SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer timeDispatcher) const;

    /**
    *  Retrieves the AnimationTimeDispatcher
    *  @return the AnimationTimeDispatcher
    */
    Animation::AnimationTimeDispatcher::SharedPointer GetAnimationTimeDispatcher() const { return GetAnimationContainer().GetAnimationTimeDispatcher(); }

    /**
     *  The default transition fragment.
     */
    class Fragment : public TransitionFragment
    {
    public:
        FEATSTD_TYPEDEF_BASE(TransitionFragment);
        FEATSTD_RTTI_DECLARATION();

        virtual void Update(TimeType deltaTime) override;

        virtual void Reverse() override;

        virtual void Finish() override;

        virtual void ApplyLateDelay(Float lateDelay) override;

        virtual ~Fragment() override;

    private:
        friend class DefaultTransitionFactory;

        Fragment(const Trigger::SharedPointer& trigger, const SharedPointer& additionalFragment);

        void SetValues(bool isRenderingEnabled, Float fadeValue, const Vector3& slideValue, const Vector3& scaleValue);

        template <class TargetType>
        void SetTargetValues(TargetType& target);

        // Generic inits and finalize used to reduce code duplication.
        template <class TargetType>
        void Init(TargetType* target, Identifier::Type targetType, const RequestFragment& requestFragment);

        template <class TargetType>
        void InitInternal(TargetType* target, const RequestFragment& requestFragment);

        template <class TargetType>
        void Finalize(TargetType* target);

        void AddAnimationTransition();
        void StartTransitionAnimation(Float elapsedTime);
        void FinishTransitionAnimation() const;

        TimeType m_elapsedTime; // Time since the fragment started execution.

        RequestFragment::Type m_requestType; // The type of request fragment (usually Activate, Deactivate) 
        Artifact m_target;                  // The target artifact.

        bool m_targetRenderingEnabled; // Should the target be rendered or not when the transition finishes.
        bool m_isActivation; // Is an Activate request fragment type or not

        // cached fading parameters/variables
        Float m_fadeVelocity;
        Float m_fadeCurrentValue;
        Float m_fadeInitialValue;
        Float m_fadeTargetValue;

        Vector2 m_fadeTimeline;

        // cached sliding parameters/variables
        Vector3 m_slideVelocity;
        Vector3 m_slideCurrentValue;
        Vector3 m_slideInitialValue;
        Vector3 m_slideTargetValue;

        Vector2 m_slideTimeline;

        // cached scaling parameters/variables
        Vector3 m_scaleVelocity;
        Vector3 m_scaleCurrentValue;
        Vector3 m_scaleInitialValue;
        Vector3 m_scaleTargetValue;

        Vector2 m_scaleTimeline;

        // cached animation parameters/variables
        Candera::Id m_animationId;
        Vector2 m_animationTimeline;
        bool m_isAnimationTimeRelative;
        bool m_isAnimationEnabled;

    }; // class DefaultTransitionFactory::Fragment

private:
    Fragment* CreateInternal(const MemoryManagement::SharedPointer<Trigger>& trigger, const RequestFragment& requestFragment, Fragment* additionalFragment) const;

    /**
    *  Handles Transition Animations
    */
    class TransitionAnimationsContainer : public Animation::AnimationPlayerListener
    {
    public:

        /**
        *  Constructor
        */
        TransitionAnimationsContainer();

        /**
        *  Destructor
        */
        virtual ~TransitionAnimationsContainer();

        /**
        *  Override pure virtual function of AnimationPlayerListener
        *  @param animationPlayer Animation Player to listen to.
        */
        virtual void OnPastEnd(Animation::AnimationPlayerBase* animationPlayer, Int32 completedIterationsCount);

        /**
        *  Override pure virtual function of AnimationPlayerListener
        *  @param animationPlayer Animation Player to listen to.
        */
        virtual void OnStartAnimation(Animation::AnimationPlayerBase* animationPlayer);

        /**
        *  Override pure virtual function of AnimationPlayerListener
        *  @param animationPlayer Animation Player to listen to.
        */
        virtual void OnFinishAnimation(Animation::AnimationPlayerBase* animationPlayer);

        /**
        *  Sets the AnimationTimeDispatcher for AnimationPlayers
        *  @param timeDispatcher AnimationTimeDispatcher to set.
        */
        void SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher);

        /**
        *  Retrieves the AnimationTimeDispatcher
        *  @return the AnimationTimeDispatcher
        */
        Animation::AnimationTimeDispatcher::SharedPointer GetAnimationTimeDispatcher() const { return m_animationTimeDispatcher; }

        /**
        *  Add AnimationPlayer to AnimationPlayer list
        *  @param animationId Candera::Id of the animation
        */
        void AddAnimationPlayer(Candera::Id animationId);

        /**
        *  Remove AnimationPlayer from AnimationPlayer list
        *  @param player AnimationPlayer to be added.
        */
        void RemoveAnimationPlayer(MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player);

        /**
        *  Add all AnimationPlayers in AnimationGroup to AnimationTimeDispatcher
        *  @param group AnimationGroupPlayer
        *  @param player AnimationPlayerBase
        */
        void AddAnimationGroupChildren(MemoryManagement::SharedPointer<Animation::AnimationGroupPlayer> groupPlayer, MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player);

        /**
        *  Return an AnimationPlayerBase identified by  animationId
        *  @param animationId Candera::Id of the AnimationPlayerBase
        *  @return the AnimationPlayerBase
        */
        MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> GetAnimationPlayer(Candera::Id animationId);

       /**
       *  Mark animation as running
       *  @param animationId Candera::Id of the AnimationPlayerBase
       */
       void SetAnimationRunning(const MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>& player);

       /**
       *  Check if animation is running
       *  @param animationId Candera::Id of the AnimationPlayerBase
       *  @return true if animation is running, false otherwise
       */
       bool IsAnimationRunning(Candera::Id animationId);

    private:
        friend class Fragment;

        struct AnimationTransitionProperties
        {
            Candera::Id m_animationId;
            MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> m_player;
            Float m_speedFactor;
            Animation::AnimationPlayer::PlayDirection m_direction;
            bool m_isRunning;

            AnimationTransitionProperties(Candera::Id animationId, MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player) :
                m_animationId(animationId),
                m_player(player),
                m_isRunning(false)
            {
                Animation::AnimationPlayer::SharedPointer animationPlayer = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(player);
                if (animationPlayer != 0) {
                    m_speedFactor = animationPlayer->GetSpeedFactor();
                    m_direction = animationPlayer->GetDirection();
                }
                else {
                    m_speedFactor = 1.0F;
                    m_direction = Animation::AnimationPlayer::Forward;
                }
            }
        };
        AnimationTransitionProperties& GetAnimationTransitionProperties(Candera::Id animationId);

        /// AnimationPlayer List
        typedef Internal::Vector<AnimationTransitionProperties> AnimationPlayerList;
        AnimationPlayerList m_animationPlayers;

        /// AnimationTimeDispatcher
        Animation::AnimationTimeDispatcher::SharedPointer m_animationTimeDispatcher;

    };

    static TransitionAnimationsContainer& GetAnimationContainer();
};


/// @}
}   // namespace Transitions
}   // namespace Candera

#endif // CANDERA_DEFAULT_TRANSITION_FRAGMENT_H
