//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DefaultTransitionFactory.h"

#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>

#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/MathDataTypes.h>

#include <Candera/EngineBase/Animation/AnimationGroupPlayer.h>
#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Scene.h>
#endif

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#endif

#include <CanderaTransitions/Request.h>
#include <CanderaTransitions/Rule.h>
#include <CanderaTransitions/TransitionTraceMapper.h>
#include <CanderaTransitions/Trigger.h>

namespace Candera {

namespace Transitions {


    /**
    * Attached to transition targets to hold data relevant to transition states. 
    * This is used to remember values that a target object had before a transition acted on it.
    */
    FEATSTD_LINT_SYMBOL(753, Candera::Transitions::TransitionProperties, "False positive, local class is referenced.")
    class TransitionProperties : public Candera::DynamicProperties::DynamicPropertyHost
    {
    public:
        static const Vector3& GetActivePosition(const CanderaObject& obj) { return obj.GetValue(CdaDynamicPropertyInstance(ActivePosition)); }
        static void SetActivePosition(CanderaObject& obj, const Vector3& value) { static_cast<void>(obj.SetValue(CdaDynamicPropertyInstance(ActivePosition), value)); }
        static bool IsActivePositionSet(const CanderaObject& obj) { return obj.IsValueSet(CdaDynamicPropertyInstance(ActivePosition)); }
        static void ClearActivePosition(CanderaObject& obj) { static_cast<void>(obj.ClearValue(CdaDynamicPropertyInstance(ActivePosition))); }

        static const Vector3& GetActiveScale(const CanderaObject& obj) { return obj.GetValue(CdaDynamicPropertyInstance(ActiveScale)); }
        static void SetActiveScale(CanderaObject& obj, const Vector3& value) { static_cast<void>(obj.SetValue(CdaDynamicPropertyInstance(ActiveScale), value)); }
        static bool IsActiveScaleSet(const CanderaObject& obj) { return obj.IsValueSet(CdaDynamicPropertyInstance(ActiveScale)); }
        static void ClearActiveScale(CanderaObject& obj) { static_cast<void>(obj.ClearValue(CdaDynamicPropertyInstance(ActiveScale))); }

        static Float GetActiveAlpha(const CanderaObject& obj) { return obj.GetValue(CdaDynamicPropertyInstance(ActiveAlpha)); }
        static void SetActiveAlpha(CanderaObject& obj, Float value) { static_cast<void>(obj.SetValue(CdaDynamicPropertyInstance(ActiveAlpha), value)); }
        static bool IsActiveAlphaSet(const CanderaObject& obj) { return obj.IsValueSet(CdaDynamicPropertyInstance(ActiveAlpha)); }
        static void ClearActiveAlpha(CanderaObject& obj) { static_cast<void>(obj.ClearValue(CdaDynamicPropertyInstance(ActiveAlpha))); }

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* /*host*/) { return 0; }

    private:
        // Required by DynamicPropertyHost.

        static const Vector3& ActivePositionDefault() 
        {
            static const Vector3 s_defaultPosition;
            return s_defaultPosition;
        }

        static const Vector3& ActiveScaleDefault() 
        {
            static const Vector3 s_defaultScale;
            return s_defaultScale;
        }

        static Float ActiveAlphaDefault()
        {
            return 0.0F;
        }

        CdaDynamicProperties(Candera::Transitions::TransitionProperties, Candera::DynamicProperties::DynamicPropertyHost);
        CdaDynamicProperty(ActivePosition, Vector3);
        CdaDynamicPropertyDefaultValue(ActivePositionDefault());
        CdaDynamicPropertyEnd();

        CdaDynamicProperty(ActiveScale, Vector3);
        CdaDynamicPropertyDefaultValue(ActiveScaleDefault());
        CdaDynamicPropertyEnd();

        CdaDynamicProperty(ActiveAlpha, Float);
        CdaDynamicPropertyDefaultValue(ActiveAlphaDefault());
        CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };
}

using namespace FeatStd;
using namespace Transitions;

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTransitions);

TransitionFragment::SharedPointer DefaultTransitionFactory::Create(const Trigger::SharedPointer& trigger)
{
    Fragment* fragment = 0;
    if ((!trigger.PointsToNull()) && (!trigger->GetRequest().PointsToNull())) {

        Fragment* secondFragment = 0;
        if (!trigger->GetRequest()->GetNext().PointsToNull()) {
            //This fragment should handle a source and destination object, add a hidden second transition fragment for the second request fragment.
            secondFragment = CreateInternal(trigger, *trigger->GetRequest()->GetNext(), 0);

            if (secondFragment == 0) {
                return TransitionFragment::SharedPointer();
            }
        }

        fragment = CreateInternal(trigger, *trigger->GetRequest(), secondFragment);
    }
    return TransitionFragment::SharedPointer(fragment);
}

void DefaultTransitionFactory::SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer timeDispatcher) const
{
    GetAnimationContainer().SetAnimationTimeDispatcher(timeDispatcher);
}

#ifdef CANDERA_3D_ENABLED
void DetectCameraRecursive(Internal::Vector<Camera*>& cameras, Node* node, UInt8 maxDepth)
{
    FEATSTD_DEBUG_ASSERT(node != 0);
    if (node != 0) {
        if (node->IsTypeOf(Camera::GetTypeId())) {
            (void)cameras.Add(static_cast<Camera*>(node));
        }

        if (maxDepth > 0) {
            for (Node* child = node->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                DetectCameraRecursive(cameras, child, --maxDepth);
            }
        }
    }
}

void GetAllCameras(Internal::Vector<Camera*>& cameras, Scene* scene)
{
    DetectCameraRecursive(cameras, static_cast<Node*>(scene), 255);
}
#endif

#ifdef CANDERA_2D_ENABLED
void DetectCameraRecursive(Internal::Vector<Camera2D*>& cameras, Node2D* node2d, UInt8 maxDepth)
{
    FEATSTD_DEBUG_ASSERT(node2d != 0);
    if (node2d != 0) {
        if (node2d->IsTypeOf(Camera2D::GetTypeId())) {
            (void)cameras.Add(static_cast<Camera2D*>(node2d));
        }

        if (maxDepth > 0) {
            for (Node2D* child = node2d->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                DetectCameraRecursive(cameras, child, --maxDepth);
            }
        }
    }
}

void GetAllCameras(Internal::Vector<Camera2D*>& cameras, Scene2D* scene2d)
{
    DetectCameraRecursive(cameras, static_cast<Node2D*>(scene2d), 255);
    FEATSTD_UNUSED2(cameras, scene2d);

}
#endif

DefaultTransitionFactory::Fragment* DefaultTransitionFactory::CreateInternal(const Trigger::SharedPointer& trigger, const RequestFragment& requestFragment, Fragment* additionalFragment) const
{
    FEATSTD_DEBUG_ASSERT(!trigger.PointsToNull()); //Guaranteed by Create function.
    Fragment* fragment = 0;

    Artifact artifactSource = Resolve(requestFragment.GetIdentifier());

    if (artifactSource.m_type != Identifier::NoneBuiltInIdentifier) {

        fragment = CANDERA_NEW(Fragment)(trigger, Fragment::SharedPointer(additionalFragment));

        if (fragment != 0) {
            switch (artifactSource.m_type) {
#ifdef CANDERA_3D_ENABLED
            case Identifier::Node3DBuiltInIdentifier:
                fragment->Init<Node>(artifactSource.m_node, Identifier::Node3DBuiltInIdentifier, requestFragment);
                break;
            case Identifier::Scene3DBuiltInIdentifier:
            {
                Internal::Vector<Camera*> cameras;
                GetAllCameras(cameras, artifactSource.m_scene);
                Fragment* cameraFragment = fragment; //add fragment to end of camera fragment list
                for (Internal::Vector<Camera*>::Iterator it = cameras.Begin(); it != cameras.End(); ++it) {
                    cameraFragment = CANDERA_NEW(Fragment)(trigger, Fragment::SharedPointer(cameraFragment)); //prepend this fragment to the list
                    cameraFragment->Init<Node>(static_cast<Node*>(*it), Identifier::Node3DBuiltInIdentifier, requestFragment);
                }
                fragment->Init<Scene>(artifactSource.m_scene, Identifier::Scene3DBuiltInIdentifier, requestFragment);
                fragment = cameraFragment; // the last created camera fragment is at the head of the list, the scene fragment at the end.
                break;
            }
#endif
#ifdef CANDERA_2D_ENABLED
            case Identifier::Node2DBuiltInIdentifier:
                fragment->Init<Node2D>(artifactSource.m_node2D, Identifier::Node2DBuiltInIdentifier, requestFragment);
                break;
            case Identifier::Scene2DBuiltInIdentifier:
            {
                FEATSTD_DEBUG_ASSERT(artifactSource.m_type == Identifier::Scene2DBuiltInIdentifier); //Ensured by Resolve and parent if.

                Internal::Vector<Camera2D*> cameras;
                GetAllCameras(cameras, artifactSource.m_scene2D);
                Fragment* cameraFragment = fragment; //add fragment to end of camera fragment list
                for (Internal::Vector<Camera2D*>::Iterator it = cameras.Begin(); it != cameras.End(); ++it) {
                    cameraFragment = CANDERA_NEW(Fragment)(trigger, Fragment::SharedPointer(cameraFragment)); //prepend this fragment to the list
                    cameraFragment->Init<Node2D>(static_cast<Node2D*>(*it), Identifier::Node2DBuiltInIdentifier, requestFragment);
                }
                fragment->Init<Scene2D>(artifactSource.m_scene2D, Identifier::Scene2DBuiltInIdentifier, requestFragment);
                fragment = cameraFragment; // the last created camera fragment is at the head of the list, the scene fragment at the end.
                break;
            }
#endif
            default:
                CANDERA_DELETE(fragment);
                fragment = 0;
                break;
            }
            
            if (0 != fragment)
            {
                // handle animation transition
                fragment->AddAnimationTransition();
            }
        }
    }

    return fragment;
}

FEATSTD_RTTI_DEFINITION(DefaultTransitionFactory::Fragment, TransitionFragment)

DefaultTransitionFactory::Fragment::Fragment(const Trigger::SharedPointer& trigger, const DefaultTransitionFactory::Fragment::SharedPointer& additionalFragment) :
Base(trigger, additionalFragment),
m_elapsedTime(0),
m_requestType(RequestFragment::Activate),
m_target(),
m_targetRenderingEnabled(true),
m_isActivation(true),
m_fadeVelocity(0.0F),
m_fadeCurrentValue(0.0F),
m_fadeInitialValue(0.0F),
m_fadeTargetValue(1.0F),
m_fadeTimeline(Math::MaxFloat(), Math::MaxFloat()),
m_slideVelocity(),
m_slideCurrentValue(),
m_slideInitialValue(),
m_slideTargetValue(),
m_slideTimeline(Math::MaxFloat(), Math::MaxFloat()),
m_scaleVelocity(),
m_scaleCurrentValue(),
m_scaleInitialValue(),
m_scaleTargetValue(),
m_scaleTimeline(Math::MaxFloat(), Math::MaxFloat()),
m_animationId(Candera::Id()),
m_animationTimeline(Math::MaxFloat(), Math::MaxFloat()),
m_isAnimationTimeRelative(false),
m_isAnimationEnabled(false)
{
    /* constructor */
}

DefaultTransitionFactory::Fragment::~Fragment() 
{
    switch (m_target.m_type) {
#ifdef CANDERA_2D_ENABLED
    case Identifier::Node2DBuiltInIdentifier:
    case Identifier::Scene2DBuiltInIdentifier:
        Finalize<Node2D>(m_target.m_node2D);
        break;
#endif
#ifdef CANDERA_3D_ENABLED
    case Identifier::Node3DBuiltInIdentifier:
    case Identifier::Scene3DBuiltInIdentifier:
        Finalize<Node>(m_target.m_node);
        break;
#endif
    default:
        /* do nothing */
        break;
    }
}

//Helper function to convert from Vector2 to Vector3, THe resulting vector's Z-component is set to zero.
Vector3 ToVector3(const Vector2& other) {
    return Vector3(other.GetX(), other.GetY(), 0.0F);
}

//Polymorphic function to pass through Vector3 to Vector3, helps to reduce code duplication for 2d/3d code.
const Vector3& ToVector3(const Vector3& other) {
    return other;
}

template <class TargetType>
void DefaultTransitionFactory::Fragment::Init(TargetType* target, Identifier::Type targetType, const RequestFragment& requestFragment)
{
#ifdef CANDERA_2D_ENABLED
    FEATSTD_DEBUG_ASSERT(target != 0); // Ensured by DefaultFragmentFactory. Init is private.

    m_target.m_type = targetType;
    m_target.m_custom = static_cast<void*>(target);

    // Initial values based on current state of object.
    m_fadeCurrentValue = target->GetAlphaValue();
    m_fadeInitialValue = target->GetAlphaValue();

    m_slideCurrentValue = ToVector3(target->GetPosition());
    m_slideInitialValue = ToVector3(target->GetPosition());

    m_scaleCurrentValue = ToVector3(target->GetScale());
    m_scaleInitialValue = ToVector3(target->GetScale());

    // ensure target is rendered to see effect.
    target->SetRenderingEnabled(true);

    InitInternal(target, requestFragment);
#else
    FEATSTD_UNUSED3(target, targetType, requestFragment);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1762, CANDERA_LINT_REASON_NONCONST)
#endif
}


template <class TargetType>
void DefaultTransitionFactory::Fragment::InitInternal(TargetType* target, const RequestFragment& requestFragment)
{
    //Asserts ensured by factory, InitInternal is private.
    FEATSTD_DEBUG_ASSERT(target != 0);
    FEATSTD_DEBUG_ASSERT(!GetTrigger().PointsToNull());
    FEATSTD_DEBUG_ASSERT(m_target.m_type != Identifier::NoneBuiltInIdentifier);
    
    m_requestFragmentType = requestFragment.GetType();
    m_isActivation = (requestFragment.GetType() == RequestFragment::Activate);
    m_targetRenderingEnabled = m_isActivation; // when the fragment finishes the isRenderingEnabled flag will be set to this value.
    m_fadeTargetValue = (m_targetRenderingEnabled ? 1.0F : 0.0F); // Target alpha value will be either 1 or 0 by default, unless specified in Hint.

    bool isSwitchTransition = true; //set to false if any parameters are set. "Switch" transition means that deactivation/activation will happen immediately without animation.

    Rule* rule = GetTrigger()->GetRule().GetPointerToSharedInstance();
    if(rule != 0) {
        const Hint& hint = rule->GetHint();
        Optional<Hint::Data> data = hint.GetDetails();

        if (data.IsSet()) {
            Float finishTime = 0.0F; //finish time will hold the highest end time from all active timelines.

            if ((*data).ActivationStrategy().IsSet()) {
                m_activationStrategy = *((*data).ActivationStrategy());
            }

            if ((*data).DeactivationStrategy().IsSet()) {
                m_deactivationStrategy = *((*data).DeactivationStrategy());
            }

            //fading
            if ((*data).Fade().IsSet()) {
                isSwitchTransition = false;
                m_fadeTimeline = ((*data).FadeTimeline().IsSet()) ? *(*data).FadeTimeline() : Vector2(0.0F, 1.0F);

                if (m_isActivation) {

                    if (TransitionProperties::IsActiveAlphaSet(*target)) {
                        m_fadeTargetValue = TransitionProperties::GetActiveAlpha(*target);
                    }
                    else {
                        m_fadeTargetValue = target->GetAlphaValue();
                    }

                    m_fadeCurrentValue = *(*data).Fade();
                    m_fadeInitialValue = m_fadeCurrentValue;
                }
                else {
                    TransitionProperties::SetActiveAlpha(*target, target->GetAlphaValue());
                    m_fadeTargetValue = *(*data).Fade();
                    m_fadeInitialValue = target->GetAlphaValue();
                }

                m_targetRenderingEnabled = (m_fadeTargetValue > 0.05F); //element can be partially faded, in that case it needs to be visible

                Float fadeDuration = (m_fadeTimeline[1] - m_fadeTimeline[0]);
                m_fadeVelocity = (m_fadeTargetValue - m_fadeCurrentValue) / ((fadeDuration != 0.0F) ? fadeDuration : Float(1.0F)); // protect from divide by zero

                finishTime = m_fadeTimeline[1];
            }

            //sliding
            if ((*data).Slide().IsSet()) {
                isSwitchTransition = false;
                m_slideTimeline = ((*data).SlideTimeline().IsSet()) ? *(*data).SlideTimeline() : Vector2(0.0F, 1.0F);

                if (m_isActivation) {

                    if (TransitionProperties::IsActivePositionSet(*target)) {
                        m_slideTargetValue = TransitionProperties::GetActivePosition(*target);
                    }
                    else {
                        m_slideTargetValue = ToVector3(target->GetPosition());
                    }
                    m_slideCurrentValue = *(*data).Slide();
                    if ((*data).IsSlideRelative()) {
                        m_slideCurrentValue += m_slideTargetValue;
                    }
                    m_slideInitialValue = m_slideCurrentValue;
                }
                else {
                    TransitionProperties::SetActivePosition(*target, ToVector3(target->GetPosition()));
                    m_slideTargetValue = *(*data).Slide();
                    if ((*data).IsSlideRelative()) {
                        m_slideTargetValue += ToVector3(target->GetPosition());
                    }
                }

                Float slideDuration = (m_slideTimeline[1] - m_slideTimeline[0]);
                m_slideVelocity = (m_slideTargetValue - m_slideCurrentValue) / ((slideDuration != 0.0F) ? slideDuration : Float(1.0F)); // protect from divide by zero

                finishTime = Math::Maximum(m_slideTimeline[1], finishTime);
            }

            //scaling
            if ((*data).Scale().IsSet()) {
                isSwitchTransition = false;
                m_scaleTimeline = ((*data).ScaleTimeline().IsSet()) ? *(*data).ScaleTimeline() : Vector2(0.0F, 1.0F);

                if (m_isActivation) {

                    if (TransitionProperties::IsActiveScaleSet(*target)) {
                        m_scaleTargetValue = TransitionProperties::GetActiveScale(*target);
                    }
                    else {
                        m_scaleTargetValue = ToVector3(target->GetScale());
                    }
                    m_scaleCurrentValue = *(*data).Scale();
                    if ((*data).IsScaleRelative()) {
                        m_scaleCurrentValue *= m_scaleTargetValue;
                    }
                    m_scaleInitialValue = m_scaleCurrentValue;
                }
                else {
                    TransitionProperties::SetActiveScale(*target, ToVector3(target->GetScale()));
                    m_scaleTargetValue = *(*data).Scale();
                    if ((*data).IsScaleRelative()) {
                        m_scaleTargetValue *= ToVector3(target->GetScale());
                    }
                }

                Float scaleDuration = (m_scaleTimeline[1] - m_scaleTimeline[0]);
                m_scaleVelocity = (m_scaleTargetValue - m_scaleCurrentValue) / ((scaleDuration != 0.0F) ? scaleDuration : Float(1.0F)); // protect from divide by zero

                finishTime = Math::Maximum(m_scaleTimeline[1], finishTime);
            }

            // set animation timeline
            if ((*data).Animation().IsSet()) {
                isSwitchTransition = false;
                m_animationId = *(*data).Animation();
                m_animationTimeline = ((*data).AnimationTimeline().IsSet()) ? *(*data).AnimationTimeline() : Vector2(0.0F, 1.0F);
                finishTime = Math::Maximum(m_animationTimeline[1], finishTime);
            }

            // activation/deactivation delay
            if (m_isActivation) {
                if ((*data).ActivationDelay().IsSet()) {
                    Float activationDelay = *(*data).ActivationDelay();
                    finishTime += activationDelay;
                }
            }
            else {
                if ((*data).DeactivationDelay().IsSet()) {
                    Float deactivationDelay = *(*data).DeactivationDelay();
                    finishTime += deactivationDelay;
                }
            }

            m_finishTime = finishTime;
        }
    }

    if (isSwitchTransition) {
        //No default transition type specified, target will be activated/deactivated in a single frame without animation.
        m_fadeTimeline = Vector2(); //set to zero
    }

}

template<class TargetType>
void DefaultTransitionFactory::Fragment::Finalize(TargetType* /*target*/) {
    /* clean up */
}

void DefaultTransitionFactory::Fragment::Update(TimeType deltaTime)
{
    FEATSTD_DEBUG_ASSERT(!IsFinished()); //ensured by TransitionStrategy.
    FEATSTD_DEBUG_ASSERT(m_target.m_type != Identifier::NoneBuiltInIdentifier); // Ensured by DefaultFragmentFactory.

    Float elapsedTime = static_cast<Float>(deltaTime + m_elapsedTime);

    if ((elapsedTime >= m_finishTime) && (!GetAnimationContainer().IsAnimationRunning(m_animationId))) {
        //fragment has finished.
        if (!IsReversed()) {
            SetValues(m_targetRenderingEnabled, m_fadeTargetValue, m_slideTargetValue, m_scaleTargetValue);
        }
        else {
            SetValues(m_targetRenderingEnabled, m_fadeInitialValue, m_slideInitialValue, m_scaleInitialValue);
            // save target values for handling future requests
            // in case of Deactivation requests, this is already handled by InitInternal()
            if (m_isActivation) {
                switch (m_target.m_type) {
#ifdef CANDERA_3D_ENABLED
                case Identifier::Node3DBuiltInIdentifier:
                case Identifier::Scene3DBuiltInIdentifier:
                    SetTargetValues<Node*>(m_target.m_node);
                    break;
#endif
#ifdef CANDERA_2D_ENABLED
                case Identifier::Node2DBuiltInIdentifier:
                case Identifier::Scene2DBuiltInIdentifier:
                    SetTargetValues<Node2D*>(m_target.m_node2D);
                    break;
#endif
                default:
                    //Do nothing.
                    break;
                }
            }

        }
        SetFinished(true);
    }
    else {
        //run fragment.
        if ((elapsedTime >= m_fadeTimeline[0]) && (elapsedTime <= m_fadeTimeline[1])) {
            m_fadeCurrentValue += m_fadeVelocity * static_cast<Float>(deltaTime);
        }

        if ((elapsedTime >= m_slideTimeline[0]) && (elapsedTime <= m_slideTimeline[1])) {
            m_slideCurrentValue += m_slideVelocity * static_cast<Float>(deltaTime);
        }

        if ((elapsedTime >= m_scaleTimeline[0]) && (elapsedTime <= m_scaleTimeline[1])) {
            m_scaleCurrentValue += m_scaleVelocity * static_cast<Float>(deltaTime);
        }
        if ((elapsedTime >= m_animationTimeline[0]) && (elapsedTime <= m_animationTimeline[1])) {
            if (m_isAnimationEnabled) {
                StartTransitionAnimation(elapsedTime);
            }
        }

        SetValues(true, m_fadeCurrentValue, m_slideCurrentValue, m_scaleCurrentValue);
    }
    m_elapsedTime = static_cast<UInt32>(elapsedTime);

    if ((!m_additionalFragment.PointsToNull()) && (!m_additionalFragment->IsFinished())) {
        m_additionalFragment->Update(deltaTime);
        SetFinished(IsFinished() && m_additionalFragment->IsFinished()); //Only set finished if partner fragment has finished too
    }
}

void DefaultTransitionFactory::Fragment::Reverse()
{
    // new finish time for fragment
    Float finishTime = 0.0F;
    Float duration = static_cast<Float>(m_elapsedTime);
    
    if (duration > m_finishTime) {
        return;
    }
    // adjust fade parameters/variables
    m_fadeTimeline[1] = duration + (duration - m_fadeTimeline[0]);
    m_fadeVelocity = (m_fadeInitialValue - m_fadeCurrentValue) / ((duration != 0.0F) ? duration : Float(1.0F)); // protect from divide by zero
    finishTime = Math::Maximum(m_fadeTimeline[1], finishTime);

    // adjust slide parameters/variables
    m_slideTimeline[1] = duration + (duration - m_slideTimeline[0]);
    m_slideVelocity = (m_slideInitialValue - m_slideCurrentValue) / ((duration != 0.0F) ? duration : Float(1.0F)); // protect from divide by zero
    finishTime = Math::Maximum(m_slideTimeline[1], finishTime);
    
    // adjust scale parameters/variables
    m_scaleTimeline[1] = duration + (duration - m_scaleTimeline[0]);
    m_scaleVelocity = (m_scaleInitialValue - m_scaleCurrentValue) / ((duration != 0.0F) ? duration : Float(1.0F)); // protect from divide by zero            
    finishTime = Math::Maximum(m_scaleTimeline[1], finishTime);

    // adjust animation 
    if (m_isAnimationEnabled) {
        // currently only AnimationPlayers can be reversed
        bool isPlayerFound = false;
        Animation::AnimationPlayer::SharedPointer player = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(GetAnimationContainer().GetAnimationPlayer(m_animationId));
        if (player != 0) {
            isPlayerFound = true;
        }
        if (!isPlayerFound) {
            // player was removed, re-add player
            AddAnimationTransition();
            player = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(GetAnimationContainer().GetAnimationPlayer(m_animationId));
        }
        if (player != 0) {
            Animation::SequenceTimeType sequenceTime = player->GetSequenceDurationMs();

            // adjunst animation timeline 
            m_animationTimeline[0] = duration + ((duration - static_cast<Float>(sequenceTime)) - m_animationTimeline[0]);
            m_animationTimeline[1] = duration + m_animationTimeline[0];
            finishTime = Math::Maximum(m_animationTimeline[1], finishTime);

            Animation::AnimationPlayer::PlayDirection playDirection = player->GetDirection();
            Animation::AnimationPlayer::PlayDirection newDirection = (playDirection == Animation::AnimationPlayer::Forward) ? Animation::AnimationPlayer::Reverse : Animation::AnimationPlayer::Forward;
            player->SetDirection(newDirection);
        }

    }

    m_finishTime = finishTime;

    SetReversed(true);

    if (!m_additionalFragment.PointsToNull()) {
        m_additionalFragment->Reverse();
    }
}

void DefaultTransitionFactory::Fragment::Finish()
{
    SetValues(m_targetRenderingEnabled, m_fadeTargetValue, m_slideTargetValue, m_scaleTargetValue);
    SetFinished(true);

    if (!m_isAnimationTimeRelative) {
        FinishTransitionAnimation();
    }
    
    if (!m_additionalFragment.PointsToNull()) {
        m_additionalFragment->Finish();
    }
}

void DefaultTransitionFactory::Fragment::ApplyLateDelay(Float lateDelay)
{
    const Vector2 lateTimeline(lateDelay, lateDelay);

    FEATSTD_DEBUG_ASSERT(!GetTrigger().PointsToNull());
    Rule* rule = GetTrigger()->GetRule().GetPointerToSharedInstance();
    if (0 != rule) {
        const Hint& hint = rule->GetHint();
        Optional<Hint::Data> data = hint.GetDetails();

        if (data.IsSet()) {
            if ((*data).Fade().IsSet()) {
                m_fadeTimeline += lateTimeline;
            }

            if ((*data).Slide().IsSet()) {
                m_slideTimeline += lateTimeline;
            }

            if ((*data).Scale().IsSet()) {
                m_scaleTimeline += lateTimeline;
            }

            if ((*data).Animation().IsSet()) {
                m_animationTimeline += lateTimeline;
            }
        }
    }

    m_finishTime += lateDelay;
}


void DefaultTransitionFactory::Fragment::SetValues(bool isRenderingEnabled, Float fadeValue, const Vector3& slideValue, const Vector3& scaleValue)
{
    switch (m_target.m_type) {
#ifdef CANDERA_3D_ENABLED
        case Identifier::Node3DBuiltInIdentifier:
        case Identifier::Scene3DBuiltInIdentifier:
        {
            Node* node = m_target.m_node;
            node->SetRenderingEnabled(isRenderingEnabled);

            if (Math::Absolute(m_fadeVelocity) > Math::EpsilonFloat()) {
                node->SetAlphaValue(fadeValue);
            }

            if (m_slideVelocity.GetSquaredLength() > Math::EpsilonFloat()) {
                node->SetPosition(slideValue);
            }

            if (m_scaleVelocity.GetSquaredLength() > Math::EpsilonFloat()) {
                node->SetScale(scaleValue);
            }

            break;
        }
#endif
#ifdef CANDERA_2D_ENABLED
        case Identifier::Node2DBuiltInIdentifier:
        case Identifier::Scene2DBuiltInIdentifier:
        {
            Node2D* node = m_target.m_node2D;
            node->SetRenderingEnabled(isRenderingEnabled);

            if (Math::Absolute(m_fadeVelocity) > Math::EpsilonFloat()) {
                node->SetAlphaValue(fadeValue);
            }

            if (m_slideVelocity.GetSquaredLength() > Math::EpsilonFloat()) {
                node->SetPosition(slideValue.GetX(), slideValue.GetY());
            }

            if (m_scaleVelocity.GetSquaredLength() > Math::EpsilonFloat()) {
                node->SetScale(scaleValue.GetX(), scaleValue.GetY());
            }

            break;
        }
#endif
        default:
            //Do nothing.
            break;
    }
}

template <class TargetType>
void DefaultTransitionFactory::Fragment::SetTargetValues(TargetType& target)
{
    TransitionProperties::SetActiveAlpha(*target, m_fadeTargetValue);
    TransitionProperties::SetActiveScale(*target, ToVector3(m_scaleTargetValue));
    TransitionProperties::SetActivePosition(*target, ToVector3(m_slideTargetValue));
}

void DefaultTransitionFactory::Fragment::AddAnimationTransition()
{
    FEATSTD_DEBUG_ASSERT(!GetTrigger().PointsToNull());
   
    Rule* rule = GetTrigger()->GetRule().GetPointerToSharedInstance();
    if (rule != 0) {
        const Hint& hint = rule->GetHint();
        Optional<Hint::Data> data = hint.GetDetails();

        //animation
        if ((*data).Animation().IsSet()) {
            m_isAnimationTimeRelative = (*data).IsAnimationTimeRelative();
            m_isAnimationEnabled = true;
            GetAnimationContainer().AddAnimationPlayer(m_animationId);
        }
    }
}

void DefaultTransitionFactory::Fragment::StartTransitionAnimation(Float elapsedTime)
{
    Animation::AnimationPlayerBase::SharedPointer player = GetAnimationContainer().GetAnimationPlayer(m_animationId);
    bool isRunning = GetAnimationContainer().IsAnimationRunning(m_animationId);

    // reverse AnimationGroupPlayers are not supported
    if (IsReversed() && (Dynamic_Cast<Animation::AnimationGroupPlayer::SharedPointer>(player) != 0)) {
        return;
    }

    if ((player != 0) && (!isRunning)) {
        // add player to time dispatcher
        if ((GetAnimationContainer().GetAnimationTimeDispatcher() != 0) && (player->GetTimeDispatcher() == 0)){
            static_cast<void>(GetAnimationContainer().GetAnimationTimeDispatcher()->AddPlayer(player));
        }

        Animation::AnimationGroupPlayer::SharedPointer groupPlayer = Dynamic_Cast<Animation::AnimationGroupPlayer::SharedPointer>(player);
        if (groupPlayer != 0){
            GetAnimationContainer().AddAnimationGroupChildren(groupPlayer, groupPlayer->GetFirstSuccessor(MemoryManagement::SharedPointer<Animation::AnimationPlayer>(0), Animation::AnimationGroupPlayer::WithPrevious));
        }

        // adjust speed for animation
        if (!m_isAnimationTimeRelative) {
            Float timeToFinish = m_animationTimeline[1] - elapsedTime;

            // adjust AnimationPlayers
            Animation::AnimationPlayer::SharedPointer animationPlayer = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(player);
            if (animationPlayer != 0){
                if (timeToFinish != 0.0F){
                    Float speedFactor = static_cast<Float>(animationPlayer->GetSequenceDurationMs()) / timeToFinish;
                    animationPlayer->SetSpeedFactor(speedFactor);
                }
                static_cast<void>(animationPlayer->Start());
            }

            // adjust AnimationGroupPlayers
            if ((groupPlayer != 0) && (!IsReversed())) {
                static_cast<void>(player->Start());
                static_cast<void>(groupPlayer->Finish(static_cast<Animation::WorldTimeType>(timeToFinish)));
            }
        }
        else {
            static_cast<void>(player->Start());
        }
    }
}

void DefaultTransitionFactory::Fragment::FinishTransitionAnimation() const
{
    Animation::AnimationPlayerBase::SharedPointer player = GetAnimationContainer().GetAnimationPlayer(m_animationId);
    if ((player != 0) && player->IsEnabled()) {
        static_cast<void>(player->Finish(1));
    }
}


DefaultTransitionFactory::TransitionAnimationsContainer::TransitionAnimationsContainer() :
m_animationTimeDispatcher(0)
{
}

DefaultTransitionFactory::TransitionAnimationsContainer::~TransitionAnimationsContainer()
{
    for (Internal::Vector<AnimationTransitionProperties>::Iterator it = m_animationPlayers.Begin(); it != m_animationPlayers.End(); ++it) {
        MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> animationPlayer = (*it).m_player;
        if (animationPlayer != 0) {
            static_cast<void>(animationPlayer->Stop());
            static_cast<void>(animationPlayer->RemoveAnimationPlayerListener(this));
        }
    }
}

DefaultTransitionFactory::TransitionAnimationsContainer& DefaultTransitionFactory::GetAnimationContainer()
{
    static DefaultTransitionFactory::TransitionAnimationsContainer s_animationContainer;
    return s_animationContainer;
}


void DefaultTransitionFactory::TransitionAnimationsContainer::SetAnimationTimeDispatcher(Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher)
{
    m_animationTimeDispatcher = timeDispatcher;
}

void DefaultTransitionFactory::TransitionAnimationsContainer::AddAnimationPlayer(Candera::Id animationId)
{
    // retrieve animation
    AssetProvider* assetProvider = ContentLoader::GetInstance().GetAssetProvider();
    MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> animationPlayerBase = assetProvider->GetAnimationById(animationId);
    
    FEATSTD_DEBUG_ASSERT(animationPlayerBase != 0);

    for (SizeType index = 0; index < m_animationPlayers.Size(); ++index) {
        if (m_animationPlayers[index].m_player == animationPlayerBase) {
            return;
        }
    }

    AnimationTransitionProperties atp(animationId, animationPlayerBase);
    if (m_animationPlayers.Add(atp)) {
        static_cast<void>(animationPlayerBase->AddAnimationPlayerListener(this));
    }
}

void DefaultTransitionFactory::TransitionAnimationsContainer::RemoveAnimationPlayer(MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player)
{
    FEATSTD_DEBUG_ASSERT(player != 0);
    for (SizeType index = 0; index < m_animationPlayers.Size(); ++index) {
        if (m_animationPlayers[index].m_player == player) {
            // restore AnimationPlayer properties
            Animation::AnimationPlayer::SharedPointer animationPlayer = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(player);
            if (animationPlayer != 0) {
                animationPlayer->SetSpeedFactor(m_animationPlayers[index].m_speedFactor);
                animationPlayer->SetDirection(m_animationPlayers[index].m_direction);
            }

            static_cast<void>(m_animationPlayers[index].m_player->RemoveAnimationPlayerListener(this));
            static_cast<void>(m_animationPlayers.Remove(index));
            break;
        }
    }
}

void DefaultTransitionFactory::TransitionAnimationsContainer::AddAnimationGroupChildren(MemoryManagement::SharedPointer<Animation::AnimationGroupPlayer> groupPlayer, MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player)
{
    if (player != 0) {
        if ((m_animationTimeDispatcher != 0) && (player->GetTimeDispatcher() == 0)) {
            static_cast<void>(m_animationTimeDispatcher->AddPlayer(player));
        }

        Animation::AnimationGroupPlayer::SharedPointer groupAnimation = Dynamic_Cast<Animation::AnimationGroupPlayer::SharedPointer>(player);
        if (groupAnimation != 0) {
            AddAnimationGroupChildren(groupAnimation, groupAnimation->GetFirstSuccessor(MemoryManagement::SharedPointer<Animation::AnimationPlayer>(0), Animation::AnimationGroupPlayer::WithPrevious));
        }
        AddAnimationGroupChildren(groupPlayer, groupPlayer->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::WithPrevious));
        AddAnimationGroupChildren(groupPlayer, groupPlayer->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::AfterPrevious));
        AddAnimationGroupChildren(groupPlayer, groupPlayer->GetNext(player));
    }
}


MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> DefaultTransitionFactory::TransitionAnimationsContainer::GetAnimationPlayer(Candera::Id animationId)
{
    for (Internal::Vector<AnimationTransitionProperties>::Iterator it = m_animationPlayers.Begin(); it != m_animationPlayers.End(); ++it) {
        if ((*it).m_animationId == animationId) {
            return (*it).m_player;
        }
    }

    return MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>();
}

void DefaultTransitionFactory::TransitionAnimationsContainer::SetAnimationRunning(const MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>& player)
{
    for (Internal::Vector<AnimationTransitionProperties>::Iterator it = m_animationPlayers.Begin(); it != m_animationPlayers.End(); ++it) {
        if ((*it).m_player == player) {
            (*it).m_isRunning = true;
        }
    }
}

bool DefaultTransitionFactory::TransitionAnimationsContainer::IsAnimationRunning(Candera::Id animationId)
{
    for (Internal::Vector<AnimationTransitionProperties>::Iterator it = m_animationPlayers.Begin(); it != m_animationPlayers.End(); ++it) {
        if ((*it).m_animationId == animationId) {
            return (*it).m_isRunning;
        }
    }
    return false;
}

void DefaultTransitionFactory::TransitionAnimationsContainer::OnPastEnd(Animation::AnimationPlayerBase* animationPlayer, Int32 completedIterationsCount)
{
    FEATSTD_UNUSED(completedIterationsCount);
    MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player(animationPlayer);

    RemoveAnimationPlayer(player);
}

void DefaultTransitionFactory::TransitionAnimationsContainer::OnStartAnimation(Animation::AnimationPlayerBase* animationPlayer)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(64, "False positive because here there is no interaction between integer and pointer types.")
    GetAnimationContainer().SetAnimationRunning(MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>(animationPlayer));
}

void DefaultTransitionFactory::TransitionAnimationsContainer::OnFinishAnimation(Animation::AnimationPlayerBase* animationPlayer)
{
    MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player(animationPlayer);
    RemoveAnimationPlayer(player);
}

}   // namespace Candera
