//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_SCRIPTSYSTEM_H
#define CANDERA_SCRIPTING_SCRIPTSYSTEM_H

#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <CanderaScripting/ObjectReferenceSystem.h>
#include <CanderaScripting/ScriptComponent.h>
#include <FeatStd/Event/EventSource.h>

namespace Candera {

namespace Scripting {

/** @addtogroup Scripting
 *  @{
 */

class ScriptEvent;
class ScriptParameters;

/**
 * @brief  ScriptSystem is a ComponentSystem that serves as a base class for handling a concrete implementation of
 *         a scripting language and its virtual machine which runs the scripts that are provided by ScriptComponents.
 *
 *         The script system handles public script parameters and certain script component properties (e.g. the
 *         'enabled' flag) according to its mode of operation, which is either running/paused or stopped.
 *         These parameters/properties have default values which are stored in the script component and passed
 *         to the instances of these parameters/properties in the virtual machine when the script system is started.
 *         Alteration of the values (by execution of scripts or native code using SetComponentLiveParameter functions)
 *         of the parameters/properties in the virtual machine do not affect the default values stored in the script
 *         component. That way, the script system can be stopped and restarted, and exhibit a deterministic behavior
 *         of execution.
 *         In a running system (when the virtual machine is 'live'), changing default values of parameters of a script
 *         component does not affect their corresponding values in the virtual machine until the script system is either
 *         stopped and restarted, or the script component is detached and reattached. That way, it is possible to create,
 *         configure, and attach new script components independently from the current mode of operation of the script
 *         system.
 */

class ScriptSystem : public Candera::EntityComponentSystem::ComponentSystem<ScriptComponent, ScriptEntity>
{
    typedef Candera::EntityComponentSystem::ComponentSystem<ScriptComponent, ScriptEntity> Base;

public:
    /**
     *  Initialize the script system.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool Init() { return true; }

    /**
     *  Shut the script system down.
     */
    virtual void ShutDown() {}

    /**
     *  Start executing the script system's Update function.
     */
    virtual void Start();

    /**
     *  Stop executing the script system's Update function, and reset the system.
     */
    virtual void Stop();

    /**
     *  Enable/Disable executing the script system's Update function.
     */
    virtual void Pause();

    /**
     *  Check if the system is paused.
     *  @return  True, if the system is paused. False, otherwise.
     */
    bool IsPaused() const { return ((!IsEnabled()) && (!IsStopped())); }

    /**
     *  Check if the system is stopped.
     *  @return  True, if the system is stopped. False, otherwise.
     */
    bool IsStopped() const { return m_isStopped; }

    /**
     *  Reset the script system
     */
    virtual void Reset();

    /**
     *  Set the priority of the given component.
     *  @param handle    The handle of the component to set the priority for.
     *  @param priority  The priority to set.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentPriority(Handle handle, Float priority);

    /**
     *  Get the priority of the given component.
     *  @param handle    The handle of the component to get the priority for.
     *  @param priority  A reference to return the priority in.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool GetComponentPriority(Handle handle, Float& priority) const;

    /**
     *  Enable or disable the given component. If the component is attached and the scripting system is
     *  running, the 'live' flag from the component's Lua instance (from the scripting VM) is set.
     *  Otherwise the flag from the ScriptComponent itself is set.
     *  @param handle     The handle of the component to enable or disable.
     *  @param isEnabled  True to enable the component. False to disable it.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentEnabled(Handle handle, bool isEnabled);

    /**
     *  Get the enabled flag from the given component. If the component is attached and the scripting
     *  system is running, the 'live' flag from the component's Lua instance (from the scripting VM) is returned.
     *  Otherwise the flag from the component is returned, which is the same as ScriptComponent::IsEnabled.
     *  @param handle     The handle of the component to enable or disable.
     *  @param isEnabled  A reference to return the enabled flag in.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool GetComponentEnabled(Handle handle, bool& isEnabled) const;

    /**
     *  Set the script of the given component.
     *  @param handle  The handle of the component to set the script for.
     *  @param script  The script to set.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentScript(Handle handle, MemoryManagement::SharedPointer<Script> script);

    /**
     *  Notify the system that the contents of an existing script have been changed.
     *  @param script  The script that has been changed.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnScriptChanged(const Script* script) { FEATSTD_UNUSED(script); return true; }

    /**
     *  Add a default parameter of type double (i.e. ScriptComponent::Number) with the given value to the
     *  component identified by the handle.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to add the parameter to.
     *  @param parameterName   The name of the parameter to add.
     *  @param parameterValue  The value of the parameter to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const;

    /**
     *  Add a default parameter of type int (i.e. ScriptComponent::Integer) with the given value to the
     *  component identified by the handle.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to add the parameter to.
     *  @param parameterName   The name of the parameter to add.
     *  @param parameterValue  The value of the parameter to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const;

    /**
     *  Add a default parameter of type bool (i.e. ScriptComponent::Boolean) with the given value to the
     *  component identified by the handle.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to add the parameter to.
     *  @param parameterName   The name of the parameter to add.
     *  @param parameterValue  The value of the parameter to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const;

    /**
     *  Add a default parameter of type string (i.e. ScriptComponent::String) with the given value to the
     *  component identified by the handle.
     *  ATTENTION: The given string is not be stored, but a copy of it is created and stored instead.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to add the parameter to.
     *  @param parameterName   The name of the parameter to add.
     *  @param parameterValue  The value of the parameter to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const;

    /**
     *  Add a default parameter of type CanderaObject pointer (i.e. ScriptComponent::ObjectReference) with the given value
     *  to the component identified by the handle.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to add the parameter to.
     *  @param parameterName   The name of the parameter to add.
     *  @param parameterValue  The value of the parameter to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const;

    /**
     *  Set the default double (i.e. ScriptComponent::Number) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const;

    /**
     *  Set the default int (i.e. ScriptComponent::Integer) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const;

    /**
     *  Set the default bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const;

    /**
     *  Set the default string (i.e. ScriptComponent::String) value of the parameter identified by its name
     *  and the given component.
     *  ATTENTION: The given string is not stored, but a copy of it is created and stored instead.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const;

    /**
     *  Set the default CanderaObject pointer (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const;

    /**
     *  Get the default double (i.e. ScriptComponent::Number) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const;

    /**
     *  Get the default int (i.e. ScriptComponent::Integer) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const;

    /**
     *  Get the default bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name
     *  and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const;

    /**
     *  Get the default string (i.e. ScriptComponent::String) value of the parameter identified by its name
     *  and the given component.
     *  ATTENTION: The returned string is owned by the component, do not delete it.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterString(Handle handle, const Char* parameterName, const Char*& parameterValue) const;

    /**
     *  Get the default CanderaObject pointer (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle           The handle of the component to get the parameter from.
     *  @param parameterName    The name of the parameter to get the value for.
     *  @param parameterValue   A reference to where the value of the parameter will be stored.
     *  @param parameterTypeId  A reference to where the TypeId of the parameterValue will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject*& parameterValue,
        TypeId& parameterTypeId) const;

    /**
     *  Get the type of the parameter identified by its name and the given component
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running. 
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle         The handle of the component to get the parameter type from.
     *  @param parameterName  The name of the parameter to get the type for.
     *  @return  The ScriptComponent::ParameterType of the parameter identified by the name, or ScriptComponent::None if
     *           the handle could not be resolved, or no parameter for the given name and component exists.
     */
    virtual ScriptComponent::ParameterType GetComponentLiveParameterType(Handle handle, const Char* parameterName) const;

    /**
     *  Get the double (i.e. ScriptComponent::Number) value of the parameter identified by its name and the
     *  given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const;

    /**
     *  Get the int (i.e. ScriptComponent::Integer) value of the parameter identified by its name and the
     *  given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const;

    /**
     *  Get the bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name and the
     *  given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const;

    /**
     *  Get the string (i.e. ScriptComponent::String) value of the parameter identified by its name and the
     *  given component from the script system's virtual machine.
     *  ATTENTION: The returned string is a copy, and owned by the caller. Thus, the caller is responsible for
     *  deleting it after use.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to get the parameter from.
     *  @param parameterName   The name of the parameter to get the value for.
     *  @param parameterValue  A reference to where the value of the parameter will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterString(Handle handle, const Char* parameterName, Char*& parameterValue) const;

    /**
     *  Get the CanderaObject pointer (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle           The handle of the component to get the parameter from.
     *  @param parameterName    The name of the parameter to get the value for.
     *  @param parameterValue   A reference to where the value of the parameter will be stored.
     *  @param parameterTypeId  A reference to where the TypeId of the parameterValue will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    bool GetComponentLiveParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject*& parameterValue,
        TypeId& parameterTypeId) const;

    /**
     *  Set the Double (i.e. ScriptComponent::Number) value of the parameter identified by its name and the
     *  given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const;

    /**
     *  Set the int (i.e. ScriptComponent::Integer) value of the parameter identified by its name and the
     *  given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const;

    /**
     *  Set the bool (i.e. ScriptComponent::Boolean) value of the parameter identified by its name and the
     *  given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const;

    /**
     *  Set the string (i.e. ScriptComponent::String) value of the parameter identified by its name and the
     *  given component in the script system's virtual machine.
     *  ATTENTION: The given string is not stored, but a copy of it is created and stored instead.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const;

    /**
     *  Set the CanderaObject pointer (i.e. ScriptComponent::ObjectReference) value of the parameter identified by its
     *  name and the given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle          The handle of the component to set the parameter for.
     *  @param parameterName   The name of the parameter to set the value for.
     *  @param parameterValue  The value of the parameter to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    bool SetComponentLiveParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const;

    /**
     *  Call a method of a script component using the given parameters.
     *  @param handle      The handle of the component for which to call the method.
     *  @param methodName  The name of the method to call.
     *  @param in          The parameters that are passed to the method.
     *  @param out         The expected return values of the method.
     *  @return  True, if the method was successfully called using the 'in' parameters, and the return values were
     *           successfully stored in the 'out' parameters.
     *           False, if the method does not exist,
     *           or threw a runtime error,
     *           or the given 'in'/'out' parameters did not match the actual parameter/return value types of the method in the script,
     *           or the script system is stopped.
     */
    virtual bool CallComponentMethod(Handle handle, const Char* methodName, const ScriptParameters& in, ScriptParameters& out) const;

    /**
     *  Get the last compile error message set by the scripting VM.
     *  @return  A pointer to the string containing the last compile error message.
     */
    const Char* GetLastCompileErrorMessage() const { return m_lastCompileErrorMessage; }

    /**
     *  Clears the last compile error message set by the scripting VM.
     */
    void ClearLastCompileErrorMessage() const { FEATSTD_DELETE_ARRAY(m_lastCompileErrorMessage); m_lastCompileErrorMessage = 0; }

    /**
     *  Adds an EventListener to the script system. Event listeners receive OnEvent calls when ScriptEvents are dispatched.
     *  @param eventListener  The event listener to be added.
     *  @return  True, if the listener could be added successfully. False, otherwise.
     */
    bool AddScriptEventListener(FeatStd::EventListener* eventListener);

    /**
     *  Removes an EventListener from the script system.
     *  @param eventListener  The event listener to be removed.
     *  @param waitForListenerRelease
     *  @return  True, if the listener could be removed successfully. False, otherwise.

     */
    bool RemoveScriptEventListener(FeatStd::EventListener* eventListener, bool waitForListenerRelease = true);

    /**
     *  Dispatch the given script event to all event listeners of the script system.
     *  Script event listeners can be native code EventListeners and/or scripts that were registered as listeners.
     *  @param scriptEvent  The script event to be dispatched.
     *  @return  True, if the scriptEvent was dispatched.
     *           False, if the script system is stopped, or the scriptEvent's parameters are empty.
     */
    virtual bool DispatchScriptEvent(const ScriptEvent& scriptEvent);

    /**
     * @brief Iterator to get parameters with their type of the given component.
     */
    class ComponentDefaultParameterIterator
    {
    public:
        ComponentDefaultParameterIterator(const ScriptSystem* scriptSystem, Handle handle);

        bool Get(const Char*& parameterName, ScriptComponent::ParameterType& parameterType) const;
        void Next() const { ++m_index; }

    private:
        const ScriptComponent* m_component;
        mutable Int m_index;
    };

    FEATSTD_RTTI_DECLARATION();

protected:
    /**
     *  Constructor
     */
    ScriptSystem();

    /**
     *  Destructor
     */
    virtual ~ScriptSystem();

    /**
     *  FixedUpdate callback, which is called 1/FixedTimeStep times per second by the UpdateSystem.
     */
    virtual void FixedUpdate() {};

    /**
     *  Update callback, which is called once per frame by the UpdateSystem.
     *  @param applicationTime  The time in seconds that has passed since the start of the application.
     *  @param deltaTime        The time in seconds that has passed since the last frame.
     */
    virtual void UpdateWithSeconds(Double applicationTime, Double deltaTime);

    /**
     *  LateUpdate callback, which is called by the UpdateSystem after FixedUpdate and Update.
     */
    virtual void LateUpdate() {};

    /**
     *  Callback after a component was detached.
     *  @param handle  The handle of the component that was detached.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnDetachComponent(Handle handle) override;

    /**
     *  Callback for derived classes to handle priority changes of components.
     *  @param handle    The handle of the component to set the priority for.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentPriority(Handle handle) { FEATSTD_UNUSED(handle); return true; }

    /**
     *  Callback for derived classes to handle enabled flag changes of components.
     *  @param handle     The handle of the component to set the enabled flag for.
     *  @param isEnabled  Enable the component (true), or disabled it (false).
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentEnabled(Handle handle, bool isEnabled) { FEATSTD_UNUSED(handle); FEATSTD_UNUSED(isEnabled); return true; }

    /**
     *  Callback for derived classes to handle enabled flag changes of components.
     *  @param handle     The handle of the component to set the priority for.
     *  @param isEnabled  A reference to return the enabled flag in.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnGetComponentEnabled(Handle handle, bool& isEnabled) const { FEATSTD_UNUSED(handle); FEATSTD_UNUSED(isEnabled); return false; }

    /**
     *  Callback for derived classes to handle script changes of components.
     *  @param handle  The handle of the component to set the script for.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnSetComponentScript(Handle handle) { FEATSTD_UNUSED(handle); return true; }

    /**
     *  Remove a parameter from the component by name.
     *  @param handle         The handle of the component to remove the parameter from.
     *  @param parameterName  The name of the parameter to be removed.
     *  @return  True, if the parameter identified by the name could be removed or if there is no such parameter.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool RemoveComponentParameter(Handle handle, const Char* parameterName) const;

    /**
     *  Adds a script component's script instance as a script event listener to the script system.
     *  The script instance receives OnEvent calls when ScriptEvents are dispatched.
     *  @param handle  The handle of the component of the script instance to add as a script event listener.
     *  @return  True, if the handle was successfully added as a script event listener. False, otherwise.
     */
    bool AddScriptEventListener(Handle handle);

    /**
     *  Remove a script component's script instance as a script event listener from the script system.
     *  @param handle  The handle of the component of the script instance to remove as a script event listener.
     *  @return  True, if the handle was successfully removed as a script event listener. False, otherwise.
     */
    bool RemoveScriptEventListener(Handle handle);

    /**
     *  Called by Update when it is being executed for the first time.
     */
    virtual void CallScriptInit() { m_isFirstUpdate = false; };

    /**
     *  Set the ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component in the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle           The handle of the component to set the parameter for.
     *  @param parameterName    The name of the parameter to set the value for.
     *  @param parameterValue   The value of the parameter to be set.
     *  @param parameterTypeId  The TypeId of the parameterValue to be set.
     *  @return  True, if the value of the parameter identified by the name could be set in the script system's
     *           virtual machine.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool SetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterTypeId) const;

    /**
     *  Get the ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter identified by
     *  its name and the given component from the script system's virtual machine.
     *  Note: 'Live' refers to the parameter when the scripting virtual machine is being executed, i.e. running.
     *  Changing a live parameter does not affect the default value of the parameter that is being restored when the
     *  script system is stopped or reset.
     *  @param handle           The handle of the component to get the parameter from.
     *  @param parameterName    The name of the parameter to get the value for.
     *  @param parameterValue   A reference to where the value of the parameter will be stored.
     *  @param parameterTypeId  A reference to where the TypeId of the parameterValue will be stored.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or the system was not initialized, or no parameter
     *           for the given name and component exists.
     */
    virtual bool GetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId& parameterTypeId) const;

    /**
     *  Add a default parameter of type ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) with the
     *  given value to the component identified by the handle.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle                The handle of the component to add the parameter to.
     *  @param parameterName         The name of the parameter to add.
     *  @param parameterValue        The value of the parameter to be added.
     *  @param parameterValueTypeId  The TypeId of the parameterValue to be added.
     *  @return  True, if the parameter with the given name and value could be added to the component.
     *           False, if the handle could not be resolved, or an error occurred.
     */
    bool AddComponentParameterObjectReference(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterValueTypeId) const;

    /**
     *  Set the default ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter
     *  identified by its name and the given component.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle                The handle of the component to set the parameter for.
     *  @param parameterName         The name of the parameter to set the value for.
     *  @param parameterValue        The value of the parameter to be set.
     *  @param parameterValueTypeId  The TypeId of the parameterValue to be added.
     *  @return  True, if the value of the parameter identified by the name could be set.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool SetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterValueTypeId) const;

    /**
     *  Get the default ObjectReferenceSystem::Handle (i.e. ScriptComponent::ObjectReference) value of the parameter
     *  identified by its name and the given component, and sync the type of the concrete object with the type of the 
     *  default parameter if possible.
     *  Node: 'Default' refers to value that the parameter is initialized with. The parameter is reset to
     *  this value, when the script system is stopped or reset.
     *  @param handle                The handle of the component to get the parameter from.
     *  @param parameterName         The name of the parameter to get the value for.
     *  @param parameterValue        A reference to where the value of the parameter will be stored.
     *  @param parameterValueTypeId  The TypeId of the parameterValue.
     *  @return  True, if the value of the parameter identified by the name could be returned in parameterValue.
     *           False, if the handle could not be resolved, or no parameter for the given name and component
     *           exists.
     */
    bool GetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName,
        Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId parameterValueTypeId) const;

    /**
     *  Set the ObjectReferenceSystem::Handle at the index in the given parameters.
     *  @param parameters       The ScriptParameters container to set the object reference in.
     *  @param objectReference  The ObjectReferenceSystem::Handle to set.
     *  @param typeId           The type of the object that is referenced by the handle.
     *  @param index            The index in to container to set the handle at.
     *  @return  True, if the handle could be set. False, otherwise.
     */
    bool SetObjectReference(ScriptParameters& parameters, Internal::ObjectReferenceSystem::Handle objectReference,
        TypeId typeId, SizeType index) const;

    /**
     *  Get the ObjectReferenceSystem::Handle from the given parameters at the index.
     *  @param parameters       The ScriptParameters container to get the object reference from.
     *  @param objectReference  A reference to an ObjectReferenceSystem::Handle to return the handle in.
     *  @param index            The index in to container to get the handle from.
     *  @return  True, if the handle could be retrieved. False, otherwise.
     */
    bool GetObjectReference(const ScriptParameters& parameters, Internal::ObjectReferenceSystem::Handle& objectReference,
        SizeType index) const;

    /**
     *  Get the ObjectReferenceSystem::Handle from a pointer to an object.
     *  @param object  The pointer to the object to get the ObjectReferenceSystem::Handle for.
     *  @return  The ObjectReferenceSystem::Handle for the object.
     */
    Internal::ObjectReferenceSystem::Handle GetObjectReference(CanderaObject* object) const;

    /**
     *  Resolve the given ObjectReferenceSystem::Handle into a pointer.
     *  @param objectReference  The ObjectReferenceSystem::Handle to resolve into a pointer.
     *  @return  The pointer to an CanderaObject that is referenced by the handle, or 0 if the handle is invalid.
     */
    CanderaObject* GetObjectPointer(Internal::ObjectReferenceSystem::Handle objectReference) const;

    /**
     *  Pointer to hold the last compile error message set by the scripting VM.
     */
    mutable Char* m_lastCompileErrorMessage;

    Candera::Internal::Vector<Handle> m_scriptEventListeners;

    Internal::ObjectReferenceSystem* m_objectReferenceSystem;

private:
    FeatStd::EventSource m_scriptEventSource;

    bool m_isFirstUpdate;
    bool m_isStopped;

    bool AddComponentParameter(Handle handle, const Char* parameterName, const ScriptComponent::ParameterType parameterType,
        TypeId parameterTypeId = TypeId()) const;

    template<typename T, typename Accessor>
    bool AccessComponentParameter(Handle handle, const Char* parameterName, T parameterValue,
        const ScriptComponent::ParameterType parameterType, Accessor accessor) const;

    // Parameter type: Double
    struct DoubleGetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, Double& parameterValue) const {
            parameterValue = parameter.m_value.m_double;
        }
    };

    struct DoubleSetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, Double parameterValue) const {
            parameter.m_value.m_double = parameterValue;
        }
    };

    // Parameter type: Integer
    struct IntegerGetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, Int64& parameterValue) const {
            parameterValue = parameter.m_value.m_integer;
        }
    };

    struct IntegerSetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, Int64 parameterValue) const {
            parameter.m_value.m_integer = parameterValue;
        }
    };

    // Parameter type: Boolean
    struct BooleanGetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, bool& parameterValue) const {
            parameterValue = parameter.m_value.m_boolean;
        }
    };

    struct BooleanSetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, bool parameterValue) const {
            parameter.m_value.m_boolean = parameterValue;
        }
    };

    // Parameter type: String
    struct StringGetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, const Char*& parameterValue) const {
            parameterValue = parameter.m_value.m_string;
        }
    };

    struct StringSetter
    {
        void operator()(ScriptComponent::ParameterInfo::Parameter& parameter, const Char* parameterValue) const {
            CANDERA_DELETE_ARRAY(parameter.m_value.m_string);
            parameter.m_value.m_string = parameterValue;
        }
    };
};

/** @} */ // end of Scripting

template<typename T, typename Accessor>
bool ScriptSystem::AccessComponentParameter(Handle handle, const Char* parameterName, T parameterValue,
    const ScriptComponent::ParameterType parameterType, Accessor accessor) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        ScriptComponent::ParameterInfo::Parameter& parameter = parameters[i].m_parameter;
        if (parameterType == parameter.m_type) {
            if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
                accessor(parameter, parameterValue);
                return true;
            }
        }
    }

    return false;
}

} // namespace Scripting

} // namespace Candera

#endif
