//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ScriptSystem.h"

#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/UpdateSystem/UpdateSystem.h>
#include <CanderaScripting/ScriptEvent.h>
#include <CanderaScripting/ScriptParameters.h>

namespace Candera {

namespace Scripting {

FEATSTD_RTTI_DEFINITION(ScriptSystem, Base)

using namespace EntityComponentSystem;

ScriptSystem::ScriptSystem()
    :
    m_lastCompileErrorMessage(0),
    m_objectReferenceSystem(0),
    m_isFirstUpdate(true),
    m_isStopped(true)
{
    SetEnabled(false);

    UpdateSystem* updateSystem = EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        UpdateSystem::Handle updateComponent = updateSystem->CreateComponent();

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
        UpdateSystem::Delegate fixedUpdateDelegate = UpdateSystem::Delegate::Update<ScriptSystem, &ScriptSystem::FixedUpdate>();
        bool success = updateSystem->SetComponentFixedUpdateDelegate(updateComponent, fixedUpdateDelegate);

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::UpdateWithSeconds, "False positive because the template arguments match.")
        UpdateSystem::Delegate updateDelegate = UpdateSystem::Delegate::UpdateWithSeconds<ScriptSystem, &ScriptSystem::UpdateWithSeconds>();
        success = updateSystem->SetComponentUpdateDelegate(updateComponent, updateDelegate) && success;

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
        UpdateSystem::Delegate lateUpdateDelegate = UpdateSystem::Delegate::Update<ScriptSystem, &ScriptSystem::LateUpdate>();
        success = updateSystem->SetComponentLateUpdateDelegate(updateComponent, lateUpdateDelegate) && success;

        success = updateSystem->AttachComponent(updateComponent, this) && success;
        FEATSTD_DEBUG_ASSERT(success);
        FEATSTD_UNUSED(success);
    }

    m_objectReferenceSystem = EntitySystem::Create<Internal::ObjectReferenceSystem>();
    FEATSTD_DEBUG_ASSERT(0 != m_objectReferenceSystem);
}

ScriptSystem::~ScriptSystem()
{
    bool success = EntitySystem::Destroy<Internal::ObjectReferenceSystem>();
    FEATSTD_DEBUG_ASSERT(success);
    FEATSTD_UNUSED(success);
    m_objectReferenceSystem = 0;

    FEATSTD_DELETE_ARRAY(m_lastCompileErrorMessage);
    m_lastCompileErrorMessage = 0;
}

void ScriptSystem::Start()
{
    if (IsEnabled()) {
        return;
    }

    if (IsPaused()) {
        Pause();
        return;
    }

    m_isStopped = false;
    SetEnabled(true);

    UpdateSystem* updateSystem = EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        updateSystem->ResetApplicationTime();
    }

    Reset();
}

void ScriptSystem::Stop()
{
    if (IsPaused()) {
        Pause(); // Unpause before stopping.
    }

    m_isStopped = true;
    SetEnabled(false);
}

void ScriptSystem::Pause()
{
    if (IsStopped()) {
        Start();
        return;
    }

    SetEnabled(IsPaused());
    
    UpdateSystem* updateSystem = EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        updateSystem->PauseApplicationTime();
        FEATSTD_DEBUG_ASSERT(updateSystem->IsApplicationTimePaused() == IsPaused());
    }
}

void ScriptSystem::Reset()
{
    m_scriptEventListeners.Clear();
    m_isFirstUpdate = true;
}

bool ScriptSystem::SetComponentPriority(Handle handle, Float priority)
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (component->m_priority != priority) {
        component->m_priority = priority;
        return OnSetComponentPriority(handle);
    }

    return true;
}

bool ScriptSystem::GetComponentPriority(Handle handle, Float& priority) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    priority = component->GetPriority();
    return true;
}

bool ScriptSystem::SetComponentEnabled(Handle handle, bool isEnabled)
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    if (component->IsAttached() && (!IsStopped())) {
        return OnSetComponentEnabled(handle, isEnabled); // notify to change 'live' parameter
    }
    else {
        component->m_isEnabled = isEnabled; // changing 'default' parameter
    }

    return true;
}

bool ScriptSystem::GetComponentEnabled(Handle handle, bool& isEnabled) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    if (component->IsAttached() && (!IsStopped())) {
        return OnGetComponentEnabled(handle, isEnabled); // return 'live' enabled flag
    }

    isEnabled = component->IsEnabled();
    return true;
}

bool ScriptSystem::SetComponentScript(Handle handle, MemoryManagement::SharedPointer<Script> script)
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    if (component->m_script != script) {
        component->m_script = script;
        static_cast<void>(RemoveScriptEventListener(handle));
        return OnSetComponentScript(handle);
    }

    return true;
}

bool ScriptSystem::AddComponentParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const
{
    if (AddComponentParameter(handle, parameterName, ScriptComponent::Number)) {
        return SetComponentDefaultParameterNumber(handle, parameterName, parameterValue);
    }

    return false;
}

bool ScriptSystem::AddComponentParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const
{
    if (AddComponentParameter(handle, parameterName, ScriptComponent::Integer)) {
        return SetComponentDefaultParameterInteger(handle, parameterName, parameterValue);
    }

    return false;
}

bool ScriptSystem::AddComponentParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const
{
    if (AddComponentParameter(handle, parameterName, ScriptComponent::Boolean)) {
        return SetComponentDefaultParameterBoolean(handle, parameterName, parameterValue);
    }

    return false;
}

bool ScriptSystem::AddComponentParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const
{
    if (AddComponentParameter(handle, parameterName, ScriptComponent::String)) {
        return SetComponentDefaultParameterString(handle, parameterName, parameterValue);
    }

    return false;
}

bool ScriptSystem::AddComponentParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const
{
    if (0 != parameterValue) {
        Internal::ObjectReferenceSystem::Handle objectReference = GetObjectReference(parameterValue);
        if (!objectReference.IsNullHandle()) {
            return AddComponentParameterObjectReference(handle, parameterName, objectReference, parameterValue->GetDynamicTypeId());
        }
    }

    return false;
}

bool ScriptSystem::AddComponentParameterObjectReference(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterValueTypeId) const
{
    if (AddComponentParameter(handle, parameterName, ScriptComponent::ObjectReference, parameterValueTypeId)) {
        return SetComponentDefaultParameterObjectReference(handle, parameterName, parameterValue, parameterValueTypeId);
    }

    return false;
}


// Parameter type: Double
bool ScriptSystem::GetComponentDefaultParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const
{
    DoubleGetter getter;
    return AccessComponentParameter<Double&, DoubleGetter>(handle, parameterName, parameterValue, ScriptComponent::Number, getter);
}

bool ScriptSystem::SetComponentDefaultParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const
{
    DoubleSetter setter;
    return AccessComponentParameter<Double, DoubleSetter>(handle, parameterName, parameterValue, ScriptComponent::Number, setter);
}

// Parameter type: Integer
bool ScriptSystem::GetComponentDefaultParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const
{
    IntegerGetter getter;
    return AccessComponentParameter<Int64&, IntegerGetter>(handle, parameterName, parameterValue, ScriptComponent::Integer, getter);
}

bool ScriptSystem::SetComponentDefaultParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const
{
    IntegerSetter setter;
    return AccessComponentParameter<Int64, IntegerSetter>(handle, parameterName, parameterValue, ScriptComponent::Integer, setter);
}

// Parameter type: Boolean
bool ScriptSystem::GetComponentDefaultParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const
{
    BooleanGetter getter;
    return AccessComponentParameter<bool&, BooleanGetter>(handle, parameterName, parameterValue, ScriptComponent::Boolean, getter);
}

bool ScriptSystem::SetComponentDefaultParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const
{
    BooleanSetter setter;
    return AccessComponentParameter<bool, BooleanSetter>(handle, parameterName, parameterValue, ScriptComponent::Boolean, setter);
}

// Parameter type: String
bool ScriptSystem::GetComponentDefaultParameterString(Handle handle, const Char* parameterName, const Char*& parameterValue) const
{
    StringGetter getter;
    return AccessComponentParameter<const Char*&, StringGetter>(handle, parameterName, parameterValue, ScriptComponent::String, getter);
}

bool ScriptSystem::SetComponentDefaultParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const
{
    Char* parameterValueCopy = CANDERA_NEW_ARRAY(Char, StringPlatform::Length(parameterValue)+1);
    if (0 == parameterValueCopy) {
        return false;
    }

    StringPlatform::Copy(parameterValueCopy, parameterValue);
    StringSetter setter;
    return AccessComponentParameter<const Char*, StringSetter>(handle, parameterName, parameterValueCopy, ScriptComponent::String, setter);
}

// Parameter type: ObjectReference
bool ScriptSystem::GetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName,
    CanderaObject*& parameterValue, TypeId& parameterTypeId) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
            ScriptComponent::ParameterInfo::Parameter& parameter = parameters[i].m_parameter;
            FEATSTD_COMPILETIME_ASSERT(sizeof(parameter.m_value.m_handle) == sizeof(Handle));
            parameterTypeId = parameter.m_typeId;
            Internal::ObjectReferenceSystem::Handle objectReference = m_objectReferenceSystem->CastToHandle(parameter.m_value.m_handle);
            Internal::ObjectReferenceComponent* objectReferenceComponent = m_objectReferenceSystem->GetPointer(objectReference);
            if (0 != objectReferenceComponent) {
                parameterValue = objectReferenceComponent->GetEntity();
            }
            else {
                parameterValue = 0;
            }

            return true;
        }
    }

    return false;
}

bool ScriptSystem::GetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId parameterValueTypeId) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        ScriptComponent::ParameterInfo::Parameter& parameter = parameters[i].m_parameter;
        if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
            Internal::ObjectReferenceSystem* objectReferenceSystem = m_objectReferenceSystem;
            Internal::ObjectReferenceSystem::Handle referenceHandle = objectReferenceSystem->CastToHandle(parameter.m_value.m_handle);
            if (parameterValueTypeId == parameter.m_typeId) {
                parameterValue = referenceHandle;
                return true;
            }

            Internal::ObjectReferenceComponent* objectReferenceComponent = objectReferenceSystem->GetPointer(referenceHandle);
            if (0 != objectReferenceComponent) {
                CanderaObject* object = objectReferenceComponent->GetEntity();
                if ((0 != object) && (object->IsTypeOf(parameterValueTypeId))) {
                    parameterValue = referenceHandle;
                    parameter.m_typeId = parameterValueTypeId;
                    return true;
                }
            }
        }
    }

    return false;
}

bool ScriptSystem::SetObjectReference(ScriptParameters& parameters, Internal::ObjectReferenceSystem::Handle objectReference,
    TypeId typeId, SizeType index) const
{
    if (parameters.Size() > index) {
        ScriptComponent::ParameterInfo::Parameter& parameter = parameters.m_parameters[index];
        if (ScriptComponent::ObjectReference == parameter.m_type) {
            parameter.m_value.m_handle = m_objectReferenceSystem->GetHandleRaw(objectReference);
            parameter.m_typeId = typeId;
            return true;
        }
    }

    return false;
}

bool ScriptSystem::GetObjectReference(const ScriptParameters& parameters, Internal::ObjectReferenceSystem::Handle& objectReference,
    SizeType index) const
{
    if (parameters.Size() > index) {
        const ScriptComponent::ParameterInfo::Parameter& parameter = parameters.m_parameters[index];
        if (ScriptComponent::ObjectReference == parameter.m_type) {
            objectReference = m_objectReferenceSystem->CastToHandle(parameter.m_value.m_handle);
            return true;
        }
    }

    return false;
}

Internal::ObjectReferenceSystem::Handle ScriptSystem::GetObjectReference(CanderaObject* object) const
{
    Internal::ObjectReferenceSystem::Handle objectReference;
    if (0 != object) {
        objectReference = EntitySystem::GetComponent<Internal::ObjectReferenceComponent>(object);
        if (objectReference.IsNullHandle()) {
            Internal::ObjectReferenceSystem* objectReferenceSystem = m_objectReferenceSystem;
            objectReference = objectReferenceSystem->CreateComponent();
            if (objectReference.IsNullHandle()) {
                return objectReference;
            }

            if (!objectReferenceSystem->AttachComponent(objectReference, object)) {
                static_cast<void>(objectReferenceSystem->DestroyComponent(objectReference)); // We already failed, no use for the return value.
                return Internal::ObjectReferenceSystem::Handle();
            }
        }
    }

    return objectReference;
}

CanderaObject* ScriptSystem::GetObjectPointer(Internal::ObjectReferenceSystem::Handle objectReference) const
{
    Internal::ObjectReferenceComponent* objectReferenceComponent = m_objectReferenceSystem->GetPointer(objectReference);
    CanderaObject* canderaObject = 0;
    if (0 != objectReferenceComponent) {
        canderaObject = objectReferenceComponent->GetEntity();
    }

    return canderaObject;
}

bool ScriptSystem::SetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const
{
    Internal::ObjectReferenceSystem::Handle objectReference;
    TypeId parameterValueTypeId = TypeId();
    if (0 != parameterValue) {
        objectReference = GetObjectReference(parameterValue);
        parameterValueTypeId = parameterValue->GetDynamicTypeId();
        if (objectReference.IsNullHandle()) {
            return false;
        }
    }

    return SetComponentDefaultParameterObjectReference(handle, parameterName, objectReference, parameterValueTypeId);
}

bool ScriptSystem::SetComponentDefaultParameterObjectReference(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterValueTypeId) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    CanderaObject* object = 0;
    Internal::ObjectReferenceComponent* objectReferenceComponent = m_objectReferenceSystem->GetPointer(parameterValue);
    if (0 != objectReferenceComponent) {
        object = objectReferenceComponent->GetEntity();
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        ScriptComponent::ParameterInfo::Parameter& parameter = parameters[i].m_parameter;
        if ((parameterValue.IsNullHandle() || (parameterValueTypeId == parameter.m_typeId)) ||
            ((0 != object) && (object->IsTypeOf(parameter.m_typeId)))) {
            if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
                parameter.m_value.m_handle = m_objectReferenceSystem->GetHandleRaw(parameterValue);
                return true;
            }
        }
    }

    return false;
}

bool ScriptSystem::GetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle& parameterValue, TypeId& parameterTypeId) const
{
    FEATSTD_UNUSED4(handle, parameterName, parameterValue, parameterTypeId);
    return false;
}

bool ScriptSystem::SetComponentLiveParameterObjectReferenceHandle(Handle handle, const Char* parameterName,
    Internal::ObjectReferenceSystem::Handle parameterValue, TypeId parameterTypeId) const
{
    FEATSTD_UNUSED4(handle, parameterName, parameterValue, parameterTypeId);
    return false;
}

ScriptComponent::ParameterType ScriptSystem::GetComponentLiveParameterType(Handle handle, const Char* parameterName) const
{
    FEATSTD_UNUSED2(handle, parameterName);
    return ScriptComponent::None;
}

bool ScriptSystem::GetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double& parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::GetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64& parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::GetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool& parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::GetComponentLiveParameterString(Handle handle, const Char* parameterName, Char*& parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::GetComponentLiveParameterObjectReference(Handle handle, const Char* parameterName,
    CanderaObject*& parameterValue, TypeId& parameterTypeId) const
{
    Internal::ObjectReferenceSystem::Handle objectReference;
    if (GetComponentLiveParameterObjectReferenceHandle(handle, parameterName, objectReference, parameterTypeId)) {
        Internal::ObjectReferenceComponent* objectReferenceComponent = m_objectReferenceSystem->GetPointer(objectReference);
        if (0 != objectReferenceComponent) {
            parameterValue = objectReferenceComponent->GetEntity();
        }
        else {
            parameterValue = 0;
        }

        return true;
    }

    return false;
}

bool ScriptSystem::SetComponentLiveParameterNumber(Handle handle, const Char* parameterName, Double parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::SetComponentLiveParameterInteger(Handle handle, const Char* parameterName, Int64 parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::SetComponentLiveParameterBoolean(Handle handle, const Char* parameterName, bool parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::SetComponentLiveParameterString(Handle handle, const Char* parameterName, const Char* parameterValue) const
{
    FEATSTD_UNUSED3(handle, parameterName, parameterValue);
    return false;
}

bool ScriptSystem::SetComponentLiveParameterObjectReference(Handle handle, const Char* parameterName, CanderaObject* parameterValue) const
{
    Internal::ObjectReferenceSystem::Handle objectReference;
    TypeId typeId = TypeId();
    if (0 != parameterValue) {
        objectReference = GetObjectReference(parameterValue);
        typeId = parameterValue->GetDynamicTypeId();
        if (objectReference.IsNullHandle()) {
            return false;
        }
    }

    return SetComponentLiveParameterObjectReferenceHandle(handle, parameterName, objectReference, typeId);
}

bool ScriptSystem::CallComponentMethod(Handle handle, const Char* methodName, const ScriptParameters& in, ScriptParameters& out) const
{
    FEATSTD_UNUSED4(handle, methodName, in, out);
    return false;
}

ScriptSystem::ComponentDefaultParameterIterator::ComponentDefaultParameterIterator(const ScriptSystem* scriptSystem, Handle handle)
    :
    m_component(0),
    m_index(0)
{
    if (0 != scriptSystem) {
        m_component = scriptSystem->GetPointer(handle);
    }
}

bool ScriptSystem::ComponentDefaultParameterIterator::Get(const Char*& parameterName, ScriptComponent::ParameterType& parameterType) const
{
    if ((0 == m_component) || (m_index >= static_cast<Int>(m_component->m_parameters.Size()))) {
        return false;
    }

    const ScriptComponent::Parameters& parameters = m_component->m_parameters;
    parameterName = parameters[m_index].m_name;
    parameterType = parameters[m_index].m_parameter.m_type;
    return true;
}

void ScriptSystem::UpdateWithSeconds(Double applicationTime, Double deltaTime)
{
    FEATSTD_UNUSED(applicationTime);
    FEATSTD_UNUSED(deltaTime);

    if (IsEnabled() && m_isFirstUpdate) {
        CallScriptInit();
    }
}

bool ScriptSystem::OnDetachComponent(Handle handle)
{
    static_cast<void>(RemoveScriptEventListener(handle));
    return true;
}

bool ScriptSystem::RemoveComponentParameter(Handle handle, const Char* parameterName) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
            FEATSTD_DELETE_ARRAY(parameters[i].m_name);
            ScriptComponent::ParameterInfo::Parameter& parameter = parameters[i].m_parameter;
            if (ScriptComponent::String == parameter.m_type) {
                FEATSTD_DELETE_ARRAY(parameter.m_value.m_string);
            }

            return parameters.Remove(i);
        }
    }

    return true;
}

bool ScriptSystem::AddComponentParameter(Handle handle, const Char* parameterName, const ScriptComponent::ParameterType parameterType,
    TypeId parameterTypeId) const
{
    ScriptComponent* component = GetPointer(handle);
    if (0 == component) {
        return false; // could not resolve handle
    }

    ScriptComponent::Parameters& parameters = component->m_parameters;
    for (SizeType i = 0; i < parameters.Size(); ++i) {
        if (0 == StringPlatform::CompareStrings(parameterName, parameters[i].m_name)) {
            return (parameterType == parameters[i].m_parameter.m_type);
        }
    }

    if (0 == parameters.GetCapacity()) {
        // Avoid initial capacity reservation of size Vector::c_MemoryInitialCapacity
        if (!parameters.Reserve(SizeType(1))) {
            return false;
        }
    }

    Char* parameterNameCopy = CANDERA_NEW_ARRAY(Char, StringPlatform::Length(parameterName)+1);
    if (0 == parameterNameCopy) {
        return false;
    }

    StringPlatform::Copy(parameterNameCopy, parameterName);
    ScriptComponent::ParameterInfo parameter(parameterNameCopy, parameterType, parameterTypeId);
    bool success = parameters.Add(parameter);
    if (!success) {
        FEATSTD_DELETE_ARRAY(parameterNameCopy);
    }

    return success;
}

bool ScriptSystem::AddScriptEventListener(FeatStd::EventListener* eventListener)
{
    return m_scriptEventSource.AddEventListener(eventListener);
}

bool ScriptSystem::RemoveScriptEventListener(FeatStd::EventListener* eventListener, bool waitForListenerRelease)
{
    return m_scriptEventSource.RemoveEventListener(eventListener, waitForListenerRelease);
}

bool ScriptSystem::DispatchScriptEvent(const ScriptEvent& scriptEvent)
{
    if (!IsStopped()) {
        if (scriptEvent.GetParameters().Size() > 0) {
            m_scriptEventSource.DispatchEvent(scriptEvent);
            return true;
        }
    }

    return false;
}

bool ScriptSystem::AddScriptEventListener(Handle handle)
{
    if (!handle.IsNullHandle()) {
        if (!m_scriptEventListeners.Contains(handle)) {
            return m_scriptEventListeners.Add(handle);
        }
    }

    return false;
}

bool ScriptSystem::RemoveScriptEventListener(Handle handle)
{
    if (!handle.IsNullHandle()) {
        for (SizeType i = 0; i < m_scriptEventListeners.Size(); ++i) {
            if (m_scriptEventListeners[i] == handle) {
                return m_scriptEventListeners.Remove(i);
            }
        }
    }

    return false;
}

} // namespace Scripting

} // namespace Candera
