//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_SCRIPTPARAMETERS_H
#define CANDERA_SCRIPTING_SCRIPTPARAMETERS_H

#include <Candera/System/Container/Vector.h>
#include <CanderaPlatform/OS/CanderaTypes.h>
#include <CanderaScripting/ScriptComponent.h>

namespace Candera {

namespace Scripting {

class ScriptSystem;

/** @addtogroup Scripting
 *  @{
 */

/**
 * @brief  ScriptParameters serves as a container for supported types of values that
 *         can be passed from native code to scripts, and vice versa.
 */

class ScriptParameters
{

public:
    /**
     *  Constructor
     */
    ScriptParameters() {}

    /**
     *  Destructor
     */
    ~ScriptParameters();

    /**
     *  Add a value of type Double to the ScriptParameters container.
     *  @param value  The value to add to the container.
     *  @return  True, if the value was added successfully. False, otherwise.
     */
    bool Add(Double value);

    /**
     *  Add a value of type Int to the ScriptParameters container.
     *  @param value  The value to add to the container.
     *  @return  True, if the value was added successfully. False, otherwise.
     */
    bool Add(Int value);

    /**
     *  Add a value of type bool to the ScriptParameters container.
     *  @param value  The value to add to the container.
     *  @return  True, if the value was added successfully. False, otherwise.
     */
    bool Add(bool value);

    /**
     *  Add a zero terminated string to the ScriptParameters container.
     *  NOTE: The given pointer will not be stored, but a copy of the given string is made.
     *  @param value  The string to add to the container.
     *  @return  True, if the value was added successfully. False, otherwise.
     */
    bool Add(const Char* value);

    /**
     *  Add a CanderaObject pointer to the ScriptParameters container.
     *  @param value  The CanderaObject pointer to add to the container.
     *  @return  True, if the value was added successfully. False, otherwise.
     */
    bool Add(CanderaObject* value);

    /**
     *  Get the value of type Double at the given index from the container.
     *  @param value  A reference to store the value in.
     *  @param index  The index of the value in the container.
     *  @return  True, if the value could successfully be retrieved.
     *           False, if the index exceeds the container, or if the value stored at index is not of type Double.
     */
    bool Get(Double& value, SizeType index = 0) const;

    /**
     *  Get the value of type Int at the given index from the container.
     *  @param value  A reference to store the value in.
     *  @param index  The index of the value in the container.
     *  @return  True, if the value could successfully be retrieved.
     *           False, if the index exceeds the container, or if the value stored at index is not of type Int.
     */
    bool Get(Int& value, SizeType index = 0) const;

    /**
     *  Get the value of type bool at the given index from the container.
     *  @param value  A reference to store the value in.
     *  @param index  The index of the value in the container.
     *  @return  True, if the value could successfully be retrieved.
     *           False, if the index exceeds the container, or if the value stored at index is not of type bool.
     */
    bool Get(bool& value, SizeType index = 0) const;

    /**
     *  Get the pointer to the string at the given index from the container.
     *  @param value  A reference to store the pointer in.
     *  @param index  The index of the string in the container.
     *  @return  True, if the string could successfully be retrieved.
     *           False, if the index exceeds the container, or if the value stored at index is not of type String.
     */
    bool Get(const Char*& value, SizeType index = 0) const;

    /**
     *  Get the CanderaObject pointer at the given index from the container.
     *  @param value  A reference to store the pointer in.
     *  @param index  The index of the CanderaObject pointer in the container.
     *  @return  True, if the CanderaObject pointer could successfully be retrieved.
     *           False, if the index exceeds the container, or if the value stored at index is not of type ObjectReference.
     */
    bool Get(CanderaObject*& value, SizeType index = 0) const;

    /**
     *  Set the value of type Double at the given index in the container.
     *  @param value  The value to set.
     *  @param index  The index in the container to set the value at.
     *  @return  True, if the value could successfully be set.
     *           False, if the index exceeds the container, or if the value at index is not of type Double.
     */
    bool Set(Double value, SizeType index = 0);

    /**
     *  Set the value of type Int at the given index in the container.
     *  @param value  The value to set.
     *  @param index  The index in the container to set the value at.
     *  @return  True, if the value could successfully be set.
     *           False, if the index exceeds the container, or if the value at index is not of type Int.
     */
    bool Set(Int value, SizeType index = 0);

    /**
     *  Set the value of type bool at the given index in the container.
     *  @param value  The value to set.
     *  @param index  The index in the container to set the value at.
     *  @return  True, if the value could successfully be set.
     *           False, if the index exceeds the container, or if the value at index is not of type bool.
     */
    bool Set(bool value, SizeType index = 0);

    /**
     *  Set the string at the given index in the container.
     *  Note: The given pointer will not be stored, but a copy of the given string is made.
     *  @param value  The string to set.
     *  @param index  The index in the container to set the string at.
     *  @return  True, if the string could successfully be set.
     *           False, if the index exceeds the container, or if the value at index is not of type String.
     */
    bool Set(const Char* value, SizeType index = 0);

    /**
     *  Set the CanderaObject pointer at the given index in the container.
     *  @param value  The CanderaObject pointer to set.
     *  @param index  The index in the container to set the string at.
     *  @return  True, if the string could successfully be set.
     *           False, if the index exceeds the container, or if the value at index is not of type ObjectReference.
     */
    bool Set(CanderaObject* value, SizeType index = 0);

    /**
     *  Get the size (i.e. number of values) of the container.
     *  @return  The size of the container.
     */
    SizeType Size() const { return m_parameters.Size(); }

    /**
     *  Clear the container.
     */
    void Clear() { m_parameters.Clear(); }

    /**
     *  Get the type of the value at the given index in the container.
     *  @param index  The index of the value in the container to get the type for.
     *  @return  The type of the value at the given index in the container.
     */
    ScriptComponent::ParameterType GetType(SizeType index = 0) const;

private:
    typedef ScriptComponent::ParameterInfo::Parameter Parameter;
    typedef Candera::Internal::Vector<Parameter, FeatStd::Internal::LinearIncreasePolicy<1> > Parameters;
    Parameters m_parameters;

    bool CreateObjectReferenceParameter(CanderaObject* value, /* out */ Parameter& parameter) const;

    friend class ScriptSystem;
};

/** @} */ // end of Scripting

} // namespace Scripting

} // namespace Candera

#endif
