//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ScriptParameters.h"

#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaScripting/ObjectReferenceSystem.h>

namespace Candera
{

namespace Scripting
{

ScriptParameters::~ScriptParameters()
{
    for (SizeType i = 0; i < m_parameters.Size(); ++i) {
        if (ScriptComponent::String == m_parameters[i].m_type) {
            CANDERA_DELETE_ARRAY(m_parameters[i].m_value.m_string);
            m_parameters[i].m_value.m_string = 0;
        }
    }
}

bool ScriptParameters::Add(Double value)
{
    Parameter parameter(ScriptComponent::Number);
    parameter.m_value.m_double = value;
    return m_parameters.Add(parameter);
}

bool ScriptParameters::Add(Int value)
{
    Parameter parameter(ScriptComponent::Integer);
    parameter.m_value.m_integer = value;
    return m_parameters.Add(parameter);
}

bool ScriptParameters::Add(bool value)
{
    Parameter parameter(ScriptComponent::Boolean);
    parameter.m_value.m_boolean = value;
    return m_parameters.Add(parameter);
}

bool ScriptParameters::Add(const Char* value)
{
    Parameter parameter(ScriptComponent::String);
    parameter.m_value.m_string = 0;

    if (0 != value) {
        Char* string = CANDERA_NEW_ARRAY(Char, StringPlatform::Length(value) + 1);
        if (0 != string) {
            StringPlatform::Copy(string, value);
            parameter.m_value.m_string = string;
        }
        else {
            return false;
        }
    }

    return m_parameters.Add(parameter);
}

bool ScriptParameters::Add(CanderaObject* value)
{
    Parameter parameter(ScriptComponent::ObjectReference);
    if (CreateObjectReferenceParameter(value, parameter)) {
        return m_parameters.Add(parameter);
    }

    return false;
}


bool ScriptParameters::Get(Double& value, SizeType index) const
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Number == m_parameters[index].m_type) {
            value = m_parameters[index].m_value.m_double;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Get(Int& value, SizeType index) const
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Integer == m_parameters[index].m_type) {
            value = static_cast<Int>(m_parameters[index].m_value.m_integer);
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Get(bool& value, SizeType index) const
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Boolean == m_parameters[index].m_type) {
            value = m_parameters[index].m_value.m_boolean;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Get(const Char*& value, SizeType index) const
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::String == m_parameters[index].m_type) {
            value = m_parameters[index].m_value.m_string;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Get(CanderaObject*& value, SizeType index) const
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::ObjectReference == m_parameters[index].m_type) {
            using namespace EntityComponentSystem;
            Internal::ObjectReferenceSystem* objectReferenceSystem = EntitySystem::Get<Internal::ObjectReferenceSystem>();
            if (0 != objectReferenceSystem) {
                Internal::ObjectReferenceSystem::Handle objectReference = objectReferenceSystem->CastToHandle(m_parameters[index].m_value.m_handle);
                Internal::ObjectReferenceComponent* objectReferenceComponent = objectReferenceSystem->GetPointer(objectReference);
                if (0 != objectReferenceComponent) {
                    value = objectReferenceComponent->GetEntity();
                }
                else {
                    value = 0;
                }

                return true;
            }
        }
    }

    return false;
}

bool ScriptParameters::Set(Double value, SizeType index)
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Number == m_parameters[index].m_type) {
            m_parameters[index].m_value.m_double = value;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Set(Int value, SizeType index)
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Integer == m_parameters[index].m_type) {
            m_parameters[index].m_value.m_integer = value;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Set(bool value, SizeType index)
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::Boolean == m_parameters[index].m_type) {
            m_parameters[index].m_value.m_boolean = value;
            return true;
        }
    }

    return false;
}

bool ScriptParameters::Set(const Char* value, SizeType index)
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::String == m_parameters[index].m_type) {
            if (0 != m_parameters[index].m_value.m_string) {
                CANDERA_DELETE_ARRAY(m_parameters[index].m_value.m_string);
                m_parameters[index].m_value.m_string = 0;
            }

            if (0 != value) {
                Char* string = CANDERA_NEW_ARRAY(Char, StringPlatform::Length(value) + 1);
                if (0 != string) {
                    StringPlatform::Copy(string, value);
                    m_parameters[index].m_value.m_string = string;
                }
            }

            return true;
        }
    }

    return false;
}

bool ScriptParameters::Set(CanderaObject* value, SizeType index)
{
    if (m_parameters.Size() > index) {
        if (ScriptComponent::ObjectReference == m_parameters[index].m_type) {
            Parameter parameter(ScriptComponent::ObjectReference);
            if (CreateObjectReferenceParameter(value, parameter)) {
                m_parameters[index] = parameter;
                return true;
            }
        }
    }

    return false;
}


ScriptComponent::ParameterType ScriptParameters::GetType(SizeType index) const
{
    if (m_parameters.Size() > index) {
        return m_parameters[index].m_type;
    }

    FEATSTD_DEBUG_FAIL();
    return ScriptComponent::None;
}

bool ScriptParameters::CreateObjectReferenceParameter(CanderaObject* value, /* out */ Parameter& parameter) const
{
    using namespace EntityComponentSystem;
    Internal::ObjectReferenceSystem* objectReferenceSystem = EntitySystem::Get<Internal::ObjectReferenceSystem>();
    if (0 != objectReferenceSystem) {
        Internal::ObjectReferenceSystem::Handle objectReference;
        TypeId typeId = TypeId();
        if (0 != value) {
            objectReference = EntitySystem::GetComponent<Internal::ObjectReferenceComponent>(value);
            if (objectReference.IsNullHandle()) {
                objectReference = objectReferenceSystem->CreateComponent();
                if (objectReference.IsNullHandle()) {
                    return false;
                }

                if (!objectReferenceSystem->AttachComponent(objectReference, value)) {
                    static_cast<void>(objectReferenceSystem->DestroyComponent(objectReference)); // We already failed, no use for the return value.
                    return false;
                }
            }

            typeId = value->GetDynamicTypeId();
        }

        FEATSTD_DEBUG_ASSERT(ScriptComponent::ObjectReference == parameter.m_type);
        parameter.m_typeId = typeId;
        parameter.m_value.m_handle = objectReferenceSystem->GetHandleRaw(objectReference);
        return true;
    }

    return false;
}

} // namespace Scripting

} // namespace Candera
