//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_SCRIPT_H
#define CANDERA_SCRIPTING_SCRIPT_H

#include <Candera/EngineBase/Common/InstanceId.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

namespace Scripting {

/** @addtogroup Scripting
 *  @{
 */


/**
 * @brief  Script is the container of a string of script code.
 */

class Script
{

public:
    FEATSTD_TYPEDEF_SHARED_POINTER(Script);

    /**
     *  Creates an instance of this class.
     *  @param script   The string of script code, or 0.
     *  @param isOwner  If true, this objects takes ownership over the script string.
     */
    static MemoryManagement::SharedPointer<Script> Create(const Char* script, bool isOwner = true);

    /**
     *  Disposes this instance and deletes the script code string if this instance is the owner.
     */
    void DisposeSelf();

    /**
     *  Set a string of script code, and take ownership of it. The previous string is deleted.
     *  The script system is notified of the change, so it can adapt its running VM.
     *  @param script   The string of script code, or 0.
     *  @param isOwner  If true, this objects takes ownership over the script string.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetScript(const Char* script, bool isOwner = true);

    /**
     *  Get the string of script code.
     *  @return  The string of script code, or 0.
     */
    const Char* GetScript() const { return m_script; }

    /**
     *   Returns the Id of this script instance.
     *   @return  The Id of this script instance.
     */
    SizeType GetInstanceId() const { return m_instanceId; }

    /**
     *  Public destructor for SharedPointer
     */
    virtual ~Script();

private:
    Script(const Char* script, bool isOwner);

    Candera::Internal::InstanceId<Script> m_instanceId;
    const Char* m_script;

    // Make this class manageable by MemoryManagement::SharedPointer
    CANDERA_SHARED_POINTER_DECLARATION();

    bool m_isOwner;
};

/** @} */ // end of Scripting

} // namespace Scripting

} // namespace Candera

#endif
