//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LuaBinding3D.h"

#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/BitmapTextureImage.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaScripting);

namespace Scripting {

namespace Internal {

const luaL_Reg LuaBinding3D::s_additionalFunctions[] = {
    // 3D only functions

    // Appearance
    { "HasAppearance", HasAppearance },
    { "SetAppearanceFrom", SetAppearanceFrom },
    { "HasMaterial", HasMaterial },
    { "HasTexture", HasTexture },
    { "HasRenderMode", HasRenderMode },
    { "HasShader", HasShader },
    { "HasShaderParamSetter", HasShaderParamSetter },
    { "SetMaterialFrom", SetMaterialFrom },
    { "SetTextureFrom", SetTextureFrom },
    { "SetRenderModeFrom", SetRenderModeFrom },
    { "SetShaderFrom", SetShaderFrom },
    { "SetShaderParamSetterFrom", SetShaderParamSetterFrom },

    // Material
    { "SetMaterialDiffuse", SetMaterialDiffuse },
    { "GetMaterialDiffuse", GetMaterialDiffuse },
    { "SetMaterialAmbient", SetMaterialAmbient },
    { "GetMaterialAmbient", GetMaterialAmbient },
    { "SetMaterialEmissive", SetMaterialEmissive },
    { "GetMaterialEmissive", GetMaterialEmissive },
    { "SetMaterialSpecular", SetMaterialSpecular },
    { "GetMaterialSpecular", GetMaterialSpecular },
    { "SetMaterialSpecularPower", SetMaterialSpecularPower },
    { "GetMaterialSpecularPower", GetMaterialSpecularPower },
    { "SetMaterialAlphaValue", SetMaterialAlphaValue },
    { "GetMaterialAlphaValue", GetMaterialAlphaValue },

    // RenderMode
    { "SetRenderModeColorWriteEnabled", SetRenderModeColorWriteEnabled },
    { "SetRenderModeColorWriteRedEnabled", SetRenderModeColorWriteRedEnabled },
    { "IsRenderModeColorWriteRedEnabled", IsRenderModeColorWriteRedEnabled },
    { "SetRenderModeColorWriteGreenEnabled", SetRenderModeColorWriteGreenEnabled },
    { "IsRenderModeColorWriteGreenEnabled", IsRenderModeColorWriteGreenEnabled },
    { "SetRenderModeColorWriteBlueEnabled", SetRenderModeColorWriteBlueEnabled },
    { "IsRenderModeColorWriteBlueEnabled", IsRenderModeColorWriteBlueEnabled },
    { "SetRenderModeColorWriteAlphaEnabled", SetRenderModeColorWriteAlphaEnabled },
    { "IsRenderModeColorWriteAlphaEnabled", IsRenderModeColorWriteAlphaEnabled },
    { "SetRenderModeDepthWriteEnabled", SetRenderModeDepthWriteEnabled },
    { "IsRenderModeDepthWriteEnabled", IsRenderModeDepthWriteEnabled },
    { "SetRenderModeDepthTestEnabled", SetRenderModeDepthTestEnabled },
    { "IsRenderModeDepthTestEnabled", IsRenderModeDepthTestEnabled },
    { "SetRenderModeBlendingEnabled", SetRenderModeBlendingEnabled },
    { "IsRenderModeBlendingEnabled", IsRenderModeBlendingEnabled },
    { "SetRenderModeStencilTestEnabled", SetRenderModeStencilTestEnabled },
    { "IsRenderModeStencilTestEnabled", IsRenderModeStencilTestEnabled },

    // Light
    { "IsLight", IsLight },
    { "SetLightAmbient", SetLightAmbient },
    { "GetLightAmbient", GetLightAmbient },
    { "SetLightDiffuse", SetLightDiffuse },
    { "GetLightDiffuse", GetLightDiffuse },
    { "SetLightSpecular", SetLightSpecular },
    { "GetLightSpecular", GetLightSpecular },
    { "SetLightDirection", SetLightDirection },
    { "GetLightDirection", GetLightDirection },
    { "GetLightWorldDirection", GetLightWorldDirection },
    { "SetLightRange", SetLightRange },
    { "GetLightRange", GetLightRange },
    { "SetLightSpotAngle", SetLightSpotAngle },
    { "GetLightSpotAngle", GetLightSpotAngle },
    { "SetLightSpotExponent", SetLightSpotExponent },
    { "GetLightSpotExponent", GetLightSpotExponent },
    { "SetLightAttenuation", SetLightAttenuation },
    { "GetLightAttenuation", GetLightAttenuation },
    { "SetLightAttenuationEnabled", SetLightAttenuationEnabled },
    { "IsLightAttenuationEnabled", IsLightAttenuationEnabled },

    // Camera
    { "SetCameraLookAtVector", SetCameraLookAtVector },
    { "GetCameraLookAtVector", GetCameraLookAtVector },
    { "GetCameraWorldLookAtVector", GetCameraWorldLookAtVector },
    { "SetCameraUpVector", SetCameraUpVector },
    { "GetCameraUpVector", GetCameraUpVector },
    { "GetCameraWorldUpVector", GetCameraWorldUpVector },
    { "GetCameraRightVector", GetCameraRightVector },
    { "GetCameraWorldRightVector", GetCameraWorldRightVector },
    { "CameraLookAtWorldPoint", CameraLookAtWorldPoint },
    { "CameraRotateAroundWorldAxis", CameraRotateAroundWorldAxis },
    { "SetCameraViewingFrustumCullingEnabled", SetCameraViewingFrustumCullingEnabled },
    { "IsCameraViewingFrustumCullingEnabled", IsCameraViewingFrustumCullingEnabled },

    // SetParamSetter
    { "SetShaderUniform", SetShaderUniform },
    { "GetShaderUniformType", GetShaderUniformType },
    { "GetShaderUniform", GetShaderUniform },

    { NULL, NULL } // sentinel
};

void LuaBinding3D::AddFunctions(lua_State* luaState)
{
    // expects the table to be on top of the stack to add the functions to
    luaL_setfuncs(luaState, s_additionalFunctions, 0);
}

const LuaBinding::CanderaReferenceType LuaBinding3D::s_canderaObjectReferenceTypes3D[] =
{
    { "Appearance", Appearance::GetTypeId() },
    { "BitmapImage3D", BitmapTextureImage::GetTypeId() },
    { "Camera", Camera::GetTypeId() },
    { "Light", Light::GetTypeId() },
    { "Material", Material::GetTypeId() },
    { "Node", Node::GetTypeId() },
    { "RenderMode", RenderMode::GetTypeId() },
    { "Shader", Shader::GetTypeId() },
    { "Texture", Texture::GetTypeId() }
};

struct CanderaShaderUniformType
{
    const Char* m_name;
    Shader::UniformType m_type;
};

static const CanderaShaderUniformType s_canderaShaderUniformTypes[] =
{
    { "Float", Shader::Float },
    { "FloatVec2", Shader::FloatVec2 },
    { "FloatVec3", Shader::FloatVec3 },
    { "FloatVec4", Shader::FloatVec4 },
    { "Integer", Shader::Integer },
    { "IntegerVec2", Shader::IntegerVec2 },
    { "IntegerVec3", Shader::IntegerVec3 },
    { "IntegerVec4", Shader::IntegerVec4 },
    { "Bool", Shader::Bool },
    { "BoolVec2", Shader::BoolVec2 },
    { "BoolVec3", Shader::BoolVec3 },
    { "BoolVec4", Shader::BoolVec4 },
    { "Sampler2D", Shader::Sampler2D },
    { "SamplerCube", Shader::SamplerCube }
};

void LuaBinding3D::Init3D(lua_State* luaState)
{
    // Set relevant ObjectReference types in Candera.ReferenceType table.
    const SizeType canderaObjectReferenceTypesSize = sizeof(s_canderaObjectReferenceTypes3D) / sizeof(CanderaReferenceType);
    AddCanderaReferenceType(luaState, &(s_canderaObjectReferenceTypes3D[0]), canderaObjectReferenceTypesSize);

    // Set relevant Shader::UniformType enums in Candera.Shader table.
    Int luaType = lua_getglobal(luaState, LuaScriptSystem::GetLuaCanderaName()); // Candera table
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    luaType = lua_getfield(luaState, -1, "Shader"); // Candera.Shader table
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    const SizeType canderaShaderUniformTypesSize = sizeof(s_canderaShaderUniformTypes) / sizeof(CanderaShaderUniformType);
    for (SizeType i = 0; i < canderaShaderUniformTypesSize; ++i) {
        lua_pushinteger(luaState, static_cast<lua_Integer>(s_canderaShaderUniformTypes[i].m_type));
        lua_setfield(luaState, -2, s_canderaShaderUniformTypes[i].m_name);
    }

    lua_pop(luaState, 2);
}


// ----------------------------------------------------------------------------
// Appearance

template<typename F>
Int LuaBinding3D::CopyAppearanceProperty(lua_State* luaState, F f)
{
    ScriptEntity* dstEntity = GetEntity(luaState);
    if (0 == dstEntity) {
        return 0;
    }

    const ScriptEntity* srcEntity = GetEntity(luaState, 2);

    if (dstEntity->IsTypeOf(Node::GetTypeId())) {
        Node* dstNode = static_cast<Node*>(dstEntity);
        Appearance::SharedPointer dstAppearance = dstNode->GetAppearance();

        if (dstAppearance != 0) {
            const Appearance* srcAppearance = 0;
            if ((0 != srcEntity) && (srcEntity->IsTypeOf(Node::GetTypeId()))) {
                const Node* srcNode = static_cast<const Node*>(srcEntity);
                srcAppearance = srcNode->GetAppearance().GetPointerToSharedInstance();
            }

            f(dstAppearance.GetPointerToSharedInstance(), srcAppearance);
        }
    }

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_COPY_APPEARANCE(Property)                                                                                                 \
struct FEATSTD_CONCAT3(CopyAppearance, Property, F) {                                                                                                     \
    void operator() (Appearance* dstAppearance, const Appearance* srcAppearance) const {                                                                  \
        if (srcAppearance != 0) {                                                                                                                         \
            FEATSTD_CONCAT2(dstAppearance->Set, Property)(FEATSTD_CONCAT2(srcAppearance->Get, Property)());                                               \
        }                                                                                                                                                 \
        else {                                                                                                                                            \
            FEATSTD_CONCAT2(dstAppearance->Set, Property)(MemoryManagement::SharedPointer<Property>(0));                                                  \
        }                                                                                                                                                 \
    }                                                                                                                                                     \
};                                                                                                                                                        \
                                                                                                                                                          \
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, FEATSTD_CONCAT3(Set, Property, From), CopyAppearanceProperty, FEATSTD_CONCAT3(CopyAppearance, Property, F))

struct HasAppearanceF {
    bool operator() (const ScriptEntity* entity) const
    {
        return (entity->IsTypeOf(Node::GetTypeId()) &&
            (!(static_cast<const Node*>(entity)->GetAppearance().PointsToNull())));
    }
};

CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, HasAppearance, IsF, HasAppearanceF)

Int LuaBinding3D::SetAppearanceFrom(lua_State* luaState)
{
    ScriptEntity* dstEntity = GetEntity(luaState);
    if (0 == dstEntity) {
        return 0;
    }

    const ScriptEntity* srcEntity = GetEntity(luaState, 2);

    if (dstEntity->IsTypeOf(Node::GetTypeId())) {
        Node* dstNode = static_cast<Node*>(dstEntity);

        if ((0 != srcEntity) && (srcEntity->IsTypeOf(Node::GetTypeId()))) {
            const Node* srcNode = static_cast<const Node*>(srcEntity);
            dstNode->SetAppearance(srcNode->GetAppearance());
        }
        else {
            dstNode->SetAppearance(Appearance::SharedPointer(0));
        }

    }

    return 0;
}

#define CANDERA_LUASCRIPTSYSTEM_APPEARANCE_HAS_F(Property)                                                      \
struct FEATSTD_CONCAT3(Has, Property, F) {                                                                      \
    bool operator() (const ScriptEntity* entity) const                                                          \
    {                                                                                                           \
        if (entity->IsTypeOf(Node::GetTypeId())) {                                                              \
            const Node* node = static_cast<const Node*>(entity);                                                \
            const Appearance::SharedPointer& appearance = node->GetAppearance();                                \
            return !(appearance.PointsToNull() || FEATSTD_CONCAT2(appearance->Get, Property)().PointsToNull()); \
        }                                                                                                       \
                                                                                                                \
        return false;                                                                                           \
    }                                                                                                           \
};

// hasRenderMode = Candera.HasRenderMode(id)
CANDERA_LUASCRIPTSYSTEM_APPEARANCE_HAS_F(RenderMode)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, HasRenderMode, IsF, HasRenderModeF)

// hasMaterial = Candera.HasMaterial(id)
CANDERA_LUASCRIPTSYSTEM_APPEARANCE_HAS_F(Material)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, HasMaterial, IsF, HasMaterialF)

// hasShader = Candera.HasShader(id)
CANDERA_LUASCRIPTSYSTEM_APPEARANCE_HAS_F(Shader)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, HasShader, IsF, HasShaderF)

// hasShaderParamSetter = Candera.HasShaderParamSetter(id)
CANDERA_LUASCRIPTSYSTEM_APPEARANCE_HAS_F(ShaderParamSetter)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, HasShaderParamSetter, IsF, HasShaderParamSetterF)

// hasTexture = Candera.HasTexture(id)
Int LuaBinding3D::HasTexture(lua_State* luaState)
{
    const ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    UInt unit = static_cast<UInt>(luaL_checkinteger(luaState, 2));

    if (entity->IsTypeOf(Node::GetTypeId())) {
        const Node* node = static_cast<const Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            Texture::SharedPointer texture = appearance->GetTexture(unit);
            if (texture != 0) {
                lua_pushboolean(luaState, 1);
                return 1;
            }
        }
    }

    return 0;
}

// Candera.SetMaterialFrom(id, sourceId)
CANDERA_LUASCRIPTSYSTEM_COPY_APPEARANCE(Material)

// Candera.SetRenderModeFrom(id, sourceId)
CANDERA_LUASCRIPTSYSTEM_COPY_APPEARANCE(RenderMode)

// Candera.SetShaderFrom(id, sourceId)
CANDERA_LUASCRIPTSYSTEM_COPY_APPEARANCE(Shader)

// Candera.SetShaderParamSetterFrom(id, sourceId)
struct CopyAppearanceShaderParamSetterF {
    void operator() (Appearance* dstAppearance, const Appearance* srcAppearance) const {
        if (srcAppearance != 0) {
            dstAppearance->SetShaderParamSetter(srcAppearance->GetShaderParamSetter());
        }
        else {
            dstAppearance->SetShaderParamSetter(MemoryManagement::SharedPointer<AbstractShaderParamSetter>(0));
        }
    }
};

CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, SetShaderParamSetterFrom, CopyAppearanceProperty, CopyAppearanceShaderParamSetterF)

// Candera.SetTextureFrom(id, sourceId)
Int LuaBinding3D::SetTextureFrom(lua_State* luaState)
{
    ScriptEntity* dstEntity = GetEntity(luaState);
    if (0 == dstEntity) {
        return 0;
    }

    UInt dstUnit = static_cast<UInt>(luaL_checkinteger(luaState, 2));
    const ScriptEntity* srcEntity = GetEntity(luaState, 3);
    UInt srcUnit = static_cast<UInt>(luaL_checkinteger(luaState, 4));

    if (dstEntity->IsTypeOf(Node::GetTypeId())) {
        Node* dstNode = static_cast<Node*>(dstEntity);
        Appearance::SharedPointer dstAppearance = dstNode->GetAppearance();
        if (dstAppearance != 0) {
            Texture::SharedPointer texture;

            if ((0 != srcEntity) && (srcEntity->IsTypeOf(Node::GetTypeId()))) {
                const Node* srcNode = static_cast<const Node*>(srcEntity);
                Appearance::SharedPointer srcAppearance = srcNode->GetAppearance();
                if (srcAppearance != 0) {
                    texture = srcAppearance->GetTexture(srcUnit);
                }
            }

            static_cast<void>(dstAppearance->SetTexture(texture, dstUnit));
        }
    }

    return 0;
}


// ----------------------------------------------------------------------------
// Material

template<typename T, typename F>
Int LuaBinding3D::SetMaterialColor(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float r = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float g = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float b = static_cast<Float>(luaL_checknumber(luaState, 4));
    Float a = static_cast<Float>(luaL_checknumber(luaState, 5));

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            Material::SharedPointer material = appearance->GetMaterial();
            if (material != 0) {
                Color color(r, g, b, a);
                f(material.GetPointerToSharedInstance(), color);
            }
        }
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding3D::GetMaterialColor(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            Material::SharedPointer material = appearance->GetMaterial();
            if (material != 0) {
                Color color = f(material.GetPointerToSharedInstance());
                lua_pushnumber(luaState, color.GetRed());
                lua_pushnumber(luaState, color.GetGreen());
                lua_pushnumber(luaState, color.GetBlue());
                lua_pushnumber(luaState, color.GetAlpha());
                return 4;
            }
        }
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding3D::SetMaterialFloat(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float value = static_cast<Float>(luaL_checknumber(luaState, 2));

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            Material::SharedPointer material = appearance->GetMaterial();
            if (material != 0) {
                f(material.GetPointerToSharedInstance(), value);
            }
        }
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding3D::GetMaterialFloat(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            Material::SharedPointer material = appearance->GetMaterial();
            if (material != 0) {
                Float value = f(material.GetPointerToSharedInstance());
                lua_pushnumber(luaState, value);
                return 1;
            }
        }
    }

    return 0;
}

// Candera.SetMaterialDiffuse(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Color, SetMaterialColor, SetMaterialDiffuse, SetDiffuse)

// r, g, b, a = Candera.GetMaterialDiffuse(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Color, GetMaterialColor, GetMaterialDiffuse, GetDiffuse)

// Candera.SetMaterialAmbient(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Color, SetMaterialColor, SetMaterialAmbient, SetAmbient)

// r, g, b, a = Candera.GetMaterialAmbient(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Color, GetMaterialColor, GetMaterialAmbient, GetAmbient)

// Candera.SetMaterialEmissive(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Color, SetMaterialColor, SetMaterialEmissive, SetEmissive)

// r, g, b, a = Candera.GetMaterialEmissive(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Color, GetMaterialColor, GetMaterialEmissive, GetEmissive)

// Candera.SetMaterialSpecular(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Color, SetMaterialColor, SetMaterialSpecular, SetSpecular)

// r, g, b, a = Candera.GetMaterialSpecular(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Color, GetMaterialColor, GetMaterialSpecular, GetSpecular)

// Candera.SetMaterialSpecularPower(id, specularPower)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Float, SetMaterialFloat, SetMaterialSpecularPower, SetSpecularPower)

// specularPower = Candera.GetMaterialSpecularPower(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Float, GetMaterialFloat, GetMaterialSpecularPower, GetSpecularPower)

// Candera.SetMaterialAlphaValue(id, alphaValue)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Material, Float, SetMaterialFloat, SetMaterialAlphaValue, SetAlphaValue)

// alphaValue = Candera.GetMaterialAlphaValue(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Material, Float, GetMaterialFloat, GetMaterialAlphaValue, GetAlphaValue)


// ----------------------------------------------------------------------------
// RenderMode

template<typename T, typename F>
Int LuaBinding3D::SetRenderModeBool(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            RenderMode::SharedPointer renderMode = appearance->GetRenderMode();
            if (renderMode != 0) {
                bool value = (lua_toboolean(luaState, 2) != 0);
                f(renderMode.GetPointerToSharedInstance(), value);
            }
        }
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding3D::GetRenderModeBool(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            RenderMode::SharedPointer renderMode = appearance->GetRenderMode();
            if (renderMode != 0) {
                bool value = f(renderMode.GetPointerToSharedInstance());
                lua_pushboolean(luaState, (value ? 1 : 0));
                return 1;
            }
        }
    }

    return 0;
}

// Candera.SetRenderModeColorWriteEnabled(id, enableRed, enableGreen, enableBlue, enableAlpha)
Int LuaBinding3D::SetRenderModeColorWriteEnabled(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            RenderMode::SharedPointer renderMode = appearance->GetRenderMode();
            if (renderMode != 0) {
                bool enableRed = (lua_toboolean(luaState, 2) != 0);
                bool enableGreen = (lua_toboolean(luaState, 3) != 0);
                bool enableBlue = (lua_toboolean(luaState, 4) != 0);
                bool enableAlpha = (lua_toboolean(luaState, 5) != 0);
                renderMode->SetColorWriteEnabled(enableRed, enableGreen, enableBlue, enableAlpha);
            }
        }
    }

    return 0;
}

// Candera.SetRenderModeColorWriteRedEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeColorWriteRedEnabled, SetColorWriteRedEnabled)

// isEnabled = Candera.IsRenderModeColorWriteRedEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeColorWriteRedEnabled, IsColorWriteRedEnabled)

// Candera.SetRenderModeColorWriteGreenEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeColorWriteGreenEnabled, SetColorWriteGreenEnabled)

// isEnabled = Candera.IsRenderModeColorWriteGreenEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeColorWriteGreenEnabled, IsColorWriteGreenEnabled)

// Candera.SetRenderModeColorWriteBlueEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeColorWriteBlueEnabled, SetColorWriteBlueEnabled)

// isEnabled = Candera.IsRenderModeColorWriteBlueEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeColorWriteBlueEnabled, IsColorWriteBlueEnabled)

// Candera.SetRenderModeColorWriteAlphaEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeColorWriteAlphaEnabled, SetColorWriteAlphaEnabled)

// isEnabled = Candera.IsRenderModeColorWriteAlphaEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeColorWriteAlphaEnabled, IsColorWriteAlphaEnabled)

// Candera.SetRenderModeDepthWriteEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeDepthWriteEnabled, SetDepthWriteEnabled)

// isEnabled = Candera.IsRenderModeDepthWriteEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeDepthWriteEnabled, IsDepthWriteEnabled)

// Candera.SetRenderModeDepthTestEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeDepthTestEnabled, SetDepthTestEnabled)

// isEnabled = Candera.IsRenderModeDepthTestEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeDepthTestEnabled, IsDepthTestEnabled)

// Candera.SetRenderModeBlendingEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeBlendingEnabled, SetBlendingEnabled)

// isEnabled = Candera.IsRenderModeBlendingEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeBlendingEnabled, IsBlendingEnabled)

// Candera.SetRenderModeStencilTestEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, RenderMode, bool, SetRenderModeBool, SetRenderModeStencilTestEnabled, SetStencilTestEnabled)

// isEnabled = Candera.IsRenderModeStencilTestEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, RenderMode, bool, GetRenderModeBool, IsRenderModeStencilTestEnabled, IsStencilTestEnabled)


// ----------------------------------------------------------------------------
// Light

struct IsLightF {
    bool operator() (const ScriptEntity* entity) const { return entity->IsTypeOf(Light::GetTypeId()); }
};

// isLight = Candera.IsLight(id)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding3D, IsLight, IsF, IsLightF)

// Candera.SetLightAmbient(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Color, SetColor, SetLightAmbient, SetAmbient)

// r, g, b, a = Candera.GetLightAmbient()
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Color, GetColor, GetLightAmbient, GetAmbient)

// Candera.SetLightDiffuse(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Color, SetColor, SetLightDiffuse, SetDiffuse)

// r, g, b, a = Candera.GetLightDiffuse()
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Color, GetColor, GetLightDiffuse, GetDiffuse)

// Candera.SetLightSpecular(id, r, g, b, a)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Color, SetColor, SetLightSpecular, SetSpecular)

// r, g, b, a = Candera.GetLightSpecular(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Color, GetColor, GetLightSpecular, GetSpecular)

// Candera.SetLightDirection(id, x, y, z)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Vector3, SetVector3, SetLightDirection, SetDirection)

// x, y, z = Candera.GetLightDirection(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Vector3, GetVector3, GetLightDirection, GetDirection)

// x, y, z = Candera.GetLightWorldDirection(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Vector3, GetVector3, GetLightWorldDirection, GetWorldDirection)

// Candera.SetLightRange(id, range)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Float, SetFloat, SetLightRange, SetRange)

// range = Candera.GetLightRange(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Float, GetFloat, GetLightRange, GetRange)

// Candera.SetLightSpotAngle(id, angle)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Float, SetFloat, SetLightSpotAngle, SetSpotAngle)

// angle = Candera.GetLightSpotAngle(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Float, GetFloat, GetLightSpotAngle, GetSpotAngle)

// Candera.SetLightSpotExponent(id, exponent)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, Float, SetFloat, SetLightSpotExponent, SetSpotExponent)

// exponent = Candera.GetLightSpotExponent(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, Float, GetFloat, GetLightSpotExponent, GetSpotExponent)

// Candera.SetLightAttenuation(id, constant, linear, quadric)
struct SetLightAttenuationF {
    void operator() (Light* light, const Vector3& vector) const
    {
        light->SetAttenuation(vector.GetX(), vector.GetY(), vector.GetZ());
    }
};

CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(LuaBinding3D, Light, SetLightAttenuation, SetVector3, SetLightAttenuationF)

// constant, linear, quadric = Candera.GetLightAttenuation(id)
struct GetLightAttenuationF {
    Vector3 operator() (const Light* light) const
    {
        const Light::Attenuation& attenuation = light->GetAttenuation();
        return Vector3(attenuation.constant, attenuation.linear, attenuation.quadratic);
    }
};

CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(LuaBinding3D, Light, GetLightAttenuation, GetVector3, GetLightAttenuationF)

// Candera.SetLightAttenuationEnabled(isAttenuationEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Light, bool, SetBool, SetLightAttenuationEnabled, SetAttenuationEnabled)

// isAttenuationEnabled = Candera.IsLightAttenuationEnabled()
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Light, bool, GetBool, IsLightAttenuationEnabled, IsAttenuationEnabled)


// ----------------------------------------------------------------------------
// Camera

// Candera.SetCameraLookAtVector(id, x, y, z)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Camera, Vector3, SetVector3, SetCameraLookAtVector, SetLookAtVector)

// x, y, z = Candera.GetCameraLookAtVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraLookAtVector, GetLookAtVector)

// x, y, z = Candera.GetCameraWorldLookAtVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraWorldLookAtVector, GetWorldLookAtVector)

// Candera.SetCameraUpVector(id, x, y, z)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Camera, Vector3, SetVector3, SetCameraUpVector, SetUpVector)

// x, y, z = Candera.GetCameraUpVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraUpVector, GetUpVector)

// x, y, z = Candera.GetCameraWorldUpVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraWorldUpVector, GetWorldUpVector)

// x, y, z = Candera.GetCameraRightVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraRightVector, GetRightVector)

// x, y, z = Candera.GetCameraWorldRightVector(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, Vector3, GetVector3, GetCameraWorldRightVector, GetWorldRightVector)

// Candera.CameraLookAtWorldPoint(id, x, y, z)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Camera, Vector3, SetVector3, CameraLookAtWorldPoint, LookAtWorldPoint)

// Candera.SetCameraViewingFrustumCullingEnabled(id, isEnabled)
CANDERA_LUASCRIPTSYSTEM_SET(LuaBinding3D, Camera, bool, SetBool, SetCameraViewingFrustumCullingEnabled, SetViewingFrustumCullingEnabled)

// isEnabled = Candera.IsCameraViewingFrustumCullingEnabled(id)
CANDERA_LUASCRIPTSYSTEM_GET(LuaBinding3D, Camera, bool, GetBool, IsCameraViewingFrustumCullingEnabled, IsViewingFrustumCullingEnabled)

// Candera.CameraRotateAroundWorldAxis(id, axisX, axisY, axisZ, angleInDegrees)
Int LuaBinding3D::CameraRotateAroundWorldAxis(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float x = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float y = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float z = static_cast<Float>(luaL_checknumber(luaState, 4));
    Float angle = static_cast<Float>(luaL_checknumber(luaState, 5));

    if (entity->IsTypeOf(Camera::GetTypeId())) {
        Camera* node = static_cast<Camera*>(entity);
        if (node->RotateAroundWorldAxis(Vector3(x, y, z), angle)) {
            lua_pushboolean(luaState, 1);
            return 1;
        }
    }

    return 0;
}


// ----------------------------------------------------------------------------
// ShaderParamSetter

// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Float, x)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec2, x, y)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec3, x, y, z)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.FloatVec4, x, y, z, w)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Integer, x)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec2, x, y)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec3, x, y, z)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.IntegerVec4, x, y, z, w)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Bool, x)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec2, x, y)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec3, x, y, z)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.BoolVec4, x, y, z, w)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.Sampler2D, x)
// Candera.SetShaderUniform(id, 'uniformName', Candera.Shader.SamplerCube, x)
Int LuaBinding3D::SetShaderUniform(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    const Char* uniformName = luaL_checkstring(luaState, 2);
    if (0 == uniformName) {
        return 0;
    }

    Shader::UniformType uniformType = static_cast<Shader::UniformType>(luaL_checkinteger(luaState, 3));

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            AbstractShaderParamSetter::SharedPointer abstractShaderParamSetter = appearance->GetShaderParamSetter();
            ShaderParamSetter* shaderParamSetter = Dynamic_Cast<ShaderParamSetter*>(abstractShaderParamSetter.GetPointerToSharedInstance());
            if (0 != shaderParamSetter) {

                // TODO: support FloatMat2, FloatMat3, FloatMat4 and arrays once data types for matrices have been added.
                Int32 vecInt[4];
                Float vecFloat[4];

                switch (uniformType) {

                case Shader::FloatVec4:
                    vecFloat[3] = static_cast<Float>(luaL_checknumber(luaState, 7));
                    FEATSTD_LINT_FALLTHROUGH("Processing 3 more values")

                case Shader::FloatVec3:
                    vecFloat[2] = static_cast<Float>(luaL_checknumber(luaState, 6));
                    FEATSTD_LINT_FALLTHROUGH("Processing 2 more values")

                case Shader::FloatVec2:
                    vecFloat[1] = static_cast<Float>(luaL_checknumber(luaState, 5));
                    FEATSTD_LINT_FALLTHROUGH("Processing 1 more value")

                case Shader::Float:
                    vecFloat[0] = static_cast<Float>(luaL_checknumber(luaState, 4));

                    static_cast<void>(shaderParamSetter->SetUniform(uniformName, uniformType, vecFloat, 1, true));
                    break;


                case Shader::IntegerVec4:
                    vecInt[3] = static_cast<Int>(luaL_checkinteger(luaState, 7));
                    FEATSTD_LINT_FALLTHROUGH("Processing 3 more values")

                case Shader::IntegerVec3:
                    vecInt[2] = static_cast<Int>(luaL_checkinteger(luaState, 6));
                    FEATSTD_LINT_FALLTHROUGH("Processing 2 more values")

                case Shader::IntegerVec2:
                    vecInt[1] = static_cast<Int>(luaL_checkinteger(luaState, 5));
                    FEATSTD_LINT_FALLTHROUGH("Processing 1 more value")

                case Shader::Sampler2D:
                    FEATSTD_LINT_FALLTHROUGH("Different type, same behavior")

                case Shader::SamplerCube:
                    FEATSTD_LINT_FALLTHROUGH("Different type, same behavior")

                case Shader::Integer:
                    vecInt[0] = static_cast<Int>(luaL_checkinteger(luaState, 4));

                    static_cast<void>(shaderParamSetter->SetUniform(uniformName, uniformType, vecInt, 1, true));
                    break;


                case Shader::BoolVec4:
                    vecInt[3] = lua_toboolean(luaState, 7);
                    FEATSTD_LINT_FALLTHROUGH("Processing 3 more values")

                case Shader::BoolVec3:
                    vecInt[2] = lua_toboolean(luaState, 6);
                    FEATSTD_LINT_FALLTHROUGH("Processing 2 more values")

                case Shader::BoolVec2:
                    vecInt[1] = lua_toboolean(luaState, 5);
                    FEATSTD_LINT_FALLTHROUGH("Processing 1 more value")

                case Shader::Bool:
                    vecInt[0] = lua_toboolean(luaState, 4);

                    static_cast<void>(shaderParamSetter->SetUniform(uniformName, uniformType, vecInt, 1, true));
                    break;

                default:
                    FEATSTD_LOG_ERROR("Trying to set unkown/unsupported uniform type for '%s' at appearance '%s'.", uniformName, appearance->GetName());
                    break;
                }
            }
        }
    }

    return 0;
}

// local uniformType = Candera.GetShaderUniformType(id, 'uniformName')
Int LuaBinding3D::GetShaderUniformType(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    const Char* uniformName = luaL_checkstring(luaState, 2);
    if (0 == uniformName) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            AbstractShaderParamSetter::SharedPointer abstractShaderParamSetter = appearance->GetShaderParamSetter();
            ShaderParamSetter* shaderParamSetter = Dynamic_Cast<ShaderParamSetter*>(abstractShaderParamSetter.GetPointerToSharedInstance());
            if (shaderParamSetter != 0) {
                Shader::UniformType uniformType;
                if (shaderParamSetter->GetUniformType(uniformName, uniformType)) {
                    lua_pushinteger(luaState, static_cast<lua_Integer>(uniformType));
                    return 1;
                }
            }
        }
    }

    return 0;
}

// local x = Candera.GetShaderUniform(id, 'uniformName')
// local x,y = Candera.GetShaderUniform(id, 'uniformName')
// local x,y,z = Candera.GetShaderUniform(id, 'uniformName')
// local x,y,z,w = Candera.GetShaderUniform(id, 'uniformName')
Int LuaBinding3D::GetShaderUniform(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    const Char* uniformName = luaL_checkstring(luaState, 2);
    if (0 == uniformName) {
        return 0;
    }

    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Appearance::SharedPointer appearance = node->GetAppearance();
        if (appearance != 0) {
            AbstractShaderParamSetter::SharedPointer abstractShaderParamSetter = appearance->GetShaderParamSetter();
            ShaderParamSetter* shaderParamSetter = Dynamic_Cast<ShaderParamSetter*>(abstractShaderParamSetter.GetPointerToSharedInstance());
            if (shaderParamSetter != 0) {
                Shader::UniformType uniformType;
                if (shaderParamSetter->GetUniformType(uniformName, uniformType)) {

                    Float* floatData = 0;
                    Int32* integerData = 0;

                    if (shaderParamSetter->IsFloatType(uniformType)) {
                        if (!shaderParamSetter->GetUniformData(uniformName, floatData)) {
                            return 0;
                        }
                    }
                    else {
                        FEATSTD_DEBUG_ASSERT(shaderParamSetter->IsIntegerType(uniformType));
                        if (!shaderParamSetter->GetUniformData(uniformName, integerData)) {
                            return 0;
                        }
                    }

                    Int topOfStack = static_cast<Int>(lua_gettop(luaState));

                    // TODO: support FloatMat2, FloatMat3, FloatMat4 and arrays once data types for matrices have been added.
                    switch (uniformType) {

                    case Shader::Float:
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[0]));
                        break;

                    case Shader::FloatVec2:
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[0]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[1]));
                        break;

                    case Shader::FloatVec3:
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[0]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[1]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[2]));
                        break;

                    case Shader::FloatVec4:
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[0]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[1]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[2]));
                        lua_pushnumber(luaState, static_cast<lua_Number>(floatData[3]));
                        break;


                    case Shader::Sampler2D:
                        FEATSTD_LINT_FALLTHROUGH("Different type, same behavior")

                    case Shader::SamplerCube:
                        FEATSTD_LINT_FALLTHROUGH("Different type, same behavior")

                    case Shader::Integer:
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[0]));
                        break;

                    case Shader::IntegerVec2:
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[0]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[1]));
                        break;

                    case Shader::IntegerVec3:
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[0]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[1]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[2]));
                        break;

                    case Shader::IntegerVec4:
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[0]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[1]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[2]));
                        lua_pushinteger(luaState, static_cast<lua_Integer>(integerData[3]));
                        break;


                    case Shader::Bool:
                        lua_pushboolean(luaState, integerData[0]);
                        break;

                    case Shader::BoolVec2:
                        lua_pushboolean(luaState, integerData[0]);
                        lua_pushboolean(luaState, integerData[1]);
                        break;

                    case Shader::BoolVec3:
                        lua_pushboolean(luaState, integerData[0]);
                        lua_pushboolean(luaState, integerData[1]);
                        lua_pushboolean(luaState, integerData[2]);
                        break;

                    case Shader::BoolVec4:
                        lua_pushboolean(luaState, integerData[0]);
                        lua_pushboolean(luaState, integerData[1]);
                        lua_pushboolean(luaState, integerData[2]);
                        lua_pushboolean(luaState, integerData[3]);
                        break;


                    default:
                        FEATSTD_LOG_ERROR("Trying to set unkown/unsupported uniform type for '%s' at appearance '%s'.", uniformName, appearance->GetName());
                        break;
                    }

                    return (static_cast<Int>(lua_gettop(luaState)) - topOfStack);
                }
            }
        }
    }

    return 0;
}

} // namespace Internal

} // namespace Scripting

} // namespace Candera
