//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_SCRIPTING_LUABINDING_H
#define CANDERA_SCRIPTING_LUABINDING_H

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <CanderaScripting/LuaScriptSystem.h>

namespace Candera {

namespace Scripting {

namespace Internal {

/** @addtogroup LuaScripting
 *  @{
 */

/**
 * @brief  LuaBinding exposes functions in Lua, regardless of CANDERA_3D_ENABLED/CANDERA_2D_ENABLED defines.
 *         LuaBinding2D exposes functions that are only available if CANDERA_2D_ENABLED is defined.
 *         LuaBinding3D exposes functions that are only available if CANDERA_3D_ENABLED is defined.
 *         LuaBinding3D2D exposes functions that are available for 3D and 2D  (e.g. the Lua function
 *         Candera.GetPosition() works on Transformable and Transformable2D) with regard to defines.
 */

class LuaBinding
{
public:
    struct CanderaReferenceType
    {
        const Char* m_name;
        TypeId m_typeId;
    };

    static Int OpenCanderaLib(lua_State* luaState);
    static void Init(lua_State* luaState);

    static bool AddFunctionsToTable(lua_State* luaState, const Char* libTableName, const luaL_Reg* functions);
    static void AddCanderaReferenceType(lua_State* luaState, const CanderaReferenceType* referenceTypes, SizeType referenceTypesCount);
    static void AddMethods(lua_State* luaState, TypeId typeId, const luaL_Reg* functions);
    static Int CreateObjectReference(lua_State* luaState, CanderaObject* object);
    template<typename T> static T* GetTypedPointer(lua_State* luaState);

protected:
    static const CanderaReferenceType s_canderaObjectReferenceTypes[];

    static ScriptEntity* GetEntity(lua_State* luaState, Int arg = 1);
    static Int ReturnIdsAsTable(lua_State* luaState, Candera::Internal::Vector<ScriptSystem::Handle>& componentContainer);

    // Template helper functions.
    // T or T3D/T2D refers to type (e.g. Node or Node/Node2D). F or F3D/F3D refers to corresponding functor.
    // Intended purpose is to produce in-lined code with as little source code duplication as possible.
    // In case of code bloat, turn functors into functions, and pass as function pointer instead of functor.

    template<typename T, typename F> static Int SetBool(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetBool(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetColor(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetColor(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetFloat(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetFloat(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetInt(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetInt(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetVector3(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetVector3(lua_State* luaState, F f);
    template<typename T, typename F> static Int SetVector2(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetVector2(lua_State* luaState, F f);
    template<typename T, typename F> static Int GetRectangle(lua_State* luaState, F f);
    template<typename T, typename TT, typename F> static Int SetTuple4(lua_State* luaState, F f);
    template<typename F> static Int IsF(lua_State* luaState, F f);

    // Helper macros
#define CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR(objectType, dataType, functorName, functionName)                              \
struct functorName {                                                                                                      \
    void operator() (objectType* object, const dataType& value) const { static_cast<void>(object->functionName(value)); } \
};

#define CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(bindingClass, objectType, luaFunctionName, functionName, functorName) \
Int bindingClass::luaFunctionName(lua_State* luaState)                                                           \
{                                                                                                                \
    return functionName<objectType>(luaState, functorName());                                                    \
}

#define CANDERA_LUASCRIPTSYSTEM_FUNCTION(bindingClass, luaFunctionName, functionName, functorName) \
Int bindingClass::luaFunctionName(lua_State* luaState)                                             \
{                                                                                                  \
    return functionName(luaState, functorName());                                                  \
}

// Setter of data in object
#define CANDERA_LUASCRIPTSYSTEM_SET(bindingClass, objectType, dataType, setter, luaFunctionName, cppFunctionName)          \
CANDERA_LUASCRIPTSYSTEM_SET_FUNCTOR(objectType, dataType, FEATSTD_CONCAT2(luaFunctionName, F), cppFunctionName)            \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(bindingClass, objectType, luaFunctionName, setter, FEATSTD_CONCAT2(luaFunctionName, F))

// Getter of data in object
#define CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR(objectType, dataType, functorName, functionName) \
struct functorName {                                                                         \
    dataType operator() (const objectType* object) const { return object->functionName(); }  \
};

#define CANDERA_LUASCRIPTSYSTEM_GET(bindingClass, objectType, dataType, getter, luaFunctionName, cppFunctionName)          \
CANDERA_LUASCRIPTSYSTEM_GET_FUNCTOR(objectType, dataType, FEATSTD_CONCAT2(luaFunctionName, F), cppFunctionName)            \
CANDERA_LUASCRIPTSYSTEM_FUNCTION_T(bindingClass, objectType, luaFunctionName, getter, FEATSTD_CONCAT2(luaFunctionName, F))

// Call object's functionCallName and push the returned bool on the Lua stack.
#define CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(functionName, objectType, functionCallName) \
Int LuaBinding::functionName(lua_State* luaState)                                                    \
{                                                                                                    \
    objectType* object = GetTypedPointer<objectType>(luaState);                                      \
    if (0 != object) {                                                                               \
        lua_pushboolean(luaState, object->functionCallName() ? 1 : 0);                               \
        return 1;                                                                                    \
    }                                                                                                \
                                                                                                     \
    return 0;                                                                                        \
}

// Call object's functionCallName and push the returned Float on the Lua stack.
#define CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_FLOAT(functionName, objectType, functionCallName) \
Int LuaBinding::functionName(lua_State* luaState)                                                     \
{                                                                                                     \
    objectType* object = GetTypedPointer<objectType>(luaState);                                       \
    if (0 != object) {                                                                                \
        Double returnValue = static_cast<Double>(object->functionCallName());                         \
        lua_pushnumber(luaState, static_cast<lua_Number>(returnValue));                               \
        return 1;                                                                                     \
    }                                                                                                 \
                                                                                                      \
    return 0;                                                                                         \
}

// Call object's functionCallName and push the returned Int on the Lua stack.
#define CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_INT(functionName, objectType, functionCallName) \
Int LuaBinding::functionName(lua_State* luaState)                                                   \
{                                                                                                   \
    objectType* object = GetTypedPointer<objectType>(luaState);                                     \
    if (0 != object) {                                                                              \
        Int returnValue = object->functionCallName();                                               \
        lua_pushinteger(luaState, static_cast<lua_Integer>(returnValue));                           \
        return 1;                                                                                   \
    }                                                                                               \
                                                                                                    \
    return 0;                                                                                       \
}

// Call object's functionCallName and push the returned Int on the Lua stack.
#define CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_SET_INT(functionName, objectType, functionCallName) \
Int LuaBinding::functionName(lua_State* luaState)                                                   \
{                                                                                                   \
    objectType* object = GetTypedPointer<objectType>(luaState);                                     \
    if (0 != object) {                                                                              \
        Int value = static_cast<Int>(luaL_checkinteger(luaState, 2));                               \
        object->functionCallName(value);                                                            \
    }                                                                                               \
                                                                                                    \
    return 0;                                                                                       \
}

// Call object's functionCallName and push the returned Int on the Lua stack.
#define CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_SET_FLOAT(functionName, objectType, functionCallName) \
Int LuaBinding::functionName(lua_State* luaState)                                                     \
{                                                                                                     \
    objectType* object = GetTypedPointer<objectType>(luaState);                                       \
    if (0 != object) {                                                                                \
        Float value = static_cast<Float>(luaL_checknumber(luaState, 2));                              \
        object->functionCallName(value);                                                              \
    }                                                                                                 \
                                                                                                      \
    return 0;                                                                                         \
}


private:
    static const luaL_Reg s_canderaLibFunctions[];
    static const luaL_Reg s_animationMethods[];

    FEATSTD_MAKE_CLASS_STATIC(LuaBinding);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaLogInfo LogInfo
     *  \code
     *  Candera.LogInfo(message)
     *  \endcode
     *  Outputs 'message' to Candera's info log.
     ** @} */
    static Int LogInfo(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions
    * @{
     *  <hr>\section CanderaLuaLogError LogError
     *  \code
     *  Candera.LogError(message)
     *  \endcode
     *  Outputs 'message' to Candera's error log.
     ** @} */
    static Int LogError(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaLogWarning LogWarning
     *  \code
     *  Candera.LogWarning(message)
     *  \endcode
     *  Outputs 'message' to Candera's warning log.
     ** @} */
    static Int LogWarning(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaGetScriptIds GetScriptIds
     *  \code
     *  local ids = Candera.GetScriptIds(id)
     *  \endcode
     *  Returns a table with all ids of script components that are attached to the node of the script component identified by 'id'.
     ** @} */
    static Int GetScriptIds(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaGetChildScriptIds GetChildScriptIds
     *  \code
     *  local ids = Candera.GetChildScriptIds(id)
     *  \endcode
     *  Returns a table with all ids of script components that are attached to child nodes of the node of the script component identified by 'id'.
     ** @} */
    static Int GetChildScriptIds(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaIs2D Is2D
     *  \code
     *  local is2D = Candera.Is2D(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' is a 2D node, 'nil' otherwise.
     ** @} */
    static Int Is2D(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaIs3D Is3D
     *  \code
     *  local is2D = Candera.Is3D(id)
     *  \endcode
     *  Returns 'true' if the node of the script component identified by 'id' is a 3D node, 'nil' otherwise.
     ** @} */
    static Int Is3D(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaGetName GetName
     *  \code
     *  local name = Candera.GetName(id)
     *  \endcode
     *  Returns the name of the node of the script component identified by 'id' as a string, or 'nil' if that node has no name.
     ** @} */
    static Int GetName(lua_State* luaState);    // CanderaObject::GetName

    /** @addtogroup LuaBindingGeneralFunctions 
     * @{
     *  <hr>\section CanderaLuaSetEnabled SetEnabled
     *  \code
     *  Candera.SetEnabled(id, isEnabled)
     *  \endcode
     *  Enables or disables the script component identified by 'id'. If 'isEnabled' evaluates to 'true', the script component is enabled,
     *  and the script's \ref LuaTutorialFunctionsCalledByCandera "OnEnable" callback is executed. If 'isEnabled' evaluates to 'false', the script component is disabled, and the script's \ref LuaTutorialFunctionsCalledByCandera "OnDisable" callback is executed.
     ** @} */

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaLuaIsEnabled IsEnabled
     *  \code
     *  local isEnabled = Candera.IsEnabled(id)
     *  \endcode
     *  Returns 'true' if the script component identified by 'id' is enabled, or 'false' if it is disabled.
     ** @} */

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaLuaAddEventListener AddEventListener
     *  \code
     *  local success = Candera.AddEventListener(id)
     *  \endcode
     *  Add the script component identified by 'id' as an event listener to the script system.
     *  Returns 'true' if the script component was successfully added as an event listener, or 'nil' if the operation failed.
     ** @} */
    static Int AddEventListener(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaLuaRemoveEventListener RemoveEventListener
     *  \code
     *  local success = Candera.RemoveEventListener(id)
     *  \endcode
     *  Remove the script component identified by 'id' as an event listener from the script system.
     *  Returns 'true' if the script component was successfully removed as an event listener, or 'nil' if the operation failed.
     ** @} */
    static Int RemoveEventListener(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaLuaDispatchEvent DispatchEvent
     *  \code
     *  local success = Candera.DispatchEvent(event)
     *  \endcode
     *  Dispatch an event to all EventListeners, which can be other script instances and/or native code.
     *  The 'event' dispatched is a Lua table that can contain integer and floating point values, boolean
     *  values, zero-terminated strings, and Candera.ReferenceType references in any order or quantity.
     *  See \ref LuaScriptingEvents "Lua Script Events" for details.
     *  Returns 'true' if the event was successfully dispatched, or 'nil' if the operation failed.
     ** @} */
    static Int DispatchEvent(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaCreateReference CreateReference
     *  \code
     *  local objectReference = Candera.CreateReference(Candera.ReferenceType.xxx)
     *  \endcode
     *  Create a null reference to a CanderaObject. Public object references can be set and retrieved by native code users in a
     *  type-safe fashion. (Remember that Lua is dynamically typed, thus variables can change types at runtime)
     *  Returns a null reference to a CanderaObject of type 'xxx'.
     ** @} */
    static Int CreateReference(lua_State* luaState);

    /** @addtogroup LuaBindingGeneralFunctions
     * @{
     *  <hr>\section CanderaIsNullReference IsNullReference
     *  \code
     *  local isNullReference = Candera.IsNullReference(variable)
     *  \endcode
     *  Returns 'true' if the variable of Candera.ReferenceType is not referencing any object, 'false' otherwise.
     ** @} */
    static Int IsNullReference(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationStart Start
     *  \code
     *  local isStarted = animation:Start()
     *  \endcode
     *  Start animation playback.
     *  Returns 'true' if the animation was started successfully, 'false' otherwise.
     ** @} */
    static Int StartAnimation(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationStop Stop
     *  \code
     *  local wasEnabled = animation:Stop()
     *  \endcode
     *  Stop animation playback.
     *  Returns 'true' if the animation was enabled before stopping, 'false' otherwise.
     ** @} */
    static Int StopAnimation(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationPause Pause
     *  \code
     *  local isPaused = animation:Pause()
     *  \endcode
     *  Pause animation playback.
     *  Returns 'true' if the animation was paused successfully, 'false' otherwise.
     ** @} */
    static Int PauseAnimation(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationResume Resume
     *  \code
     *  local isResumed = animation:Resume()
     *  \endcode
     *  Resume playback of a paused animation.
     *  Returns 'true' if the animation was resumed successfully, 'false' otherwise.
     ** @} */
    static Int ResumeAnimaton(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationIsPaused IsPaused
     *  \code
     *  local isPaused = animation:IsPaused()
     *  \endcode
     *  Returns 'true' if the animation is paused, 'false' otherwise.
     ** @} */
    static Int IsAnimationPaused(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationIsPlaying IsPlaying
     *  \code
     *  local isPlaying = animation:IsPlaying()
     *  \endcode
     *  Returns 'true' if the animation is playing, 'false' otherwise.
     ** @} */
    static Int IsAnimationEnabled(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationGetSpeedFactor GetSpeedFactor
     *  \code
     *  local speedFactor = animation:GetSpeedFactor()
     *  \endcode
     *  Returns the speed factor of the animation playback. Negative values mean reverse playback.
     ** @} */
    static Int GetAnimationSpeedFactor(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationSetSpeedFactor SetSpeedFactor
     *  \code
     *  animation:SetSpeedFactor(speedFactor)
     *  \endcode
     *  Sets the speed factor of the animation playback. Negative values mean reverse playback.
     ** @} */
    static Int SetAnimationSpeedFactor(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationGetRepeatCount GetRepeatCount
     *  \code
     *  local repeatCount = animation:GetRepeatCount()
     *  \endcode
     *  Returns how often the animation will be played back.
     ** @} */
    static Int GetAnimationRepeatCount(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationSetRepeatCount SetRepeatCount
     *  \code
     *  animation:SetRepeatCount(repeatCount)
     *  \endcode
     *  Sets how often the animation will be played back.
     ** @} */
    static Int SetAnimationRepeatCount(lua_State* luaState);

    /** @addtogroup LuaBindingAnimation
     * @{
     *  <hr>\section LuaAnimationFinish Finish
     *  \code
     *  bool willBeFinished = animation:Finish(timeToFinishInSeconds)
     *  \endcode
     *  Finish animation playback within the given time.
     *  Returns 'true' if the animation playback will be finished within the given time, 'false' otherwise.
     ** @} */
    static Int AnimationFinish(lua_State* luaState);
};

template<typename T>
T* LuaBinding::GetTypedPointer(lua_State* luaState)
{
    return static_cast<T*>(LuaScriptSystem::GetPointerFromLuaObjectReference(luaState));
}

template<typename T, typename F>
Int LuaBinding::SetBool(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        bool value = (lua_toboolean(luaState, 2) != 0);
        f(object, value);
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetBool(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        bool value = f(object);
        lua_pushboolean(luaState, (value ? 1 : 0));
        return 1;
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::SetColor(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float red = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float green = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float blue = static_cast<Float>(luaL_checknumber(luaState, 4));
    Float alpha = static_cast<Float>(luaL_checknumber(luaState, 5));

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        Color color(red, green, blue, alpha);
        f(object, color);
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetColor(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        const Color& color = f(object);
        lua_pushnumber(luaState, color.GetRed());
        lua_pushnumber(luaState, color.GetGreen());
        lua_pushnumber(luaState, color.GetBlue());
        lua_pushnumber(luaState, color.GetAlpha());
        return 4;
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::SetFloat(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float value = static_cast<Float>(luaL_checknumber(luaState, 2));
    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        f(object, value);
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetFloat(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        Float value = f(object);
        lua_pushnumber(luaState, value);
        return 1;
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::SetInt(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Int value = static_cast<Int>(luaL_checkinteger(luaState, 2));
    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        f(object, value);
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetInt(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        Int value = f(object);
        lua_pushinteger(luaState, value);
        return 1;
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::SetVector3(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float x = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float y = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float z = static_cast<Float>(luaL_checknumber(luaState, 4));

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        f(object, Vector3(x, y, z));
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetVector3(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        const Vector3& vector = f(object);
        lua_pushnumber(luaState, vector.GetX());
        lua_pushnumber(luaState, vector.GetY());
        lua_pushnumber(luaState, vector.GetZ());
        return 3;
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::SetVector2(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float x = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float y = static_cast<Float>(luaL_checknumber(luaState, 3));

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        f(object, Vector2(x, y));
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetVector2(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        const Vector2& vector = f(object);
        lua_pushnumber(luaState, vector.GetX());
        lua_pushnumber(luaState, vector.GetY());
        return 2;
    }

    return 0;
}

template<typename T, typename TT, typename F>
Int LuaBinding::SetTuple4(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Float value1 = static_cast<Float>(luaL_checknumber(luaState, 2));
    Float value2 = static_cast<Float>(luaL_checknumber(luaState, 3));
    Float value3 = static_cast<Float>(luaL_checknumber(luaState, 4));
    Float value4 = static_cast<Float>(luaL_checknumber(luaState, 5));

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        TT tuple(value1, value2, value3, value4);
        f(object, tuple);
    }

    return 0;
}

template<typename T, typename F>
Int LuaBinding::GetRectangle(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (entity->IsTypeOf(T::GetTypeId())) {
        T* object = static_cast<T*>(entity);
        const Rectangle& value = f(object);
        lua_pushnumber(luaState, value.GetLeft());
        lua_pushnumber(luaState, value.GetTop());
        lua_pushnumber(luaState, value.GetWidth());
        lua_pushnumber(luaState, value.GetHeight());
        return 4;
    }

    return 0;
}

template<typename F>
Int LuaBinding::IsF(lua_State* luaState, F f)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    if (f(entity)) {
        lua_pushboolean(luaState, 1);
        return 1;
    }

    return 0;
}

/** @} */ // end of LuaScripting

} // namespace Internal

} // namespace Scripting

} // namespace Candera

#endif
