//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LuaBinding.h"

#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaScripting/ScriptEvent.h>
#include <CanderaScripting/ScriptParameters.h>

#if defined(CANDERA_2D_ENABLED)
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <CanderaScripting/Lua/LuaBinding2D.h>
#endif

#if defined(CANDERA_3D_ENABLED)
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <CanderaScripting/Lua/LuaBinding3D.h>
#endif

#if defined(CANDERA_3D_ENABLED) || defined(CANDERA_2D_ENABLED)
#include <CanderaScripting/Lua/LuaBinding3D2D.h>
#endif

namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaScripting);

namespace Scripting {

namespace Internal {

// ----------------------------------------------------------------------------
// Candera functions exposed in Lua

const luaL_Reg LuaBinding::s_canderaLibFunctions[] = {
    // Logging
    { "LogInfo", LogInfo },
    { "LogError", LogError },
    { "LogWarning", LogWarning },
    // ScripComponent
    { "GetScriptIds", GetScriptIds },
    { "GetChildScriptIds", GetChildScriptIds },
// SetEnabled (implemented in Lua, no binding required)
// IsEnabled (implemented in Lua, no binding required)
    // Transformable(2D)
    { "Is2D", Is2D },
    { "Is3D", Is3D },
    { "GetName", GetName },
    // ScriptListener
    { "AddEventListener", AddEventListener },
    { "RemoveEventListener", RemoveEventListener },
    { "DispatchEvent", DispatchEvent },
    // ObjectReference
    { "CreateReference", CreateReference },
    { "IsNullReference", IsNullReference },
    { NULL, NULL } // sentinel
};

const LuaBinding::CanderaReferenceType LuaBinding::s_canderaObjectReferenceTypes[] =
{
    { "CanderaObject", CanderaObject::GetTypeId() },
    { "Animation", Animation::AnimationPlayerBase::GetTypeId() }
};

// Animation
const luaL_Reg LuaBinding::s_animationMethods[] = {
    { "Start", StartAnimation },
    { "Stop", StopAnimation },
    { "Pause", PauseAnimation },
    { "Resume", ResumeAnimaton },
    { "IsPaused", IsAnimationPaused },
    { "IsPlaying", IsAnimationEnabled },
    { "GetSpeedFactor", GetAnimationSpeedFactor },
    { "SetSpeedFactor", SetAnimationSpeedFactor },
    { "GetRepeatCount", GetAnimationRepeatCount },
    { "SetRepeatCount", SetAnimationRepeatCount },
    { "Finish", AnimationFinish },
    { NULL, NULL }
};


Int LuaBinding::OpenCanderaLib(lua_State* luaState)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1960, "Violates MISRA C++ 2008 Required Rule 5-18-1. Third party macro")
    luaL_newlib(luaState, s_canderaLibFunctions);

#if defined(CANDERA_3D_ENABLED) || defined(CANDERA_2D_ENABLED)
    Internal::LuaBinding3D2D::AddFunctions(luaState);
#endif

#if defined(CANDERA_3D_ENABLED)
    Internal::LuaBinding3D::AddFunctions(luaState);
#endif

#if defined(CANDERA_2D_ENABLED)
    Internal::LuaBinding2D::AddFunctions(luaState);
#endif

    return 1;
}

void LuaBinding::AddMethods(lua_State* luaState, TypeId typeId, const luaL_Reg* functions)
{
    Int luaType = luaL_getmetatable(luaState, typeId);
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);
    luaL_setfuncs(luaState, functions, 0);
}

Int LuaBinding::CreateObjectReference(lua_State* luaState, CanderaObject* object)
{
    LuaScriptSystem* scriptSystem = LuaScriptSystem::s_this;
    FEATSTD_DEBUG_ASSERT(0 != scriptSystem);
    TypeId typeId = (0 == object) ? TypeId() : object->GetDynamicTypeId();
    Internal::ObjectReferenceSystem::Handle objectReference = scriptSystem->GetObjectReference(object);
    return (LuaScriptSystem::s_this->CreateLuaObjectReference(luaState, typeId, objectReference)) ? 1 : 0;
}

void LuaBinding::Init(lua_State* luaState)
{
    // Create the meta table that identifies Lua Candera.ReferenceType user data.
    Int success = static_cast<Int>(luaL_newmetatable(luaState, LuaScriptSystem::s_canderaInternalObjectReferenceTypeMetaTable));
    FEATSTD_DEBUG_ASSERT(0 != success); // Assert that there is no meta table name collision.
    FEATSTD_UNUSED(success);
    lua_pop(luaState, 1);

    // Set relevant ObjectReference types in Candera.ReferenceType table.
    const SizeType canderaObjectReferenceTypesSize = sizeof(s_canderaObjectReferenceTypes) / sizeof(CanderaReferenceType);
    AddCanderaReferenceType(luaState, &(s_canderaObjectReferenceTypes[0]), canderaObjectReferenceTypesSize);

    // Bind Animation
    AddMethods(luaState, Animation::AnimationPlayerBase::GetTypeId(), s_animationMethods);

#if defined(CANDERA_3D_ENABLED)
    Internal::LuaBinding3D::Init3D(luaState);
#else
    FEATSTD_UNUSED(luaState);
#endif

#if defined(CANDERA_2D_ENABLED)
    Internal::LuaBinding2D::Init2D(luaState);
#else
    FEATSTD_UNUSED(luaState);
#endif
}

bool LuaBinding::AddFunctionsToTable(lua_State* luaState, const Char* libTableName, const luaL_Reg* functions)
{
    if (StringPlatform::CompareStrings(libTableName, LuaScriptSystem::s_canderaName) == 0) {
        return false; // Do not allow binding to "Candera" from external users.
    }

    if (StringPlatform::CompareStrings(libTableName, LuaScriptSystem::s_canderaInternalName) == 0) {
        return false; // Do not allow binding to "Candera_Internal" from external users.
    }

    bool wasSuccessful = true;
    Int luaType = lua_getglobal(luaState, libTableName);
    if (LUA_TTABLE == luaType) {
        // expects the table to be on top of the stack to add the functions to
        luaL_setfuncs(luaState, functions, 0);
    }
    else if (LUA_TNIL == luaType) {
        luaL_newlibtable(luaState, libTableName);
        luaL_setfuncs(luaState, functions, 0);
        lua_setglobal(luaState, libTableName);
    }
    else {
        // variable by that name already exists, but is not a table
        wasSuccessful = false;
    }

    lua_pop(luaState, 1);

    return wasSuccessful;
}

ScriptEntity* LuaBinding::GetEntity(lua_State* luaState, Int arg)
{
    UInt32 handleRaw = static_cast<UInt32>(luaL_checkinteger(luaState, arg));
    FEATSTD_DEBUG_ASSERT(0 != LuaScriptSystem::s_this);
    const ScriptComponent* component = LuaScriptSystem::s_this->GetPointer(LuaScriptSystem::s_this->CastToHandle(handleRaw));
    if (0 == component) {
        FEATSTD_LOG_ERROR("Component id '%d' is not valid. No action was performed.", handleRaw);
        return 0;
    }

    return component->GetEntity();
}

Int LuaBinding::ReturnIdsAsTable(lua_State* luaState, Candera::Internal::Vector<ScriptSystem::Handle>& componentContainer)
{
    if (componentContainer.Size() <= 0) {
        return 0;
    }

    lua_createtable(luaState, static_cast<Int>(componentContainer.Size()), 0);
    for (SizeType i = 0; i < componentContainer.Size(); ++i) {
        FEATSTD_DEBUG_ASSERT((0 != LuaScriptSystem::s_this->GetPointer(componentContainer[i])));
        lua_pushnumber(luaState, static_cast<lua_Number>(i + 1));
        lua_pushinteger(luaState, static_cast<lua_Integer>(LuaScriptSystem::s_this->GetHandleRaw(componentContainer[i])));
        lua_settable(luaState, -3);
    }

    return 1;
}

void LuaBinding::AddCanderaReferenceType(lua_State* luaState, const CanderaReferenceType* referenceTypes, SizeType referenceTypesCount)
{
    Int luaType = lua_getglobal(luaState, LuaScriptSystem::GetLuaCanderaName()); // Candera table
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    luaType = lua_getfield(luaState, -1, "ReferenceType"); // Candera.ReferenceType table
    FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
    FEATSTD_UNUSED(luaType);

    for (SizeType i = 0; i < referenceTypesCount; ++i) {
        const SizeType sizeOfUserData = sizeof(TypeId);
        TypeId* typeIdPointer = FeatStd::Internal::PointerToPointer<TypeId*>(lua_newuserdata(luaState, sizeOfUserData));
        if (0 != typeIdPointer) {
            luaType = luaL_getmetatable(luaState, LuaScriptSystem::s_canderaInternalObjectReferenceTypeMetaTable);
            FEATSTD_DEBUG_ASSERT(LUA_TTABLE == luaType);
            FEATSTD_UNUSED(luaType);
            Int success = static_cast<Int>(luaL_newmetatable(luaState, referenceTypes[i].m_typeId));
            FEATSTD_DEBUG_ASSERT(0 != success); // Assert that there is no meta table name collision.
            FEATSTD_UNUSED(success);

            lua_pushvalue(luaState, -1); // Duplicate 'derived' metatable on stack.
            lua_setfield(luaState, -2, "__index"); // Set __index of metatable field to own metatable.

            lua_pushvalue(luaState, -2); // Duplicate 'base' metatable on stack.
            lua_setfield(luaState, -2, "__metatable"); // Set __metatable of 'derived' metatable to 'base' metatable.

            lua_pop(luaState, 1); // Pop 'derived' metatable from stack.

            static_cast<void>(lua_setmetatable(luaState, -2)); // Lua 5.3 reference manual does not describe the meaning of the return value.
            *typeIdPointer = referenceTypes[i].m_typeId;

            // Native code users must be able to add custom reference types after the table has been sealed for script users.
            // Therefore we need to use lua_rawset instead of "lua_setfield(luaState, -2, referenceTypes[i].m_name);"
            static_cast<void>(lua_pushstring(luaState, referenceTypes[i].m_name));
            lua_insert(luaState, -2);
            lua_rawset(luaState, -3);
         }
    }

    lua_pop(luaState, 2);
}


// ----------------------------------------------------------------------------
// Base

// Candera.LogError(msg)
Int LuaBinding::LogError(lua_State* luaState)
{
    const Char* logMessage = luaL_checkstring(luaState, 1);
    FEATSTD_LOG_ERROR("%s", logMessage);

    lua_pop(luaState, 1);
    return 0;
}

// Candera.LogWarning(msg)
Int LuaBinding::LogWarning(lua_State* luaState)
{
    const Char* logMessage = luaL_checkstring(luaState, 1);
    FEATSTD_LOG_WARN("%s", logMessage);

    lua_pop(luaState, 1);
    return 0;
}

// Candera.LogInfo(msg)
Int LuaBinding::LogInfo(lua_State* luaState)
{
    const Char* logMessage = luaL_checkstring(luaState, 1);
    FEATSTD_LOG_INFO("%s", logMessage);

    lua_pop(luaState, 1);
    return 0;
}

// ids = Candera.GetScriptIds(id)
Int LuaBinding::GetScriptIds(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Candera::Internal::Vector<ScriptSystem::Handle> componentContainer;

    EntityComponentSystem::EntitySystem::ComponentIterator<ScriptComponent> componentIterator(entity);
    ScriptSystem::Handle tempHandle;
    while (componentIterator.Get(tempHandle)) {
        FEATSTD_DEBUG_ASSERT(!tempHandle.IsNullHandle());
        static_cast<void>(componentContainer.Add(tempHandle));
        componentIterator.Next();
    }

    return ReturnIdsAsTable(luaState, componentContainer);
}

#ifdef CANDERA_3D_ENABLED
class Traverser3D : public ConstTreeTraverser
{
public:
    Traverser3D(Candera::Internal::Vector<ScriptSystem::Handle>& componentContainer, Node* root)
        : ConstTreeTraverser(), m_componentContainer(componentContainer), m_root(root) {}
protected:
    virtual TraverserAction ProcessNode(const Node& node) override
    {
        if (&node != m_root) {
            EntityComponentSystem::EntitySystem::ComponentIterator<ScriptComponent> componentIterator(&node);
            ScriptSystem::Handle tempHandle;
            while (componentIterator.Get(tempHandle)) {
                FEATSTD_DEBUG_ASSERT(!tempHandle.IsNullHandle());
                static_cast<void>(m_componentContainer.Add(tempHandle));
                componentIterator.Next();
            }
        }

        return ProceedTraversing;
    }
private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(Traverser3D);
    Candera::Internal::Vector<ScriptSystem::Handle>& m_componentContainer;
    Node* m_root;
};
#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
class Traverser2D : public ConstTreeTraverser2D
{
public:
    Traverser2D(Candera::Internal::Vector<ScriptSystem::Handle>& componentContainer, Node2D* root)
        : ConstTreeTraverser2D(), m_componentContainer(componentContainer), m_root(root) {}
protected:
    virtual TraverserAction ProcessNode(const Node2D& node) override
    {
        if (&node != m_root) {
            EntityComponentSystem::EntitySystem::ComponentIterator<ScriptComponent> componentIterator(&node);
            ScriptSystem::Handle tempHandle;
            while (componentIterator.Get(tempHandle)) {
                FEATSTD_DEBUG_ASSERT(!tempHandle.IsNullHandle());
                static_cast<void>(m_componentContainer.Add(tempHandle));
                componentIterator.Next();
            }
        }

        return ProceedTraversing;
    }
private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(Traverser2D);
    Candera::Internal::Vector<ScriptSystem::Handle>& m_componentContainer;
    Node2D* m_root;
};
#endif // CANDERA_2D_ENABLED

// ids = Candera.GetChildScriptIds(id)
Int LuaBinding::GetChildScriptIds(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    Candera::Internal::Vector<ScriptSystem::Handle> componentContainer;

    #ifdef CANDERA_3D_ENABLED
    if (entity->IsTypeOf(Node::GetTypeId())) {
        Node* node = static_cast<Node*>(entity);
        Traverser3D traverser(componentContainer, node);
        traverser.Traverse(*node);
        return ReturnIdsAsTable(luaState, componentContainer);
    }
    #endif // CANDERA_3D_ENABLED

    #ifdef CANDERA_2D_ENABLED
    if (entity->IsTypeOf(Node2D::GetTypeId())) {
        Node2D* node2D = static_cast<Node2D*>(entity);
        Traverser2D traverser(componentContainer, node2D);
        traverser.Traverse(*node2D);
        return ReturnIdsAsTable(luaState, componentContainer);
    }
    #endif // CANDERA_2D_ENABLED

    return 0;
}

struct Is2DF {
    bool operator() (const ScriptEntity* entity) const {
#ifdef CANDERA_2D_ENABLED
        return entity->IsTypeOf(Transformable2D::GetTypeId());
#else
        FEATSTD_UNUSED(entity);
        return false;
#endif
    }
};

struct Is3DF {
    bool operator() (const ScriptEntity* entity) const {
#ifdef CANDERA_3D_ENABLED
        return entity->IsTypeOf(Transformable::GetTypeId());
#else
        FEATSTD_UNUSED(entity);
        return false;
#endif
    }
};

// is2D = Candera.Is2D(id)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding, Is2D, IsF, Is2DF)

// is3D = Candera.Is3D(id)
CANDERA_LUASCRIPTSYSTEM_FUNCTION(LuaBinding, Is3D, IsF, Is3DF)

// name = Candera.GetName(id)
Int LuaBinding::GetName(lua_State* luaState)
{
    ScriptEntity* entity = GetEntity(luaState);
    if (0 == entity) {
        return 0;
    }

    const Char* name = entity->GetName();
    if (0 == name) {
        return 0;
    }

    static_cast<void>(lua_pushstring(luaState, name)); // Returns pointer to internal copy, which we don't need.
    return 1;
}

Int LuaBinding::AddEventListener(lua_State* luaState)
{
    FEATSTD_DEBUG_ASSERT(0 != LuaScriptSystem::s_this);
    ScriptSystem::Handle handle = LuaScriptSystem::s_this->CastToHandle(static_cast<UInt32>(luaL_checkinteger(luaState, 1)));
    if (LuaScriptSystem::s_this->AddScriptEventListener(handle)) {
        lua_pushboolean(luaState, 1);
        return 1;
    }

    return 0;
}

Int LuaBinding::RemoveEventListener(lua_State* luaState)
{
    FEATSTD_DEBUG_ASSERT(0 != LuaScriptSystem::s_this);
    ScriptSystem::Handle handle = LuaScriptSystem::s_this->CastToHandle(static_cast<UInt32>(luaL_checkinteger(luaState, 1)));
    if (LuaScriptSystem::s_this->RemoveScriptEventListener(handle)) {
        lua_pushboolean(luaState, 1);
        return 1;
    }

    return 0;
}

static const Char* s_unsupportedEventTypeErrorMsg = "Candera.DispatchEvent argument uses an unsupported type at lua index %d. Aborting DispatchEvent.";

Int LuaBinding::DispatchEvent(lua_State* luaState)
{
    FEATSTD_DEBUG_ASSERT(0 != LuaScriptSystem::s_this);
    bool isTable = lua_istable(luaState, 1) != 0;
    if (!isTable) {
        FEATSTD_LOG_ERROR("Candera.DispatchEvent argument is missing, or not a table.");
        return 0;
    }

    lua_len(luaState, 1);
    Int tableSize = lua_tointeger(luaState, -1);
    lua_pop(luaState, 1);

    ScriptParameters scriptParameters;
    for (Int iterationIndex = 1; iterationIndex <= tableSize; ++iterationIndex) {
        lua_pushinteger(luaState, iterationIndex);
        Int argumentType = lua_gettable(luaState, -2);
        switch (argumentType) {
        case LUA_TNUMBER: {
            if (lua_isinteger(luaState, -1) != 0) {
                Int value = static_cast<Int>(lua_tointeger(luaState, -1));
                static_cast<void>(scriptParameters.Add(value));
            }
            else {
                Double value = lua_tonumber(luaState, -1);
                static_cast<void>(scriptParameters.Add(value));
            }
            break;
        }

        case LUA_TBOOLEAN: {
            bool value = lua_toboolean(luaState, -1);
            static_cast<void>(scriptParameters.Add(value));
            break;
        }

        case LUA_TSTRING: {
            const Char* value = lua_tostring(luaState, -1);
            static_cast<void>(scriptParameters.Add(value));
            break;
        }

        case LUA_TUSERDATA: {
            LuaScriptSystem::ObjectReference* objectReference = LuaScriptSystem::GetLuaObjectReference(luaState, -1);
            if (0 != objectReference) {
                CanderaObject* value = LuaScriptSystem::s_this->GetObjectPointer(objectReference->m_handle);
                static_cast<void>(scriptParameters.Add(value));
            }
            else {
                FEATSTD_LOG_ERROR(s_unsupportedEventTypeErrorMsg, iterationIndex);
                lua_pop(luaState, 1);
                return 0;
            }
            break;
        }

        default:
            FEATSTD_LOG_ERROR(s_unsupportedEventTypeErrorMsg, iterationIndex);
            lua_pop(luaState, 1);
            return 0;
        }

        lua_pop(luaState, 1);
    }

    ScriptEvent scriptEvent(scriptParameters);
    if (LuaScriptSystem::s_this->DispatchScriptEvent(scriptEvent)) {
        lua_pushboolean(luaState, 1);
        return 1;
    }

    return 0;
}

Int LuaBinding::CreateReference(lua_State* luaState)
{
    TypeId* typeIdPointer = FeatStd::Internal::PointerToPointer<TypeId*>(luaL_checkudata(luaState, 1,
        LuaScriptSystem::s_canderaInternalObjectReferenceTypeMetaTable));
    if (0 == typeIdPointer) {
        return 0;
    }

    if (LuaScriptSystem::CreateLuaObjectReference(luaState, *typeIdPointer, Internal::ObjectReferenceSystem::Handle())) {
        return 1;
    }

    return 0;
}

Int LuaBinding::IsNullReference(lua_State* luaState)
{
    CanderaObject* canderaObject = GetTypedPointer<CanderaObject>(luaState);
    lua_pushboolean(luaState, (canderaObject == 0) ? 1 : 0);
    return 1;
}

CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(StartAnimation, Animation::AnimationPlayerBase, Start)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(StopAnimation, Animation::AnimationPlayerBase, Stop)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(PauseAnimation, Animation::AnimationPlayerBase, Pause)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(ResumeAnimaton, Animation::AnimationPlayerBase, Resume)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(IsAnimationPaused, Animation::AnimationPlayerBase, IsPaused)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_BOOL(IsAnimationEnabled, Animation::AnimationPlayerBase, IsEnabled)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_FLOAT(GetAnimationSpeedFactor, Animation::AnimationPlayerBase, GetSpeedFactor)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_SET_FLOAT(SetAnimationSpeedFactor, Animation::AnimationPlayerBase, SetSpeedFactor)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_GET_INT(GetAnimationRepeatCount, Animation::AnimationPlayerBase, GetRepeatCount)
CANDERA_LUASCRIPTSYSTEM_OBJECTREFERENCE_SET_INT(SetAnimationRepeatCount, Animation::AnimationPlayerBase, SetRepeatCount)

Int LuaBinding::AnimationFinish(lua_State* luaState)
{
    Animation::AnimationPlayerBase* animationPlayer = GetTypedPointer<Animation::AnimationPlayerBase>(luaState);
    if (0 != animationPlayer) {
        Double timeToFinishInSeconds = static_cast<Double>(luaL_checknumber(luaState, 2));
        Animation::WorldTimeType timeToFinishInMilliseconds = static_cast<Animation::WorldTimeType>(timeToFinishInSeconds * 1000.0);
        bool success = animationPlayer->Finish(timeToFinishInMilliseconds);
        lua_pushboolean(luaState, success ? 1 : 0);
        return 1;
    }

    return 0;
}


} // namespace Internal

} // namespace Scripting

} // namespace Candera
