//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/Util/NumericUtil.h>
#include "SerializationPlatform.h"
#include <FeatStd/Diagnostics/Debug.h>

namespace Candera {
    UInt SerializationPlatform::ToBinary(const Char* buf, Float& dest, Int* count)
{
    //start dest from 0
    dest = 0.0F;

    if (buf == 0) {
        return 0;
    }

#if defined(FEATSTD_FIXED_POINT_ARITHMETIC)
    const Char* p = buf;
    bool negSign = false;

    //check sign
    if (*p == '-') {
        ++p;
        negSign = true;
    }
    if (*p == '+') {
        ++p;
    }

    //remove spaces
    while((*p == ' ') || (*p == '\t') || (*p == '\n') || (*p == '\r')) {
        ++p;
    }

    //TODO: interpret special values (infinity and NaN).
    //fail if there is no number
    if (((*p < '0') || (*p > '9')) && (*p != '.')) {
        return 0;
    }

    //read integer part
    while((*p >= '0') && (*p <= '9')){
        dest = (dest * Float(10)) + Float(*p - '0');
        p++;
    }

    //read fractional part
    if (*p == '.'){
        ++p;

        Float mag = 1.0F;
        while((*p >= '0') && (*p <= '9')){
            mag /= Float(10);
            dest += mag * Float(*p - '0');
            p++;
        }
    }

    //read decimal exponent
    if ((*p == 'e') || (*p == 'E')){
        ++p;

        Int exponentCount;
        Float exponent;
        if (ToBinary(p, exponent, &exponentCount) != 1) {
            return 0;
        }

        p += exponentCount;
        dest *= MathPlatform::Power(Float(10), exponent);
    }

    //add sign
    if (negSign){
        dest = -dest;
    }

    //store number of characters processed.
    if (count != 0) {
        *count = p - buf;
    }
#else
    Char separator = '\0';

    Int cnt = sscanf(buf, "%f%c", &dest, &separator);

    if (cnt < 1) {
        return 0;
    }

    cnt = 0;
    while(*buf != separator) {
        cnt++;
        buf++;
    }

    if (count != 0) {
        *count = cnt;
    }
#endif

    return 1;
}

    UInt SerializationPlatform::ToBinary(const Char* buf, Char* dest, Int* count)
{
    if ((buf == 0) || (dest == 0)) {
        return 0;
    }
    StringPlatform::Copy(dest, buf);
    if (count != 0) {
        *count = FeatStd::Internal::NumericConversion<Int>(StringPlatform::Length(buf));
    }
    return 1;
}

    UInt SerializationPlatform::FromBinary(Char* buf, UInt sizeOfBuf, Float src, Int* count)
{
    if ((buf == 0) || (sizeOfBuf == 0)) {
        return 0;
    }
    //TODO: interpret special values (infinity and NaN).

#if defined(FEATSTD_FIXED_POINT_ARITHMETIC)
    static const Int MaxExponent = 8;
    static const Int MinExponent = -4;

    Char* p = buf;

    //write sign
    if (src < 0.0F){
        *p = '-';
        p++;
        src = -src;
    }

    Float exponent = 0.0F;

    if (src != 0.0F) {
        Float logSrc = MathPlatform::Log10(src);
        exponent = MathPlatform::Floor(logSrc);

        if ((exponent >= Float(MaxExponent)) || (exponent <= Float(MinExponent))){
            src *= MathPlatform::Power(Float(1) / Float(10), exponent);
        }
        else {
            exponent = 0.0F;
        }
    }

    Float floorSrc = MathPlatform::Floor(src);
    Float fracSrc = src - floorSrc;

    //write integral part
    Int written;
    if (FromBinary(p, sizeOfBuf - (p - buf), UInt(floorSrc), &written) != 1){
        buf[0] = '0';
        return 0;
    }
    else {
        p += written;
    }

    //write fractional part
    if (fracSrc != 0.0F) {
        if ((p - buf) >= Int(sizeOfBuf - 1)) {
            buf[0] = '\0';
            return 0;
        }
        *p = '.';
        p++;

        Float mag = 1.0F;
        Float tenth = Float(1) / Float(10);
        Float epsilon = MathPlatform::Power(tenth, Float(MaxExponent));
        for (Int i = 0; i < MaxExponent; i ++) {
            if ((p - buf) >= Int(sizeOfBuf - 1)) {
                buf[0] = '\0';
                return 0;
            }

            mag *= tenth;
            if (mag < epsilon){
                break;
            }

            Float digit = MathPlatform::Floor(fracSrc / mag);
            if (digit > 9.0F) {
                digit = 9.0F;
            }
            fracSrc -= digit * mag;
            *p = Char(digit + static_cast<Float>('0'));
            p++;

            if (fracSrc < epsilon) {
                break;
            }
        }
    }

    //write exponent
    if (exponent != 0.0F) {
        if ((p - buf) >= Int(sizeOfBuf - 2)) {
            buf[0] = '\0';
            return 0;
        }
        *p = 'E';
        p++;
        if (exponent < 0.0F) {
            *p = '-';
            p++;
            exponent = -exponent;
        }
        else {
            *p = '+';
            p++;
        }

        if (FromBinary(p, sizeOfBuf - (p - buf), UInt(exponent), &written) != 1){
            buf[0] = '0';
            return 0;
        }
        else {
            p += written;
        }
    }

    *p = '\0';

    //store number of characters processed.
    if (count != 0) {
        *count = p - buf;
    }
#else
    static const Int c_tmpBufferSize = 17;
    Char tmp[c_tmpBufferSize];

    Int cnt = sprintf(tmp, "%.8G", src);
    FEATSTD_DEBUG_ASSERT(cnt > 0);
    FEATSTD_DEBUG_ASSERT(cnt < c_tmpBufferSize);

    if ((cnt < 1) || (UInt(cnt) > sizeOfBuf)) {
        return 0;
    }

    StringPlatform::CopyPartial(buf, tmp, sizeOfBuf);
    if (count != 0) {
        *count = cnt;
    }
#endif

    return 1;
}

UInt SerializationPlatform::FromBinary(Char* buf, UInt sizeOfBuf, Int src, Int* count)
{
    if ((buf == 0) || (sizeOfBuf == 0)) {
        return 0;
    }

    Char* p = buf;

    //write sign
    if (src < 0){
        *p = '-';
        p++;
        sizeOfBuf --;
        src = -src;
    }
    Int uicount;
    if (FromBinary(p, sizeOfBuf, UInt(src), &uicount) == 0){
        buf[0] = '\0';
        return 0;
    }
    else {
        p += uicount;
    }

    //store number of characters processed.
    if (count != 0) {
        *count = FeatStd::Internal::NumericConversion<Int>(FeatStd::Internal::PointerDiff(p, buf));
    }

    return 1;
}

UInt SerializationPlatform::FromBinary(Char* buf, UInt sizeOfBuf, UInt src, Int* count)
{
    if ((buf == 0) || (sizeOfBuf == 0)) {
        return 0;
    }

    Char* p = buf;

    if (src == 0){
        //set value to 0
        if (FeatStd::Internal::PointerDiff(p, buf) >= OffsetType(sizeOfBuf - 1)) {
            buf[0] = '\0';
            return 0;
        }

        *p = Char('0');
        p++;
    }
    else {
        //write inverted number
        while(src != 0) {
            if (FeatStd::Internal::PointerDiff(p, buf) >= OffsetType(sizeOfBuf - 1)) {
                buf[0] = '\0';
                return 0;
            }

            UInt quotient = src / 10;
            UInt singleDigit = src % 10;

            *p = Char(singleDigit + '0');
            p++;

            src = quotient;
        }
    }

    *p = '\0';

    //invert number
    for (Int digit = FeatStd::Internal::NumericConversion<Int>(FeatStd::Internal::PointerDiff(p, buf) / 2); digit > 0; --digit){
        Char aux = p[- digit];
        p[- digit] = buf[digit - 1];
        buf[digit - 1] = aux;
    }

    //store number of characters processed.
    if (count != 0) {
        *count = FeatStd::Internal::NumericConversion<Int>(FeatStd::Internal::PointerDiff(p, buf));
    }

    return 1;
}

UInt SerializationPlatform::FromBinary(Char* buf, UInt sizeOfBuf, const Char* src, Int* count)
{
    if ((buf == 0) || (sizeOfBuf == 0)) {
        return 0;
    }

    UInt srclen = FeatStd::Internal::NumericConversion<UInt>(StringPlatform::Length(src));
    if (srclen >= sizeOfBuf) {
        buf[0] = '\0';
        return 0;
    }

    StringPlatform::Copy(buf, src);

    if (count != 0) {
        *count = srclen;
    }
    return 1;
}

const Char* SerializationPlatform::SkipSeparators(const Char* buf, const Char* separators)
{
    if ((buf == 0) || (separators == 0)) {
        return 0;
    }

    bool isSeparator;
    do {
        isSeparator = false;

        Char c = *buf;
        buf++;
        for(const Char* p = separators; *p != '\0'; p++){
            if (*p == c) {
                isSeparator = true;
                break;
            }
        }
    } while(isSeparator);

    return buf - 1;
}

Char* SerializationPlatform::WriteSeparators(Char* buf, UInt sizeOfBuf, const Char* separators)
{
    if ((buf == 0) || (separators == 0)) {
        return 0;
    }

    bool notWrittenEos = true;
    for (UInt i = 1; i < sizeOfBuf; i++) {
        *buf = *separators;
        if (*separators++ == '\0'){
            notWrittenEos = false;
            break;
        }
        else {
            buf ++;
        }
    }

    if (notWrittenEos && (sizeOfBuf != 0)){
        *buf = '\0';
    }

    return buf;
}
} //namespace Candera

