//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_WAYLAND_CONTEXT_H)
#define GENIVI_WAYLAND_CONTEXT_H

#include <Candera/Environment.h>

// Forward declaration of wayland structures
CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, "Forward declarations from 3rd party library need to remain in global namespace.")
struct wl_display;
struct wl_registry;
struct wl_compositor;
struct serverinfo;
#if (defined(USE_WESTON_IVI_SHELL))
struct ivi_surface;
struct ivi_application;
#endif

namespace Candera
{

/**
 * @brief GeniviWaylandContext represents the connection to
 *        a wayland compositor, which is further used
 *        to create wayland EGL windows.
 *        In Wayland a display represents a wayland compositor,
 *        where one or multiple surfaces are handled, arranged and blended.
 *        An application using wayland typically renders into such a surface.
 *        A wayland display therefore is not necessarily the physical display,
 *        but depends on it's implementation.
 */
class GeniviWaylandContext
{

public:

    /**
     *  Constructor
     */
    GeniviWaylandContext();
    /**
     *  Destructor
     */
    ~GeniviWaylandContext();

    /**
     * @brief Upload fetches the wayland compositor and other important properties
     *        from the provided display handle.
     * @param handle The native display handle to fetch properties from.
     *        If called from extern, the created wl_display* or the
     *        result from CreateNativeDisplayHandle is to be passed here.
     * @return True if all properties could be fetched, false otherwise, e.g. if handle is 0.
     */
    bool Upload(EGLNativeDisplayType handle);

    /**
     * @brief Unload destroys the fetched compositor and the display handle, if they are not 0.
     */
    void Unload();

    /**
     * @brief Returns the wayland compositor fetched from the native display handle.
     *        The compositor is further used to create wayland surfaces.
     * @return The wayland compositor fetched from the native display handle.
     */
    wl_compositor * GetCompositor() const;

    /**
     * @brief Fetches the connection Id to the wayland server.
     *        This one is further used to computer the native handle, needed
     *        to create layer manager surfaces.
     * @return The connection Id to the wayland server.
     */
    UInt32 GetConnectionId() const;

    /**
     * @brief Connects to the specified wayland display, and fetches its handle.
     *        This function can also be used from external components to create
     *        wayland display handles.
     * @param displayId The id of the wayland display to connect to.
     * @return The handle to the native display.
     */
    void * CreateNativeDisplayHandle(const Char * displayId);

    /**
     * @brief Returns the native display handle of this WaylandContext.
     * @return The native display handle of this WaylandContext.
     */
    void * GetNativeDisplayHandle() const;

#if (defined(USE_WESTON_IVI_SHELL))
    void * GetIviApplication();
#endif

private:
    static void WaylandGlobalListener(void * data,
                                      struct wl_registry * waylandRegistry,
                                      uint32_t name,
                                      const Char * interface,
                                      uint32_t version);

    wl_registry_listener m_listener;

    static void WaylandGlobalRemoveListener(void * data,
                                            struct wl_registry * waylandRegistry,
                                            uint32_t name);

    static void WaylandServerInfoListener(void * data,
                                          struct serverinfo * serverInfo,
                                          uint32_t connectionId);

    wl_display * m_handle;
    wl_compositor * m_compositor;
    wl_registry * m_registry;
    uint32_t m_connectionId;
#if (defined(USE_WESTON_IVI_SHELL))
    ivi_application *m_iviApplication;
#endif
};

}

#endif
