//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GeniviTargetDisplay.h"
#include <CanderaPlatform/Device/Genivi/Target/GeniviDevicePackageInterface.h>//"GeniviDevicePackageInterface.h"
#include <CanderaPlatform/Device/Genivi/GeniviTrace.h>//"GeniviTrace.h"
#include "GeniviWaylandContext.h"
#include <WaylandInc.h>

#include <ilm/ilm_client.h>

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <FeatStd/Util/PointerUtil.h>


namespace Candera
{

//GeniviTargetDisplay implementation.
GeniviTargetDisplay::GeniviTargetDisplay(Int displayId) :
    GeniviDisplay(displayId),
    m_waylandDisplayId(0)
{
}

GeniviTargetDisplay::~GeniviTargetDisplay()
{
}

bool GeniviTargetDisplay::UploadInternal(const CommonSettings& settings)
{
    if (m_param.isUploaded) {
        return false;
    }

    if (m_waylandContext.CreateNativeDisplayHandle(m_waylandDisplayId) == 0) {
        CANDERA_DEVICE_LOG_ERROR("Wayland context of display could not be initialized.");
        return false;
    }

    if (!AttachNativeHandle(settings.width, settings.height, m_waylandContext.GetNativeDisplayHandle())){
        return false;
    }
   m_param.isUploaded = true;
   return true;
}

bool GeniviTargetDisplay::AttachNativeHandle(Int intendedWidth, Int intendedHeight, void* handle)
{
    if (m_param.eglDisplay != 0) {
        return false;
    }

#if (defined(USE_WESTON_IVI_SHELL))
    if (ilm_initWithNativedisplay(reinterpret_cast<t_ilm_nativedisplay>(handle)) != ILM_SUCCESS) {
        CANDERA_DEVICE_LOG_ERROR("IVI layer manager could not be initialized.");
        return false;
    }
    if (ilmClient_init(reinterpret_cast<t_ilm_nativedisplay>(handle)) != ILM_SUCCESS) {
        CANDERA_DEVICE_LOG_ERROR("IVI client could not be initialized.");
        return false;
    }
#elif (defined(USE_LAYERMANAGER))
    if(ilm_init() != ILM_SUCCESS) {
        CANDERA_DEVICE_LOG_ERROR("IVI layer manager could not be initialized.");
        return false;
    }
#endif

    m_param.height = intendedHeight;
    m_param.width = intendedWidth;
    m_param.handle = FeatStd::Internal::PointerToPointer<EGLNativeDisplayType>(handle);

    if (!m_waylandContext.Upload(m_param.handle)) {
        CANDERA_DEVICE_LOG_ERROR("Wayland Context could not be uploaded.");
        return false;
    }

    EGLDisplay eglDisplay = eglGetDisplay(m_param.handle);
    if (eglDisplay == EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_ERROR("Could not acquire egl display");
        m_waylandContext.Unload();
        return false;
    }
    m_param.eglDisplay = eglDisplay;

    EGLint major;
    EGLint minor;
    EGLBoolean success;

    // initialize display
    success = eglInitialize(eglDisplay, &major, &minor);
    if (success == EGL_FALSE) {
        CANDERA_DEVICE_LOG_ERROR("Could not initialize egl display");
        m_waylandContext.Unload();
        return false;
    }

    // initialize API
    success = eglBindAPI(EGL_OPENGL_ES_API);
    if (success == EGL_FALSE) {
         CANDERA_DEVICE_LOG_ERROR("Could not bind egl api");
        m_waylandContext.Unload();
        return false;
    }

    ApplyChanges();
    return true;
}

void GeniviTargetDisplay::UnloadInternal()
{
    if (m_param.eglDisplay != 0) {
        if (eglTerminate(m_param.eglDisplay)==EGL_FALSE){
#ifdef FEATSTD_GLERRORS_AND_EGLERRORS_ENABLED
            CANDERA_DEVICE_LOG_ERROR("eglTerminate for display %d failed with error %s",
                                     GetDisplayId(),
                                     EglTraceMapper::MapEglErrorToString(eglGetError()));
#endif
        }
    }

    m_waylandContext.Unload();

    static_cast<void>(ilm_destroy());

    m_param.handle = 0;
    m_param.eglDisplay=0;
    m_param.handle=0;
    m_param.height=0;
    m_param.width=0;
    m_param.isUploaded=false;
}

GeniviWaylandContext* GeniviTargetDisplay::GetWaylandContext()
{
    return &m_waylandContext;
}

void GeniviTargetDisplay::SetWaylandDisplayId(const Char* displayId)
{
    m_waylandDisplayId = displayId;
}

const Char* GeniviTargetDisplay::GetWaylandDisplayId() const
{
    return m_waylandDisplayId;
}

UInt32 GeniviTargetDisplay::GetNumberOfIlmWaylandMapItems() const
{
    return m_ilmWaylandMap.Size();
}

GeniviTargetDisplay::IlmWaylandMapItem* GeniviTargetDisplay::GetIlmWaylandMapItem(UInt32 index)
{
    IlmWaylandMapItem* result = 0;
    if (static_cast<SizeType>(index) < m_ilmWaylandMap.Size()) {
        result = &(m_ilmWaylandMap[index]);
    }

    return result;
}

}

