//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENIVI_SURFACE_H)
#define GENIVI_SURFACE_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduSupport.h>

#include <ilm/ilm_types.h>

namespace Candera
{

/** @addtogroup GeniviGDITarget
 *  @{
 */

class GeniviSurfaceProperties;

/**
 * @brief GeniviSurface is a class representing a surface to render into.
 */
class GeniviSurface : public GduSupport
{
    friend class GeniviWaylandFrameBuffer;
    public:

        typedef void(*CallbackFunction)(t_ilm_surface surfaceId);

        /**
         *  Constructor
         */
        GeniviSurface();
        /**
         *  Destructor
         */
        ~GeniviSurface();

        /**
         *  Retrieve the window handle associated with this surface.
         *  @return The window handle of this surface, 0 if not uploaded.
         */
        EGLNativeWindowType GetWindow() const { return m_window; }
        
        /**
        *  Retrieve the Wayland surface handle associated with this surface.
        *  @return The Wayland surface handle of this surface, 0 if not uploaded.
        */
        wl_surface * GetWaylandSurface() const { return m_waylandSurface; }

        /**
         *  Retrieve the height of this surface. This value is updated with Upload only.
         *  Overrides function in Window.
         *  @return The height of the associated window.
         */
        virtual Int GetHeight() const;
        /**
         *  Retrieve the width of this surface. This value is updated with Upload only.
         *  Overrides function in Window.
         *  @return The width of the associated window.
         */
        virtual Int GetWidth() const;

        /**
         *  Returns the X position of the Window on the display.
         *  X Position of ILM layer is therefore added to window position.
         *  @return     X position of the Window.
         */
        virtual Int GetX() const;

        /**
         *  Returns the Y position of the Window.
         *  Y Position of ILM layer is therefore added to window position.
         *  @return     Y position of the Window.
         */
        virtual Int GetY() const;

        /**
         *  Does nothing.
         *  @param surfaceProperties unused.
         */
        void ApplyChanges(const GeniviSurfaceProperties & surfaceProperties);

        /**
         *  Upload the current surface object.
         *  @param displayId        The display ID that is uploaded.
         *  @param surfaceProperties  The surface properties that are uploaded.
         *  @return True, if successful, false otherwise.
         */
        bool Upload(Int displayId,
                    GeniviSurfaceProperties & surfaceProperties);

         /**
         *  Attach a native window handle to the current surface object.
         *  @param handle            handle to a native window.
         *  @param displayId        The display ID that is uploaded.
         *  @param surfaceProperties  The surface properties that are uploaded.
         *  @return True, if successful, false otherwise.
         */
        bool AttachNativeWindowHandle(EGLNativeWindowType handle,
                                      Int displayId,
                                      GeniviSurfaceProperties & surfaceProperties);

        /**
         * @brief Gets the properties of this GeniviSurface.
         * @return The properties of this GeniviSurface.
         */
        const GeniviSurfaceProperties * GetProperties() const { return m_properties; }

        /**
         *  Unload the current surface object.
         *  This releases all the objects created by Upload.
         */
        void Unload();

        /**
         * Sets a callback function for when any GeniviSurface is unloaded.
         * @param callbackFunction the function pointer to be called.
         */
        static void SetUnloadCallback(CallbackFunction callbackFunction) { s_callbackFunction = callbackFunction; }

         /**
          * Fetches the callback function for when any GeniviSurface is unloaded.
          * @returns the function pointer.
          */
        static CallbackFunction GetUnloadCallback() { return s_callbackFunction; }

    private:
        EGLNativeWindowType CreateNativeWindow(Int displayId);

        EGLNativeWindowType m_window;

        GeniviSurfaceProperties * m_properties;

        Int m_display;

        wl_surface * m_waylandSurface;

        static CallbackFunction s_callbackFunction;
};

/** @}*/ //end of GeniviGDITarget

}

#endif
